/* ****************************************************************************
 *
 *	File: CosNumeric.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.Arrays;
import java.util.Locale;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosNumberParseRuntimeException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException.CosParseErrorType;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.util.BooleanHolder;
import com.adobe.internal.pdftoolkit.core.util.StringOps;

/**
 * Represents a COS numeric value object as defined in section 3.2.2 of
 * the PDF Reference Manual version 1.4.
 */
public class CosNumeric extends CosScalar
{
	private final static DoubleToStringFormatter customFormatter = new DoubleToStringFormatter(6, '.', '-', false);
	private final static int NCACHEDINTEGERS = 4096;
	private static Integer[] mCachedIntegers = new Integer[NCACHEDINTEGERS];
	private final static DecimalFormat mPDFDecimalFormat;
	
	
	static {
		DecimalFormatSymbols dfs = new DecimalFormatSymbols(Locale.US);
		dfs.setZeroDigit('0');
		dfs.setDecimalSeparator('.');
		dfs.setMinusSign('-');
		mPDFDecimalFormat = new DecimalFormat(".######E0", dfs);
		mPDFDecimalFormat.setGroupingUsed(false);
	}

	private byte[] mInputRep;	// Byte string representation underlying the object
	private Number mValue;		// Numeric value underlying the object

	/**
	 * 
	 * Constructs a COS numeric object initialized with the specified value.
	 *
	 * @param doc		Document containing the object
	 * @param value		Numeric value to initialize the object
	 * @param info		Info for the object
	 */
	CosNumeric(CosDocument doc, Number value, CosObjectInfo info)
	{
		super(doc, info);
		if (value instanceof Integer) {
			int val = value.intValue();
			if (val >= 0 && val < NCACHEDINTEGERS) {
				if (mCachedIntegers[val] == null)
					mCachedIntegers[val] = (Integer)value;
				value = mCachedIntegers[val];
			}
		}
		mInputRep = null;
		mValue = value;
	}

	/**
	 * 
	 * Constructs a COS numeric object initialized with the specified value.
	 *
	 * @param doc		Document containing the object
	 * @param inputRep	Byte string value to initialize the object
	 * @param info		Info for the object
	 */
	CosNumeric(CosDocument doc, byte[] inputRep, CosObjectInfo info)
		throws PDFCosParseException
	{
		super(doc, info);
		
		if (inputRep != null) {// cache the input array. We shall convert this to number format
								// later if required.
			int count = 0;
			while (count < inputRep.length) {
				if (inputRep[count] == 0)
					break;
				count++;
			}
			if (count < inputRep.length) {
				byte[] copy = new byte[count];
				System.arraycopy(inputRep, 0, copy, 0, count);
				inputRep = copy;
			}
		}
		mInputRep = inputRep;
	}

	/**
	 * Generates the number from byte array on demand.
	 */
	private void generateNumberValue(){
		if(this.mValue != null)
			return;
		Number value = null;		
		try
		{
			BooleanHolder wasRepaired = getDocument().getOptions().getRepairEnabled() ? new BooleanHolder(false) : null;
			value = CosToken.readNumber(mInputRep, wasRepaired);
			if(wasRepaired != null && wasRepaired.getValue())// if number was repaired while reading 
 				                      //then mark the object and document as repaired.
				this.setRepaired(true);
		} catch (PDFCosParseException e)
		{
			if(e.hasErrorType(CosParseErrorType.NumberParseError))
			{
				throw new PDFCosNumberParseRuntimeException(e);
			}				
		}
		if (value instanceof Integer) {
			int val = value.intValue();
			if (val >= 0 && val < NCACHEDINTEGERS) {
				if (mCachedIntegers[val] == null)
					mCachedIntegers[val] = (Integer)value;
				value = mCachedIntegers[val];
			}
			mInputRep = null;
		}
		mValue = value;
	}
	/**
	 * 
	 * Constructs a COS numeric object initialized from another CosNumeric.
	 *
	 * @param doc		Document containing the object
	 * @param source	CosNumeric to clone
	 */
	CosNumeric(CosDocument doc, CosNumeric source)
		throws PDFCosParseException, IOException, PDFIOException, PDFSecurityException
	{
		super(doc, null);
		if (source.isIndirect()) {
			CosObjectInfo info = doc.newObjectInfo();
			setInfo(info);
			info.setObject(this);
			info.markDirty();
		}
		mInputRep = source.mInputRep;
		mValue = source.mValue;
	}

	/**
	 * 
	 * return the type of this CosObject
	 */
	@Override
	public int getType()
	{
		return t_Numeric;
	}

	/**
	 * Obtain the object's value as an integer.
	 *
	 * @return Value of the object as an integer.
	 */
	@Override
	public int intValue()
	{
		generateNumberValue();
		return mValue.intValue();
	}

	/**
	 * This API returns value of this number as a legacy 16.16 fixed point number.
	 * Logic of this method is copied from "Convert.cpp" file in Acrobat..
	 *
	 * @return int
	 */
	public int fixedValue()
		throws PDFCosParseException
	{
		generateNumberValue();
		if (mValue instanceof Integer || mValue instanceof Long) {
			long value = mValue.longValue();
			if (value > Short.MAX_VALUE || value < Short.MIN_VALUE)
				throw new PDFCosParseException("Fixed value out of range");
			return (int)(value << 16);
		} else {
			double value = mValue.doubleValue();
			if (value == 32768.0)
				return Integer.MAX_VALUE;
			if (value > Short.MAX_VALUE || value < Short.MIN_VALUE)
				throw new PDFCosParseException("Fixed value out of range");
			value *= 65536.0;
			if (value >= 0)
				value += 0.5;
			else
				value -= 0.5;
			return (int)value;
		}
	}

	/**
	 * Obtain the object's value as a long.
	 *
	 * @return Value of the object as a long.
	 */
	@Override
	public long longValue()
	{
		generateNumberValue();
		return mValue.longValue();
	}

	/**
	 * Obtain the object's value as a floating point number.
	 *
	 * @return Value of the object in floating point.
	 */
	@Override
	public double doubleValue()
	{
		generateNumberValue();
		return mValue.doubleValue();
	}

	/**
	 * Obtain the object's value as a Java Number object
	 *
	 * @return Number object representing the object's value.
	 */
	@Override
	public Number numberValue()
	{
		generateNumberValue();
		return mValue;
	}

	@Override
	public Object getValue()
	{
		generateNumberValue();
		return numberValue();
	}

	
	@Override
	void writeOut(OutputByteStream outStream, boolean inString, boolean inDebugMode)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
			if (mInputRep != null) {
				outStream.write(mInputRep);
			} else {
				if (mValue instanceof Double || mValue instanceof Float) {
					String raw = customFormatter.appendFormatted(mValue.doubleValue());
					if(raw != null)
					{
						outStream.write(StringOps.toByteArray(raw));
					}
					else
					{
						raw = CosNumeric.mPDFDecimalFormat.format(mValue);
						char curChar;
						int count = 0;
						int length = raw.length();
						boolean sign = false;
						boolean expSign = false;
						boolean expSeen = false;
						int digitPos = 0;
						int exponent = 0;
						int numDigits = 0;
						while (count < length) {
							curChar = raw.charAt(count);
							if (curChar == '-') {
								if (expSeen)
									expSign = true;
								else
									sign = true;
							} else if (curChar == 'E') {
								expSeen = true;
							} else if (curChar >= '0' && curChar <= '9') {
								if (expSeen) {
									exponent *= 10;
									exponent += (curChar - '0');
								} else {
									if (digitPos == 0)
										digitPos = count;
									numDigits++;
								}
							}
							count++;
						}
						if (numDigits == 0 || !expSeen) {
							outStream.write('0');
						} else {
							if (sign)
								outStream.write('-');
							if (expSign) {
								outStream.write('0');
								outStream.write('.');
								for (count = 0; count < exponent; count++)
									outStream.write('0');
								for (count = 0; count < numDigits; count++)
									outStream.write(raw.charAt(digitPos + count));
							} else {
								if (exponent >= numDigits) {
									for (count = 0; count < numDigits; count++)
										outStream.write(raw.charAt(digitPos + count));
									for (count = numDigits; count < exponent; count++)
										outStream.write('0');
								} else {
									if (exponent == 0)
										outStream.write('0');
									for (count = 0; count < exponent; count++)
										outStream.write(raw.charAt(digitPos + count));
									if (numDigits != exponent + 1 || raw.charAt(digitPos + exponent) != '0') {
										outStream.write('.');
										for (count = exponent; count < numDigits; count++)
											outStream.write(raw.charAt(digitPos + count));
									}
								}
							}
						}
					}
						
					
				} else {
					outStream.write(StringOps.toByteArray(String.valueOf(mValue.longValue())));
				}
			}
	}
	
	/**
	 *  This method returns true if both CosNumeric have same integer or real value inside.
	 *  Returns false if passed CosObject is not an instance of CosNumeric.
	 *  @param value
	 *  @return boolean
	 */	
	@Override
	public boolean equals(CosObject value){
		if(!(value instanceof CosNumeric) || value.getDocument() != this.getDocument())
			return false;
		if(value == this)
			return true;
		CosNumeric numericValue = (CosNumeric) value;
		byte[] numericValueBytes1 = numericValue.mInputRep;
		byte[] numericValueBytes2 = this.mInputRep;
		if(numericValueBytes1 != null && numericValueBytes2 != null)
			return Arrays.equals(numericValueBytes1, numericValueBytes2);
		else {
			this.generateNumberValue();
			((CosNumeric)value).generateNumberValue();
			return numericValue.mValue.equals(this.mValue);
		}
	}
}
