/* ****************************************************************************
 *
 *	File: EncryptingOutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

// @author itenenbo

package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.io.OutputStream;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityConfigurationException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandler;
import com.adobe.internal.pdftoolkit.core.securityframework.EncryptionHandlerState;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * This class encrypts the content before writing it to the specified CosStream's OutputByteStream.
 */
public class EncryptingOutputStream extends OutputStream
{
	private EncryptionHandlerState mEncryptionHandler;
	private boolean mNeedsEncryption;
	private byte[] mStreamKey;
	private OutputByteStream mOutStream;
	boolean mInitialized;
	boolean mDone;

	/**
	 * This constructor uses the supplied {@link EncryptionHandlerState} instance for all its
	 * encryption-related operations. The caller may use
	 * {@link CosEncryption#getStreamEncryptionStateHandler(CosStream)}
	 * to get reusable EncryptionHandlerState instance. This instance may not be
	 * used with any other EncryptingOutputStream object or DecryptingInputStream object
	 * until all writing to the specified CosObject's OutputByteStream completes.
	 * @param cosStream CosStream object to which encrypted content is written
	 * @param destination OutputByteStream associated with the cosStream object.
	 * @param encryptionHandler EncryptionHandlerState object used for encryption in this
	 * EncryptingOutputStream object
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 * @throws IOException 
	 */
	EncryptingOutputStream(CosStream cosStream, OutputByteStream destination, EncryptionHandlerState encryptionHandler)
		throws PDFIOException, PDFSecurityException, PDFCosParseException, IOException
	{
		mNeedsEncryption = cosStream.needsEncryption();
		mEncryptionHandler = encryptionHandler;
		initialize(cosStream, destination);
	}

	/**
	 * This constructor creates a EncryptingOutputStream object with its own copy of the
	 * {@link EncryptionHandlerState}. It is save to use this instance for asynchronous
	 * writes to the destination OutputByteStream.
	 * @param cosStream The CosStream to which the content is written.
	 * @param destination The OutputByteStream associated with the cosStream.
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws PDFCosParseException
	 * @throws IOException 
	 */
	EncryptingOutputStream(CosStream cosStream, OutputByteStream destination)
		throws PDFIOException, PDFSecurityException, PDFCosParseException, IOException
	{
		mNeedsEncryption = cosStream.needsEncryption();
		if (mNeedsEncryption) {
			ASName cryptFilter = cosStream.getCryptFilter();
			EncryptionHandler streamEncryption = cosStream.getDocument().getEncryption().getStreamEncryptionHandler(cryptFilter == null ? null : cryptFilter.asString(true));
			if (streamEncryption == null)
				throw new PDFSecurityConfigurationException("Cannot find Security Handler for a stream");
			mEncryptionHandler = streamEncryption.createEncryptionHandlerState();
		}
		initialize(cosStream, destination);
	}

	private void initialize(CosStream cosStream, OutputByteStream destination)
		throws IOException, PDFSecurityException, PDFCosParseException, PDFIOException
	{
		if (mNeedsEncryption) {
			if (mEncryptionHandler == null)
				throw new PDFSecurityConfigurationException("Cannot find Security Handler for a stream");
			mStreamKey = cosStream.getDocument().getEncryption().getStreamEncryptionKey(cosStream, true);
		}
		mOutStream = destination;
		mInitialized = false;
		mDone = false;
	}

	@Override
	public void write(int buffer)
		throws IOException
	{
		byte[] in = {(byte)buffer};
		write(in);
	}

	@Override
	public void write(byte[] buffer)
		throws IOException
	{
		write(buffer, 0, buffer.length);
	}

	@Override
	public void write(byte[] buffer, int off, int len)
		throws IOException
	{
		if (mDone)
			throw new IOException("Cannot write to a closed stream");
		byte[] decrypted = buffer;
		int offset = off;
		int bufLen = len;
		try {
			if (mNeedsEncryption) {
				decrypted = mInitialized ?
					mEncryptionHandler.update(buffer, off, len) :
					mEncryptionHandler.init(buffer, off, len, mStreamKey, EncryptionHandlerState.ENCRYPT);
				if (decrypted == null)
					decrypted = new byte[0];
				offset = 0;
				bufLen = decrypted.length;
				mInitialized = true;
			}
			mOutStream.write(decrypted, offset, bufLen);
		} catch (PDFSecurityException e) {
			IOException excp = new IOException();
			excp.initCause(e);
			throw excp;
		}
		
	}

	@Override
	public void flush()
		throws IOException
	{
		if (mDone)
			throw new IOException("Cannot flush closed stream");
		mOutStream.flush();
		
	}

	/* (non-Javadoc)
	 * @see java.io.OutputStream#close()
	 * This method does not close the underlining OutputByteStream to which it writes
	 * encrypted data.
	 */
	@Override
	public void close()
		throws IOException
	{
		if (mDone)
			throw new IOException("Cannot close already closed stream");
		if (mNeedsEncryption) {
			byte[] encrypted;
			try {
				encrypted = mEncryptionHandler.finish();
			} catch (PDFSecurityException e) {
				IOException excp = new IOException();
				excp.initCause(e);
				throw excp;
			}
			if (encrypted.length > 0) {
				mNeedsEncryption = false;
				write(encrypted);
			}
		}
		mDone = false;
	}
}