/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import java.util.ArrayList;
import java.util.List;


/**
 * NodeListFilter filters a node and its children to produce a list of
 * nodes that satisfy some filter test.
 * <p>
 * A derived class will typically override the {@link #accept(Node)} method,
 * and can further refine the behaviour by overriding {@link #init()},
 * {@link #checkChildren(Node)} or {@link #checkFinished()}.
 *
 * @exclude from published api.
 */
public class NodeListFilter {
	
	public NodeListFilter() {
	}

	/**
	 * Determines whether node is to be included in the filtered list of nodes. 
	 * Derived classes will want to override this behaviour.
	 * 
	 * @param node
	 *            the node to test for inclusion
	 * @return <code>true</code> if the node is to be included. The default implementation
	 *         always returns <code>true</code>.
	 */
	public boolean accept(Node node) {
		return true;
	}

	/**
	 * Determines whether the filter should search the children of this node.
	 * 
	 * @param node
	 *            the node to check
	 * @return <code>true</code> if the filter should search children of <code>node</code>.
	 */
	public boolean checkChildren(Node node) {
		return true;
	}

	/**
	 * Determines whether a search should continue or not. 
	 * We may want to abort the search in cases such as where we're
	 * searching for a specific node or the first occurrence of a node and wish
	 * to stop searching when we've found it.
	 * 
	 * @return <code>true</code> if we're done, <code>false</code> if we're to keep searching.
	 */
	public boolean checkFinished() {
		return false;
	}

	/**
	 * This method is called when a search is started.
	 */
	public void init() {
	}
	
	/**
	 * Searches the startNode all descendants of startNode.
	 * <br>
	 * Only nodes that are accepted by the filter will be added to the list.
	 * 
	 * @param startNode 
	 *            the node to start searching from.
	 *            
	 * @return a list of nodes
	 */
	public List<Node> filterNodes(Node startNode) {
		return filterNodes(startNode, 0);
	}
	
	/**
	 * Searches the startNode all descendants of startNode down to the level specified.
	 * <br>
	 * Only nodes that are accepted by the filter will be added to <code>nodeList</code>.
	 * 
	 * @param startNode 
	 *            the node to start searching from.
	 * @param level 
	 *            <br>
	 *            <br>
	 *            0 = Exhaustively search all descendants of startNode<br>
	 *            1 = Search immediate children.<br>
	 *            2 = Search immediate children and grandchildren<br>
	 *            n = Search descendants down to the nth level.<br>
	 * @return a list of nodes
	 */
	public List<Node> filterNodes(Node startNode, int level, List<Node> nodeList) {
		
		init();

		filterNodes(startNode, 1, level, nodeList);
		
		return nodeList;	
	}

	/**
	 * Searches the startNode all descendants of startNode down to the level specified.
	 * <br>
	 * Only nodes that are accepted by the filter will be added to the result list.
	 * 
	 * @param startNode 
	 *            the node to start searching from.
	 * @param level 
	 *            <br>
	 *            <br>
	 *            0 = Exhaustively search all descendants of startNode<br>
	 *            1 = Search immediate children.<br>
	 *            2 = Search immediate children and grandchildren<br>
	 *            n = Search descendants down to the nth level.<br>
	 *            
	 * @return a list of nodes
	 */
	public List<Node> filterNodes(Node startNode, int level) {
		return filterNodes(startNode, level, new ArrayList<Node>());
	}
	
	private void filterNodes(Node startNode, int curLevel, int targetLevel, List<Node> nodeList) {
		
		if (targetLevel == 0 || curLevel <= targetLevel) {
			int nextLevel = 0;

			if (targetLevel != 0)
				nextLevel = ++curLevel;

			for (Node child = startNode.getFirstXFAChild(); child != null; child = child.getNextXFASibling()) {
				
				if (checkFinished())
					return;

				if (accept(child))
					nodeList.add(child);

				if (checkChildren(child))
					filterNodes(child, nextLevel, targetLevel, nodeList);
			}
		}
	}
}
