/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

import com.adobe.xfa.ut.Assertions;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;


/**
 * This class holds the definition for a script function.
 * 
 * @exclude from published api.
 */
public final class ScriptFuncObj {
	private final Class<?> mClass;
	private final String msName; // interned
	private final int meRetType;
	private final int[] meParamTypes;
	private final int mnMinParam;
	private final int mnXFAVersion; // intro version
	private final int mnAvailability; // 0 means deprecated
	private final int mnVersionDep; // deprecated version
	
	private final Method mMethod;
	private final boolean mbHasMethodDep;
	
	private final Method mPermsMethod;
	
	
	public ScriptFuncObj(Class<?> clazz, String name, String sFunc, int eRetType,
			int[] paramTypes, int minParams, int nVersion, int nAvailability, int nVersionDep) {
		this(clazz, name, sFunc, eRetType, paramTypes, minParams, nVersion, nAvailability, null, nVersionDep);
	}

	@FindBugsSuppress(code="ES,EI2")
	public ScriptFuncObj(Class<?> clazz, String sName, String sFunc, int eRetType,
			int[] paramTypes, int minParams, int nVersion, int nAvailability,
			String sPermsMethod, int nVersionDep) {

		if (Assertions.isEnabled)
			assert sName == sName.intern();
		
		assert clazz != null;
		assert sName != null;
		assert sFunc != null;
		assert paramTypes != null;

		mClass = clazz;
		msName = sName;
		meRetType = eRetType;
		meParamTypes = paramTypes;
		mnMinParam = minParams;
		mnXFAVersion = nVersion; // intro version
		mnAvailability = nAvailability; // 0 means deprecated
		mnVersionDep = nVersionDep; // intro version
		
		boolean bHasMethodDep = false;		
		Method m = null; 
		
		try {
			m = mClass.getMethod(sFunc, Obj.class, Arg.class, Arg[].class);
		}
		catch (NoSuchMethodException ex) {
		}
		
		// If there was no "normal" method, see if there is one with a DependencyTracker
		if (m == null) {
			try {
				m = mClass.getMethod(sFunc, Obj.class, Arg.class, Arg[].class, DependencyTracker.class);
				
				bHasMethodDep = true;
			}
			catch (NoSuchMethodException ex) {
				assert false;
			}
		}
		
		mMethod = m;
		mbHasMethodDep = bHasMethodDep;
		
		m = null;
		if (sPermsMethod != null) {
			try {
				m = mClass.getMethod(sPermsMethod, Obj.class, Arg[].class);
				assert m.getReturnType() == boolean.class;
			}
			catch (NoSuchMethodException ex) {
				assert false;
			}
		}
		mPermsMethod = m;
	}

	public boolean invoke(Obj scriptThis, Arg retVal, Arg[] parms, DependencyTracker dependencyTracker) {	
		
		boolean bError = false;
		try {
			// mMethod can't be null if script tables were constructed correctly			
			if (hasDependencyTracker())
				mMethod.invoke(scriptThis, scriptThis, retVal, parms, dependencyTracker);
			else
				mMethod.invoke(scriptThis, scriptThis, retVal, parms);
		} catch (IllegalAccessException e3) {
			bError = true;	// Should never happen if script tables are constructed correctly
		} catch (InvocationTargetException e2) {
			Throwable t = e2.getCause();
			if (t instanceof ExFull) {
				((ExFull) t).resolve();
			    throw (ExFull) t;
			}
			bError = true;
		}
		
		return bError;
	}
	
	public boolean invokePermsFunc(Obj scriptThis, Arg[] parms) {
		
		if (mPermsMethod == null)
			return true;
		
		boolean bAllow = false;
		try {
			Boolean bResult = (Boolean)mPermsMethod.invoke(null, scriptThis, parms);
			bAllow = bResult.booleanValue();
		}
		catch (IllegalAccessException ex) {
			assert false;
		}
		catch (InvocationTargetException ex) {
			assert false;
		}
		
		return bAllow;
	}

	public int getMinParam() {
		return mnMinParam;
	}

	@FindBugsSuppress(code="EI")
	public int[] getParamTypes() {
		return meParamTypes;
	}

	public String getName() {
		return msName;
	}

	public int getXFAVersion() {
		return mnXFAVersion;
	}

	public int getAvailability() {
		return mnAvailability;
	}

	public int getVersionDeprecated() {
		return mnVersionDep;
	}

	public int getRetType() {
		return meRetType;
	}
	
	public boolean hasDependencyTracker() {
		return mbHasMethodDep;  
	}
}
