/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.template.formatting;

import com.adobe.xfa.Attribute;
import com.adobe.xfa.Element;
import com.adobe.xfa.Node;
import com.adobe.xfa.ProtoableNode;
import com.adobe.xfa.STRS;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.XFA;

/**
 * Functionality associated with the XFA <color> element.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class Color extends ProtoableNode {
	/**
	 * Return the string value for a given set of RGB values,
	 * 
	 * @param nRed -
	 *            Red RGB value
	 * @param nGreen -
	 *            Green RGB value
	 * @param nBlue -
	 *            Blue RGB value
	 * @param cDelimiter -
	 *            an optional parameter that will return the individual values
	 *            of the color delimeted by the delimiter string, or commas if
	 *            omitted.
	 */
	public static String toString(int nRed, int nGreen, int nBlue,
									char cDelimiter /* =',' */) {
		// ensure red is valid
		if (nRed > 255)
			nRed = 255;
		if (nRed < 0)
			nRed = 0;

		// ensure green is valid
		if (nGreen > 255)
			nGreen = 255;
		if (nGreen < 0)
			nGreen = 0;

		// ensure blue is valid
		if (nBlue > 255)
			nBlue = 255;
		if (nBlue < 0)
			nBlue = 0;
		
		return Integer.toString(nRed) + cDelimiter + Integer.toString(nGreen)
				+ cDelimiter + Integer.toString(nBlue);
	}

	/**
	 * Encoded color value
	 */
	private int mnRGB;

	/**
	 * cached white value.
	 */
	private final StringAttr mWhite;

	public Color(Element parent, Node prevSibling) {
		super(parent, prevSibling, null, XFA.COLOR, XFA.COLOR, null,
				XFA.COLORTAG, XFA.COLOR);
		mWhite = new StringAttr(null, null, XFA.VALUE, STRS.WHITECOLOR, false);
		mnRGB = -1;
	}

	public Attribute defaultAttribute(int eTag) {

		if (eTag == XFA.VALUETAG) {
			if (getXFAParent() instanceof Fill
					&& (getXFAParent().getXFAParent() == null || 
						!(getXFAParent().getXFAParent().getClassTag() == XFA.FONTTAG))) {
				

				return mWhite;
			}
		}

		// Not special case, so let the base class handle it
		return super.defaultAttribute(eTag);
	}
	// XTG CL#760811
	@Override
	public boolean isContextSensitiveAttribute(int eTag) {
		return (eTag == XFA.VALUETAG) ||
			super.isContextSensitiveAttribute(eTag);
	}

	/**
	 * Compare if two Color objects define the same color
	 * 
	 * @param object
	 *            a Color object to compare with this Color
	 * @return true if they define the same color, false otherwise.
	 */
	public boolean equals(Object object) {
		
		if (this == object)
			return true;
		
		// This overrides Object.equals(boolean) directly, so...
		if (object == null)
			return false;
		
		if (object.getClass() != getClass())
			return false;
		
		Color oColor = (Color) object;
		
		return getRed()   == oColor.getRed()   &&
			   getGreen() == oColor.getGreen() &&
			   getBlue()  == oColor.getBlue();
	}
	
	/**
	 * Returns a hash code value for the object. This method is unsupported.
	 * @exclude from published api.
	 */
	public int hashCode() {
		return mnRGB;
	}
	
	/**
	 * Get the blue value of the RGB triplet
	 * 
	 * @return the blue value for the RGB triplet.
	 */
	public int getBlue() {
		if (mnRGB < 0) {
			// The internal member var is uninitialized.
			// Update it from xml dom side but be careful to NOT
			// call Node.setAttribute(). In other words we
			// only want to update mnRGB from xml dom side!
			// Use getAttrValue() instead of toString() for performance reasons.
			String sColor = getAttribute(XFA.VALUETAG).getAttrValue();
			setFromString(sColor);
		}

		return (mnRGB & 0X0000FF);
	}

	/**
	 * Get the converted grayscale value of the RGB triplet as a percentage
	 * 
	 * @return the greyscale percentage for the RGB triplet.
	 */
	public double getGrayRate() {
		int nGrayScaleValue = getGrayScale();

		// Return the grayscale value as a shading percentage
		// 0 = white 100 = black
		return (255 - nGrayScaleValue) / 2.55d;
	}

	/**
	 * Get the converted grayscale value of the RGB triplet
	 * 
	 * @return the greyscale value for the RGB triplet.
	 */
	public int getGrayScale() {
		if (mnRGB < 0) {
			// The internal member var is uninitialized.
			// Update it from xml dom side but be careful to NOT
			// call Node.setAttribute(). In other words we
			// only want to update mnRGB from xml dom side!
			// Use getAttrValue() instead of toString() for performance reasons.
			String sColor = getAttribute(XFA.VALUETAG).getAttrValue();
			setFromString(sColor);
		}

		int nRed = getRed();
		int nGreen = getGreen();
		int nBlue = getBlue();

		return (((nRed * 30) + (nGreen * 59) + (nBlue * 11)) / 100);
	}

	/**
	 * Get the green value of the RGB triplet
	 * 
	 * @return the green value for the RGB triplet.
	 */
	public int getGreen() {
		if (mnRGB < 0) {
			// The internal member var is uninitialized.
			// Update it from xml dom side but be careful to NOT
			// call Node.setAttribute(). In other words we
			// only want to update mnRGB from xml dom side!
			// Use getAttrValue() instead of toString() for performance reasons.
			String sColor = getAttribute(XFA.VALUETAG).getAttrValue();
			setFromString(sColor);
		}
		return (mnRGB & 0X00FF00) >> 8;
	}

	/**
	 * Get the red value of the RGB triplet
	 * 
	 * @return the red value for the RGB triplet.
	 */
	public int getRed() {
		if (mnRGB < 0) {
			// The internal member var is uninitialized.
			// Update it from xml dom side but be careful to NOT
			// call Node.setAttribute(). In other words we
			// only want to update mnRGB from xml dom side!
			// Use getAttrValue() instead of toString() for performance reasons.
			String sColor = getAttribute(XFA.VALUETAG).getAttrValue();
			setFromString(sColor);
		}
		return (mnRGB & 0XFF0000) >> 16;
	}

	public Attribute newAttribute(int eTag, String value) {
		if (eTag == XFA.VALUETAG && value.length() == 0)
			return defaultAttribute(eTag);

		return super.newAttribute(eTag, value);
	}

	public void setAttribute(Attribute sValue, int eTag) {
		if (eTag == XFA.VALUETAG)
			mnRGB = -1; // invalidate cache
		super.setAttribute(sValue, eTag);
	}

	/**
	 * Set the blue value for the RGB triplet.
	 * 
	 * @param nBlue
	 *            the red value for the RGB triplet.
	 */
	public void setBlue(int nBlue) {
		// update internal member (for performance)
		setBlueInternal(nBlue);

		// update xml dom values so it will persist
		String sColor = toString(',');
		super.setAttribute(new StringAttr(XFA.VALUE, sColor), XFA.VALUETAG);
	}

	public void setBlueInternal(int nBlue) {
		if (mnRGB < 0)
			mnRGB = 0;
		else
			mnRGB &= 0XFFFF00;
		mnRGB |= nBlue;
	}

	/**
	 * Set the Color.
	 * 
	 * @param nRed
	 *            the red value for the RGB triplet.
	 * @param nGreen
	 *            the green value for the RGB triplet.
	 * @param nBlue
	 *            the blue value for the RGB triplet.
	 */
	public void setColor(int nRed, int nGreen, int nBlue) {
		if (nRed == 0 && nGreen == 0 && nBlue == 0) {
			mnRGB = 0;
		} else {
			// Set the internal member variable (for performance)
			setRedInternal(nRed);
			setGreenInternal(nGreen);
			setBlueInternal(nBlue);
		}
		// Apply to xml dom so that it will persist.
		String sColor = toString(',');
		super.setAttribute(new StringAttr(XFA.VALUE, sColor), XFA.VALUETAG);
	}

	/**
	 * Set the Color.
	 * 
	 * @param sColor
	 *            the string value representing the RGB triplet. This String must be interned.
	 */
	public void setColor(String sColor) {
		// Set the internal member variable (for performance)
		setFromString(sColor);
		// Apply to xml dom so that it will persist.
		String sCol = toString(',');
		super.setAttribute(new StringAttr(XFA.VALUE, sCol), XFA.VALUETAG);
	}

	private void setFromString(String sColor) {
		// Optimize for the majority of cases (white and black).
		if (sColor.equals("255,255,255")) {
			mnRGB = 0xFFFFFF;
		} else if (!sColor.equals("0,0,0")) {

			int offset = collectColor(3, sColor, sColor.length()-1);
			offset = collectColor(2, sColor, offset);
			collectColor(1, sColor, offset);
		}
	}
	private int collectColor(int rgb, String sColor, int offset) {
		boolean c1=false, c2=false, c3=false;
		int value256 = 0;
		
		while (offset >=0) {
			char c = sColor.charAt(offset);
			if (c == ',')
				break;
			if (c >= '0' && c <= '9') {
				int val = (int)c - (int)'0';
				if (!c3) {
					c3 = true;
					value256 = val;
				}
				else if (!c2) { 
					c2 = true;
					value256 += val * 10;
				}
				else if (!c1) {
					c1 = true;
					value256 += val * 100;
				}
			}
			offset--;
		}
		if (rgb == 1)
			setRedInternal(value256);
		else if (rgb == 2) 
			setGreenInternal(value256);
		else if (rgb == 3) 
			setBlueInternal(value256);
		
		return --offset;
	}

	/**
	 * Set the green value for the RGB triplet.
	 * 
	 * @param nGreen
	 *            the red value for the RGB triplet.
	 */
	public void setGreen(int nGreen) {
		// update internal member (for performance)
		setGreenInternal(nGreen);

		// update xml dom values so it will persist
		String sColor = toString(',');
		super.setAttribute(new StringAttr(XFA.VALUE, sColor), XFA.VALUETAG);
	}

	public void setGreenInternal(int nGreen) {
		if (mnRGB < 0)
			mnRGB = 0;
		else
			mnRGB &= 0XFF00FF;
		mnRGB |= nGreen << 8;
	}

	/**
	 * Set the red value for the RGB triplet.
	 * 
	 * @param nRed
	 *            the red value for the RGB triplet.
	 */
	public void setRed(int nRed) {
		// update internal member (for performance)
		setRedInternal(nRed);

		// update xml dom values so it will persist
		String sColor = toString(',');
		super.setAttribute(new StringAttr(XFA.VALUE, sColor), XFA.VALUETAG);
	}

	public void setRedInternal(int nRed) {
		if (mnRGB < 0)
			mnRGB = 0;
		else
			mnRGB &= 0X00FFFF;
		mnRGB |= nRed << 16;
	}

	/**
	 * Return the string value for this color.
	 * 
	 * @param cDelimiter -
	 *            an optional parameter that will return the individual values
	 *            of the color delimeted by the delimiter string, or commas if
	 *            omitted.
	 */
	public String toString(char cDelimiter/* = ',' */) {
		return toString(getRed(), getGreen(), getBlue(), cDelimiter);
	}

}
