package com.adobe.xfa.text;

import com.adobe.xfa.ut.Rect;
import com.adobe.xfa.ut.UnitSpan;

/**
 * This class represents a text stream (see class jfTextStream) that can
 * be displayed.  In order to display a stream, AXTE must perform a text
 * layout to do word wrapping, justification, and so on.  This abstract
 * base class class extends jfTextStream, serving as the base class for
 * the concrete displayable stream classes: jfTextBlock, jfTextLabel and
 * jfTextRegion.
 * <p>
 * The layout actually happens through a companion class, jfTextDisplay.
 * When the application needs to perform layout-related operations, it
 * calls the Createdisplay() method.  The first call for a jfTextDispStr
 * instance will create the companion display object; subsequent calls
 * simply return a pointer to the already created display.
 * </p>
 * <p>
 * Note that there are a number of operations on other text classes that
 * may not work until the display exists.  For example, caret/position
 * translation and vertical movement.
 * </p>
 * @exclude from published api.
*/

public class TextDispStr extends TextSparseStream {
	private TraditionalFrame mpoFrame;

/**
 * Default constructor.
 * <p>
 * The text stream contains no content and has no pool/mapping
 * assocotiation.  The display is not automatically created.
 */
	public TextDispStr () {
	}

/**
 * Copy constructor.
 * <p>
 * Copy all stream content from the source stream.	The display is not
 * automatically created.
 * @param oSource - Source stream to copy content from.
 */
	public TextDispStr (TextDispStr oSource) {
		super (oSource);
	}

/**
 * Copy constructor with graphics source information.
 * <p>
 * Copy all stream content from the source stream, but using the
 * optional graphic attribute pool.  The display is not automatically
 * created.
 * @param oSource - Source text stream to copy content from.
 * @param poPool - Graphic attribute pool to use.
 */
//	public TextDispStr (TextStream oSource, GFXAttrPool poPool) {
//		TextSparseStream = oSource, poPool;
//	}

/**
 * Constructor with source text string.
 * <p>
 * Create a text stream whose initial content is copied from the given
 * string.	The text stream initially has no attribute pool association.
 * The display is not automatically created.
 * @param sSource - String whose contents are to be copied to the text
 * stream.
 */
	public TextDispStr (String sSource) {
		super (sSource);
	}
	
	public boolean equals(Object object) { // NOPMD - UselessOverridingMethod - here for documentation/FindBugs suppression
		// The fields added in this derived class do not participate in equality
		return super.equals(object);
	}
	
	public int hashCode() { // NOPMD - UselessOverridingMethod - here for documentation/FindBugs suppression
		// The fields added in this derived class do not participate in equality
		return super.hashCode();
	}

/**
 * Query the graphical extent of the text.
 * <p>
 * This method returns the graphical extent of the text as a rectangle.
 * This extent is defined as the minimum of the text's graphical size
 * and the maximum graphical size (see below) of the displayable stream.
 * For example, consider a text block with a small amount of text.	The
 * returned value will be the size of the text block.  If that block's
 * text grows to the point where scrolling is requied, the returned
 * value will be the total graphical extent of all of the block's text.
 * @return A rectangle indicating the extent of the text.  Note that the
 * extent includes justification but not scroll offsets.  For example, a
 * small amount of right-aligned text means that the left side of the
 * returned rectangle will be non-zero.  If the text is so large that
 * there is a vertical scroll bar, the top will be zero even if the
 * vertical alignment is bottom.
 */
	public Rect extent () {
		return mpoFrame.getExtent();
	}

/**
 * Query the extent of the underlying text.
 * <p>
 * This likely exists for historical reasons only.	It simply returns
 * the same extent as the Extent() method.	In some (but not all) cases,
 * it will create the display for the caller.
 * return A rectangle indicating the extent of the text.
 */
	public Rect runtimeExtent () {
// If it can grow in either width or height (i.e., not a block), we may
// have to compute the extent.
// if it contains any run-time data (fields or embedded objects), compute
// the extent by creating a display.
		if ((maxWidth().value() <= 0) || (maxHeight().value() <= 0)) {
			TextRange oRange = new TextRange (this);
			if ((oRange.countField() > 0) && (oRange.countEmbed() > 0)) {
//				createDisplay();	// TODO:
			}
		}
		return extent();
	}

/**
 * Query whether the displayable stream allows horizontal overflow.
 * <p>
 * A displayable stream can be told to restrict its content horizontally
 * to the graphical space alotted, or to allow the content to overflow
 * (effectively turning off word-wrapping).  This is not to be confused
 * with the MaxSize() method of the base text stream class which
 * restricts by number of characters.  This is a legacy feature that
 * applied to single-line text objects.  There were cases where the
 * graphical space wasn't quite large enough for the maximum number of
 * characters.	Allowing overflow prevented word-wrapping or scrolling.
 * @return TRUE if the stream allows horizontal overflow; FALSE if not.
 * The default is FALSE (though some displayable stream types inherently
 * allow it).
 */
	public boolean allowOverflow () {
		return getTraditionalFrame().suppressWordWrap();
	}

/**
 * Change the value of the AllowOverflow setting.
 * A displayable stream can be told to restrict its content horizontally
 * to the graphical space alotted, or to allow the content to overflow
 * (effectively turning off word-wrapping).  This is not to be confused
 * with the MaxSize() method of the base text stream class which
 * restricts by number of characters.  This is a legacy feature that
 * applied to single-line text objects.  There were cases where the
 * graphical space wasn't quite large enough for the maximum number of
 * characters.	Allowing overflow prevented word-wrapping or scrolling.
 * Changing the setting will cause the stream to be redisplayed if there
 * is a display associated.
 * @param bNewAllow - TRUE if the stream is to allow horizontal
 * overflow; FALSE if not (though some displayable stream types
 * inherently allow overflow).
 */
	public void allowOverflow (boolean bNewAllow) {
		if (bNewAllow == allowOverflow()) {
			return;
		}

		getTraditionalFrame().setSuppressWordWrap (bNewAllow);
		if (display() != null) {
			display().update();
		}
	}

/**
 * Query whether vertical justification occurs at a point.
 * <p>
 * The text label class performs vertical justification at a point.  The
 * text block and region have a nominal height.  Vertical justification
 * occurs within this height.  This method asks the stream what sort of
 * vertical justification it supports.
 * @return TRUE if the stream justifies vertically at a point; FALSE if
 * it justifies vertically within a nominal height.
 */
	public boolean justifyVPoint () {
		return getTraditionalFrame().alignVPoint();
	}

/**
 * Query whether horizontal justification occurs at a point.
 * <p>
 * The text label class performs horizontal justification at a point.
 * The text block and region have a nominal width.	Horizontal
 * justification occurs within this width.	This method asks the stream
 * what sort of horizontal justification it supports.
 * @return TRUE if the stream justifies horizontally at a point; FALSE
 * if it justifies horizontally within a nominal width.
 */
	public boolean justifyHPoint () {
		return getTraditionalFrame().alignHPoint();
	}

/**
 * Query whether the stream has an unlimited width.
 * <p>
 * The text label can grow arbitrarily in X.  A text region may have a
 * maximum width, or it may allow unlimited horizontal growth.	This
 * method asks the object whether it has an unlimited width.
 * @return TRUE if the width is unlimited; FALSE if there is a limit on
 * the width.
 */
	public boolean unlimitedWidth () {
		return getTraditionalFrame().unlimitedWidth();
	}

/**
 * Query whether the stream has an unlimited height.
 * <p>
 * The text label can grow arbitrarily in Y.  A text region may have a
 * maximum height, or it may allow unlimited vertical growth.  This
 * method asks the object whether it has an unlimited height.
 * @return TRUE if the height is unlimited; FALSE if there is a limit on
 * the height.
 */
	public boolean unlimitedHeight () {
		return getTraditionalFrame().unlimitedHeight();
	}

/**
 * Query the number of comb cells to use for comb justification.
 * <p>
 * Comb justification requires that the number of available cells be
 * known so that shorter text strings can be placed accordingly.  This
 * method returns the number of cells to use.  If the number is zero,
 * the stream's MaxSize() value is used.  If that is also zero, the
 * actual number of _characters_ is used.  The reason for a number of
 * cells separate from MaxSize() is because the latter counts characters
 * and in some languages, several characters may share the same cell
 * (e.g., for separate accent characters).
 */
	public int combCells () {
		return getTraditionalFrame().combCells();
	}

/**
 * Change the number of comb cells to use for comb justification.
 * <p>
 * For more information on this attribute, please see the first overload
 * of this method.
 * @param nCombCells - New number of comb cells.
 */
	public void combCells (int nCombCells) {
		if (nCombCells == combCells()) {
			return;
		}

		getTraditionalFrame().setCombCells (nCombCells);
		if (display() != null) {
			display().update();
		}
	}

/**
 * Assign this displayable stream's content from the given displayable
 * stream.
 * <p>
 * Replace this stream's content with a copy of the content of the given
 * stream.	The graphic source information is <b>not</b> copied.  In
 * other words, fonts will be re-mapped in this stream's font service
 * and attributes will be re-pooled in any attribute pool associated
 * with this stream.  The display pointer is not copied, as each stream
 * has its own display; nor is a display automatically created.  For
 * some reason, this method does not copy the extent or AllowOverflow
 * flag of the source.
 * @param oSource - Stream containing source content to copy.
 */
// Javaport: overriding method not needed if it merely calls super.
//	public void copyFrom (TextDispStr oSource) {
//		super.copyFrom (oSource);
//	}

/**
 * Assign this displayable stream's content from the given base class
 * stream.
 * <p>
 * Replace this stream's content with a copy of the content of the given
 * stream.	The graphic source information is <b>not</b> copied.  In
 * other words, fonts will be re-mapped in this stream's font service
 * and attributes will be re-pooled in any attribute pool associated
 * with this stream.
 * @param oSource - Stream containing source content to copy.
 */
// Javaport: overriding method not needed if it merely calls super.
//	public void copyFrom (TextStream oSource) {
//		super.copyFrom (oSource);
//	}

/**
 * Compare text streams for content equality.
 * <p>
 * Compare this stream against the one passed on the parameter oCompare
 * for content equality.  The graphics sources of the streams are not
 * compared.  To be equal, the streams' content must match in all
 * aspects: raw text, attributes, embedded field content, and so on.
 * The display does not participate in the comparison.	For some reason,
 * the extent and AllowOverflow flag also do not participate in the
 * comparison.
 * @param oCompare - Stream to compare against
 * @return TRUE if the streams are equal; FALSE otherwise.
 */
// Javaport: overriding method not needed if it merely calls super.
//	public boolean equals (Object oCompare) {
//		return super.equals (oCompare);
//	}

/**
 * Compare text streams for content inequality.
 * <p>
 * Compare this stream against the one passed on the parameter oCompare
 * for content inequality.	The graphics sources of the streams are not
 * compared.  This is the exact opposite of the equality comparison.
 * The display does not participate in the comparison.	For some reason,
 * the extent and AllowOverflow flag also do not participate in the
 * comparison.
 * @param oCompare - Stream to compare against
 * @return TRUE if the streams are unequal; FALSE otherwise.
 */
	public boolean notEqual (TextDispStr oCompare) {
		return ! equals (oCompare);
	}

/**
 * Return the stream's minimum width.
 * <p>
 * All streams have both a minimum and maximum width, which may or may
 * not be the same.  This method returns the minimum.
 * @return Minimum width allowed for the stream.  A value of zero is
 * possible, as is a value less than zero.	In the latter case, it
 * indicates that the stream is horizontally justified at a point,
 * though callers should use the JustifyHPoint() method to test for
 * this.
 * @return Minimum width of the displayable stream.
 */
	public UnitSpan minWidth () {
		return getTraditionalFrame().minWidth();
	}

/**
 * Return the stream's minimum height.
 * <p>
 * All streams have both a minimum and maximum height, which may or may
 * not be the same.  This method returns the minimum. @return Minimum
 * height allowed for the stream.  A value of zero is possible, as is a
 * value less than zero.  In the latter case, it indicates that the
 * stream is verrically justified at a point, though callers should use
 * the JustifyVPoint() method to test for this.
 * @return Minimum height of the displayable stream.
 */
	public UnitSpan minHeight () {
		return getTraditionalFrame().minHeight();
	}

/**
 * Return the stream's maximum width.
 * <p>
 * All streams have both a minimum and maximum width, which may or may
 * not be the same.  This method returns the maximum. @return maximum
 * width allowed for the stream.  A value of zero is possible, as is a
 * value less than zero.  In the latter case, it indicates that the
 * stream has unlimited width, though callers should use the
 * UnlimitedWidth() method to test for this.
 * @return maximum width of the displayable stream.
 */
	public UnitSpan maxWidth () {
		return getTraditionalFrame().maxWidth();
	}

/**
 * Return the stream's maximum height.
 * <p>
 * All streams have both a minimum and maximum height, which may or may
 * not be the same.  This method returns the maximum. @return maximum
 * height allowed for the stream.  A value of zero is possible, as is a
 * value less than zero.  In the latter case, it indicates that the
 * stream has unlimited height, though callers should use the
 * Unlimitedheight() method to test for this.
 * @return maximum height of the displayable stream.
 */
	public UnitSpan maxHeight () {
		return getTraditionalFrame().maxHeight();
	}

/**
 * Return the stream's justify height.
 * <p>
 * All display streams have a notion of justify height and width which
 * are the extents by which the justification is to take place.  The
 * default is max height however it can differ from the stream min/max
 * sizes.  Note that this value may never be set, in which case the
 * justification uses default. Callers should call
 * EnforceJustifyHeight() to test if a specific justification height has
 * been set.
 */
	public UnitSpan justifyHeight () {
		return getTraditionalFrame().alignmentHeight();
	}

/**
 * Return TRUE if a specific height is to be used to vertically justify
 * a stream and FALSE otherwise.
 */
	public boolean enforceJustifyHeight () {
		return getTraditionalFrame().enforceAlignmentHeight();
	}

/**
 * Return the stream's justify width.
 * <p>
 * All display streams have a notion of justify height and width which
 * are the extents by which the justification is to take place.  This
 * can be different from the stream min/max sizes.	This value may never
 * be set, in which case the justification uses the stream sizes.
 * Callers should call EnforceJustifyHeight() to test if a specific
 * justification height has been set.
 */
	public UnitSpan justifyWidth () {
		return getTraditionalFrame().alignmentWidth();
	}

/**
 * Return TRUE if a specific width is to be used to horizontally justify
 * a stream and FALSE otherwise.
 */
	public boolean enforceJustifyWidth () {
		return getTraditionalFrame().enforceAlignmentWidth();
	}

// Inherited from class TextSparseStream
	public TextFrame onLoadFrame (int nIndex) {
		assert (false);
		return null;
	}

// proprietary: (for use by class TextDisplay)
	boolean testFitSize () {
		return getTraditionalFrame().testFitSize();
	}

/*
	TextDisplay LoadLayout (TextLayout poLayout, TextScroller poScroller, GFXEnv poGfxEnv) {
		TextDisplay poDisplay = display();
		if (poDisplay == null) {
			poDisplay = new TextDisplay (poScroller, poGfxEnv);
			poDisplay.connectStream (this, true);
		}

		PopulateFrame (0, poLayout);
		poDisplay.create();

		return poDisplay;
	}
*/

/*
	void Format (TextLayout poLayout, boolean bAllowCharGlyphs, TextScroller poScroller, GFXEnv poGfxEnv) {
		TextDisplay poDisplay = CreateDisplay (poScroller, poGfxEnv);
		mpoFrame.format (poLayout, bAllowCharGlyphs);
	}
*/

	TraditionalFrame getTraditionalFrame () {
		assert (mpoFrame != null);
		return mpoFrame;
	}

	void setTraditionalFrame (TraditionalFrame poFrame) {
		assert (mpoFrame == null);
		mpoFrame = poFrame;
		appendFrameRef (poFrame);
	}
}
