/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and may be covered by U.S. and Foreign Patents,
 * patents in process, and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.analytics.sitecatalyst.util;

import java.util.Iterator;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Value;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.commons.json.JSONArray;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SitecatalystJsonItemReader {
    
    /** Default logger */
    private final Logger LOGGER = LoggerFactory.getLogger(getClass());

    // used for incrementally storing the entries
    private String incrementalIndexKey = "";
    private int startIndex = -1;

    public SitecatalystJsonItemReader(String incrementalIndexKey, int startIndex) {
        this.incrementalIndexKey = incrementalIndexKey;
        this.startIndex = startIndex;
    }

    public SitecatalystJsonItemReader() {
    }

    public void readObject(Node parent, String key, JSONObject obj) throws RepositoryException, JSONException {
        Node child;
        if (!parent.hasNode(key)) {
            child = parent.addNode(key);
        } else {
            child = parent.getNode(key);
        }

        Iterator<String> keyIter = obj.keys();
        while (keyIter.hasNext()) {
            String childKey = keyIter.next();
            // do not store any jcr:* properties
            if (childKey.matches("jcr:.+")) {
                LOGGER.debug("skipping JCR property " + childKey);
                continue;
            }
            // CQ-7515 Privilege escalation to admin
            // do not allow keys with path traversal sequences
            if (childKey.matches("(^\\/|.*?\\.\\.\\/).*?")) {
                LOGGER.warn("skipping path traversal sequence " + childKey);
                continue;
            }
            if (!obj.isNull(childKey)) {
                Object value = obj.get(childKey);
                if (value instanceof JSONObject) {
                    readObject(child, childKey, (JSONObject) value);
                } else if (value instanceof JSONArray) {
                    readArray(child, childKey, (JSONArray) value);
                } else if (value.toString().matches("\\d+")) {
                    child.setProperty(childKey, Long.parseLong(value.toString()));
                } else {
                    child.setProperty(childKey, value.toString());
                }
            }

        }
    }

    public void readArray(Node parent, String key, JSONArray array) throws RepositoryException, JSONException {
        Object[] values = new Object[array.length()];
        Node child = null;

        int useIndex = 0;
        if (this.startIndex >= 0
                && StringUtils.isNotBlank(this.incrementalIndexKey)) {
            if (this.incrementalIndexKey.equalsIgnoreCase(key)) {
                useIndex = this.startIndex;
            }
        }

        for (int i = 0; i < array.length(); i++) {
            if (!array.isNull(i)) {
                String childKey = Integer.toString(i + useIndex);
                Object value = array.get(i);
                if (value instanceof JSONObject) {
                    if (child == null) {
                        if (!parent.hasNode(key)) {
                            child = parent.addNode(key);
                        } else {
                            child = parent.getNode(key);
                        }
                    }
                    readObject(child, childKey, (JSONObject) value);
                    // if we added one object, do not add simple values anymore
                    values = null;
                } else if (value instanceof JSONArray) {
                    if (child == null) {
                        if (!parent.hasNode(key)) {
                            child = parent.addNode(key);
                        } else {
                            child = parent.getNode(key);
                        }
                    }
                    readArray(child, childKey, (JSONArray) value);
                    // if we added one array, do not add simple values anymore
                    values = null;
                } else {
                    if (values != null) {
                        values[i] = value;
                    }
                }
            }
        }
        // add as multi value property
        if (values != null) {
            boolean isLong = true;
            for (int i = 0; i < values.length; i++) {
                if (!values[i].toString().matches("\\d+")) {
                    isLong = false;
                    break;
                }
            }
            if (isLong) {
                Value[] longValues = new Value[values.length];
                for (int i = 0; i < values.length; i++) {
                    longValues[i] = parent.getSession().getValueFactory().createValue(
                        Long.parseLong(values[i].toString()));
                }
                parent.setProperty(key, longValues);
            } else {
                String[] stringValues = new String[values.length];
                for (int i = 0; i < values.length; i++) {
                    stringValues[i] = values[i].toString();
                }
                parent.setProperty(key, stringValues);
            }
        }
    }
}