/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import javax.servlet.ServletRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Doctypes Available in the W3C Recommendations
 * @deprecated Use {@link com.day.cq.widget.Doctype} instead
 */
public enum Doctype {

    /**
     * HTML 5 doctype.
     * The DOCTYPE declaration is &lt;!DOCTYPE html&gt; and is case-insensitive in the
     * HTML syntax. DOCTYPEs from earlier versions of HTML were longer because
     * the HTML language was SGML-based and therefore required a reference to a
     * DTD. With HTML5 this is no longer the case and the DOCTYPE is only needed
     * to enable standards mode for documents written using the HTML syntax.
     * Browsers already do this for &lt;!DOCTYPE html&gt;..
     */
    HTML_5("<!DOCTYPE HTML>"),

    /**
     * HTML 4.01 Strict
     * This DTD contains all HTML elements and attributes, but does not include
     * presentational or deprecated elements (like font).
     * Framesets are not allowed.
     */
    HTML_401_STRICT("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01//EN\" " +
            "\"http://www.w3.org/TR/html4/strict.dtd\">"),

    /**
     * HTML 4.01 Transitional
     * This DTD contains all HTML elements and attributes, including
     * presentational and deprecated elements (like font).
     * Framesets are not allowed.
     */
    HTML_401_TRANS("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\" " +
            "\"http://www.w3.org/TR/html4/loose.dtd\">"),

    /**
     * HTML 4.01 Frameset
     * This DTD is equal to HTML 4.01 Transitional, but allows the use of
     * frameset content.
     */
    HTML_401_FRAMESET("<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Frameset//EN\" " +
            "\"http://www.w3.org/TR/html4/frameset.dtd\">"),

    /**
     * XHTML 1.0 Strict
     * This DTD contains all HTML elements and attributes, but does not include
     * presentational or deprecated elements (like font). Framesets are not
     * allowed. The markup must also be written as well-formed XML.
     */
    XHTML_10_STRICT("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\" " +
            "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">"),

    /**
     * XHTML 1.0 Transitional
     * This DTD contains all HTML elements and attributes, including
     * presentational and deprecated elements (like font). Framesets are not
     * allowed. The markup must also be written as well-formed XML.
     */
    XHTML_10_TRANS("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Transitional//EN\" " +
            "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd\">"),

    /**
     * XHTML 1.0 Frameset
     * This DTD is equal to XHTML 1.0 Transitional, but allows the use of
     * frameset content.
     */
    XHTML_10_FRAMESET("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Frameset//EN\" " +
            "\"http://www.w3.org/TR/xhtml1/DTD/xhtml1-frameset.dtd\">"),

    /**
     * XHTML 1.1
     * This DTD is equal to XHTML 1.0 Strict, but allows you to add modules
     * (for example to provide ruby support for East-Asian languages).
     */
    XHTML_11("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\" " +
            "\"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">");

    /**
     * name of the Doctype request attribute
     */
    public static final String REQUEST_ATTRIBUTE_NAME = Doctype.class.getName();

    /**
     * default logger
     */
    private static final Logger log = LoggerFactory.getLogger(Doctype.class);

    /**
     * declaration
     */
    private final String declaration;

    private Doctype(String declaration) {
        this.declaration = declaration;
    }

    /**
     * Returns the declaration string of this doctype.
     * @return declaration string
     */
    public String getDeclaration() {
        return declaration;
    }

    /**
     * Checks if this doctype is XHTML.
     * @return <code>true</code> if this doctype is XHTML.
     */
    public boolean isXHTML() {
        return name().startsWith("XHTML_");
    }

    /**
     * Convenience method that retieves the doctype from the request attribute
     * and checks if the doctype is XHTML.
     *
     * @param req servlet request
     * @return <code>true</code> if the doctype is XHTML.
     * @since 5.3
     */
    public static boolean isXHTML(ServletRequest req) {
        Doctype d = fromRequest(req);
        return d != null && d.isXHTML();
    }

    /**
     * Returns the current doctype of this request.
     * @param req servlet request
     * @return current doctype or <code>null</code> if not set.
     * @since 5.3
     */
    public static Doctype fromRequest(ServletRequest req) {
        com.day.cq.widget.Doctype newDT = com.day.cq.widget.Doctype.fromRequest(req);
        if ( newDT != null ) {
            return Doctype.valueOf(newDT.name());
        }
        Object o = req.getAttribute(REQUEST_ATTRIBUTE_NAME);
        if (o == null) {
            return null;
        } else if (o instanceof Doctype) {
            // update new doctype
            com.day.cq.widget.Doctype.valueOf(((Doctype)o).name()).toRequest(req);
            return (Doctype) o;
        } else  {
            log.error("Object stored in request is not of an instance of Doctype: {}. Returning null.", o);
            return null;
        }
    }

    /**
     * Sets the current doctype of this request.
     * @param req servlet request
     * @since 5.3
     */
    public void toRequest(ServletRequest req) {
        Doctype prev = fromRequest(req);
        if (prev != null && !prev.equals(this)) {
            log.warn("Overwriting existing request doctype ({}) with differnt value {}", prev, this);
        }
        req.setAttribute(REQUEST_ATTRIBUTE_NAME, this);
        // update new doctype
        com.day.cq.widget.Doctype.valueOf(this.name()).toRequest(req);
        if (log.isDebugEnabled()) {
            log.debug("Setting request doctype to {}", this);
        }
    }

}
