/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.commons.inherit;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import com.day.cq.commons.ValueMapWrapper;
import com.day.cq.commons.jcr.JcrConstants;

/**
 * An {@link InheritanceValueMap} for a given {@link Resource} that will inherit values from
 * ancestors up to its ancestral component, but no higher.
 *
 * <p>
 *     For example, given: <code>/content/parent/page/jcr:content/footer/image/@width</code>,
 *     the <code>ComponentInheritanceValueMap</code> will search for a width property in:
 *     <ul>
 *         <li><code>/content/parent/page/jcr:content/footer/image/@width</code></li>
 *         <li><code>/content/parent/page/jcr:content/footer/@width</code></li>
 *         <li><code>/content/parent/page/jcr:content/@width</code></li>
 *     </ul>
 *     Having not found it in any of those locations, it will then return <code>null</code>.
 *
 * <p>
 *     Note that <code>ComponentInheritanceValueMap</code> searches <b>only</b> the component
 *     hierarchy.  It will <b>not</b> (for instance), look in:
 *     <ul>
 *         <li><code>/content/parent/jcr:content/footer/image/@width</code></li>
 *     </ul>
 *     See {@link com.day.cq.commons.inherit.HierarchyNodeInheritanceValueMap} for that
 *     functionality.
 */
public class ComponentInheritanceValueMap extends ValueMapWrapper implements InheritanceValueMap {

    protected Resource resource;

    /**
     * Will wrap the {@link ValueMap} returned by
     * <code>resource.adaptTo(ValueMap.class)</code>, or use an empty map if
     * <code>resource</code> is null or if it doesn't adapt to a ValueMap. The
     * inheritance is internally resolved by fetching the parent resource and
     * wrapping it into an {@link ComponentInheritanceValueMap} as well.
     *
     * @param resource
     *            the resource to start from
     */
    public ComponentInheritanceValueMap(Resource resource) {
        super(ResourceUtil.getValueMap(resource));
        this.resource = resource;
    }

    /**
     * Use this if the underlying {@link Resource} for a {@link ValueMap} is not
     * available and no inheritance is needed. Using the inheritance-enabled
     * {@link #getInherited(String, Class) getter}
     * {@link #getInherited(String, Object) methods} will behave exactly like
     * the normal ValueMap getters.
     *
     * @param map
     *            a ValueMap to wrap
     */
    public ComponentInheritanceValueMap(ValueMap map) {
        super(map);
        this.resource = null;
    }

    @SuppressWarnings("unchecked")
    public <T> T get(String name, Class<T> type) {
        // overwritten to fix NPE
        if (type == null) {
            return (T) get(name);
        }
        return super.get(name, type);
    }

    public <T> T getInherited(String name, Class<T> type) {
        T value = get(name, type);
        if (value == null) {
            value = getParentComponentValue(name, type);
        }
        return value;
    }

    @SuppressWarnings("unchecked")
    public <T> T getInherited(String name, T defaultValue) {
        Class<T> type;
        if (defaultValue == null) {
            type = null;
        } else {
            // special handling in case the default value implements one
            // of the interface types supported by the convertToType method
            type = (Class<T>) defaultValue.getClass();
        }

        T value = getInherited(name, type);
        if (value == null) {
            value = defaultValue;
        }

        return value;
    }

    protected <T> T getParentComponentValue(final String name, final Class<T> type) {
        // stop inheritance in case resource is null or a page's content resource
        if (resource == null || resource.getName().equals(JcrConstants.JCR_CONTENT)) {
            return null;
        }

        Resource parent = resource.getParent();
        if(parent == null) {
            return null;
        }

        return new ComponentInheritanceValueMap(parent).getInherited(name, type);
    }
}
