/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.dam.scene7.api;

import java.util.List;
import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.annotation.versioning.ProviderType;

/**
 * Provides access to the Scene7 cloud configuration
 */
@ProviderType
public interface S7ConfigResolver {

    /**
     * @return  *new* ResourceResolver able to read Scene7 Configurations.
     *          WARNING: caller is responsible to close it!
     */
    ResourceResolver getS7ConfigReaderResolver();

    /**
     * Returns a S7Config with the corresponding parameters applied.
     * @param resourceResolver - resource resolver to access Scene7 configuration
     * @param email - scene7 account name used for connect to Scene7 server
     * @param password - scene7 account password used for connect to Scene7 server
     * @param region - scene7 server region
     * @return the S7Config that match provided username, password and region
     */
    S7Config getS7Config(ResourceResolver resourceResolver, String email,
            String password, String region);

    /**
     * Returns the S7Config generated from the configPath
     * 
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param configPath
     *            - path to the target s7 cloud configPath settings
     * @return S7Config referenced by passed in configPath, return null if cannot get s7Config
     */
    S7Config getS7Config(ResourceResolver resourceResolver, String configPath);

    /**
     * Returns a copy of the given S7Config pointing to the shared company
     * 
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param s7Config
     *            - config to copy from
     * @return an s7Config configured for the shared company
     */
    S7Config getSharedS7Config(ResourceResolver resourceResolver,
            S7Config s7Config);

    /**
     * Returns all the S7 configurations loaded from the children of a given
     * root path.
     * It's deprecated from AEM6.4 due to 0DT change, this can only be used for retrieving old configuration under /etc
     * use getS7Configurations(ResourceResolver resourceResolver) instead
     *
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param configRootPath
     *            - root path where the configurations will be loaded from
     * @return list of <code>S7Config</code> configurations
     */
    @Deprecated
    List<S7Config> getS7Configurations(ResourceResolver resourceResolver, String configRootPath);

    /**
     * Returns all the S7 configurations, include both old configuration(under /etc) and new configuration after 0DT change(under /conf)
     *
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @return list of <code>S7Config</code> configurations
     */
    List<S7Config> getS7Configurations(ResourceResolver resourceResolver);

    /**
     * Returns default S7 configuration
     * Deprecated after 0DT chagne in AEM6.4, use the getDefaultS7Config(ResourceResolver resourceResolver) instead
     * 
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @param configRootPath
     *            - root path where the configurations will be loaded from
     * @return default <code>S7Config</code> configuration object
     */
    @Deprecated
    S7Config getDefaultS7Config(ResourceResolver resourceResolver,
            String configRootPath);

    /**
     * Returns default S7 configuration
     * If run under DMS7 mode, return DMS7 configuration, otherwise, return S7Connect configuration
     *
     * @param resourceResolver
     *            - resource resolver to access Scene7 configuration
     * @return default <code>S7Config</code> configuration object
     */
    S7Config getDefaultS7Config(ResourceResolver resourceResolver);

    /**
     * Returns default S7 configuration, resolved using the S7 config service user
     * If run under DMS7 mode, return DMS7 configuration, otherwise, return S7Connect configuration
     *
     * @return default <code>S7Config</code> configuration object
     * @deprecated WARNING: this method embeds the new ResourceResolver into returned S7Config instance!
     *   Caller is responsible to call {@link S7Config#close()},
     *     or use {@link #getDefaultS7Config(ResourceResolver)} instead.
     */
    @Deprecated
    S7Config getDefaultS7Config();

    /**
     * Retrieves the Scene7 configuration associated with a DAM asset resource
     * by successively analysing the resource and then its parents. If no
     * S7Config is found this method returns default configuration instead.
     *
     * @param resourceResolver  - resource resolver to access Scene7 configuration
     * @param resource          - a DAM asset resource
     * @return the associated Scene7 configuration or default configuration,
     *          null if no configuration and no default configuration found
     */
    S7Config getS7ConfigForResource(ResourceResolver resourceResolver, Resource resource);

    /**
     * Retrieves the Scene7 configuration associated with a DAM asset resource
     * by successively analysing the resource and then its parents. If no
     * S7Config is found this method returns default configuration instead.
     *
     * @param resourceResolver  - resource resolver to access Scene7 configuration
     * @param resource          - a DAM asset resource
     * @param ignoreSelectiveSync          - ignore selective sync and return default config 
     * @return the associated Scene7 configuration or default configuration,
     *          null if no configuration and no default configuration found
     */
    S7Config getS7ConfigForResource(ResourceResolver resourceResolver, Resource resource, boolean ignoreSelectiveSync);

    /**
     * Retrieves the Scene7 configuration associated with a DAM asset resource
     * by successively analysing the resource and then its parents. If no
     * S7Config is found this method returns null.
     * 
     * @param resource
     *            a DAM asset resource
     * @return the associated Scene7 configuration; null if no configuration is
     *         found
     * @deprecated  this method returns unreadable config!
     *    Use {@link #getS7ConfigForResource(ResourceResolver, Resource)} instead.
     */
    @Deprecated
    S7Config getS7ConfigForResource(Resource resource);

    /**
     * Retrieves the Scene7 configuration associated with an image server URL.
     * Returns null if no S7Config is found.
     * 
     * @param resourceResolver
     *            resource resolver to access Scene7 configuration
     * @param imageUrl
     *            the image server URL
     * @return return associated Scene7 configuration; null if no configuration
     *         is found
     * @deprecated this method is broken
     */
    @Deprecated
    S7Config getS7ConfigForImageUrl(ResourceResolver resourceResolver,
            String imageUrl);

    /**
     * Retrieves the Scene7 configuration associated with asset by content path
     * when Scene7 metadata is not available
     *
     * @param resourceResolver resource resolver to access Scene7 configuration
     * @param assetPath        the asset content path
     * @return associated Scene7 configuration; null if no configuration is found
     */
    S7Config getS7ConfigForAssetPath(ResourceResolver resourceResolver,
            String assetPath);

    /**
     * Looks up scene7 configuration for the specified root id (company name?)
     * @param resourceResolver resource resolver to access Scene7 configuration
     * @param rootId catalog root id
     * @return S7Config object that corresponds to the specified rootId or null if none found.
     */
    S7Config getS7ConfigForRootId(ResourceResolver resourceResolver, String rootId);

    /**
     * Retrieves the Scene7 configuration path associated with a DAM asset
     * resource by successively analysing the resource and then its parents. If
     * no S7Config is found this method returns null.
     * 
     * @param resolver
     *            resource resolver associated with user session
     * @param resource
     *            a DAM asset resource
     * @return the associated Scene7 configuration path; null if no
     *         configuration is found
     */
    String getS7ConfigPathForResource(ResourceResolver resolver,
            Resource resource);

    /**
     * Retrieves the Scene7 configuration associated with an image server URL.
     * Returns null if no S7Config is found.
     * 
     * @param resolver
     *            resource resolver associated with user session
     * @param domain
     *            delivery domain associated with configuration
     * @param companyName
     *            companyName associated with configuration
     * @return return associated Scene7 configuration path; null if no
     *         configuration is found
     */
    String getS7ConfigPathForImageUrl(ResourceResolver resolver, String domain,
            String companyName);

    /**
     * Helper method to extract domain and company information related to an
     * image url
     * 
     * @param imageUrl
     *            image url embedding the domain and company information
     * @return return a map with keys 'domain' and 'companyName' if available
     */
    Map<String, String> parseDomainAndCompanyNameForUrl(String imageUrl);

    /**
     * Retrieves the Scene7 configuration associated with an asset path. Returns
     * null if no S7Config is found.
     * 
     * @param resolver
     *            resource resolver associated with user session
     * @param assetPath
     *            asset path associated with configuration
     * @return return associated Scene7 configuration path; null if no
     *         configuration is found
     */
    String getS7ConfigPathForAssetPath(ResourceResolver resolver,
            String assetPath);

    /**
     * Temporary method for checking if DMS7 feature is enalbed
     * @return return if the DMS7 mode is enabled
     */
    Boolean isDMS7Enabled();

    /**
     * Sets the preview server jwt token and expirty for Scene7 configuration provided
     *
     * @param token
     *            String containing JWT token
     * @param s7config
     *            Scene7Config for which token has to be set
     */
    void setPreviewServerTokenForS7Config(String token, S7Config s7config);

    /**
     * Scene7 region (server mapping)
     */
    public static final String REGION = "region";

    /**
     * Scene7 User email ID
     */
    public static final String EMAIL = "email";

    /**
     * Scene7 User password ID
     */
    public static final String PASSWORD = "password";

    /**
     * Scene7 generated user handle ID
     */
    public static final String USER_HANDLE = "userHandle";

    /**
     * Scene7 generated company handle ID
     */
    public static final String COMPANY_HANDLE = "companyHandle";

    /**
     * The "root path" ID in the S7 sense...see getRootPath()
     */
    public static final String ROOT_PATH = "rootPath";

    /**
     * cq root folder ID in the scene7 company folder hierarchy, is not the "root path" in the S7 (see getRootPath())
     */
    public static final String CQ_ROOT_PATH = "cqRootPath";

    /**
     * controls s7 updates and synchronization of assets/folders already managed by s7
     */
    public static final String SYNC_ENABLED = "syncEnabled";

    /**
     * controls s7 publish state updates and synchronization of assets/folders
     * already managed by s7
     */
    public static final String PUBLISH_ENABLED = "publishEnabled";

    /**
     * Default char set
     */
    public static final String CHARSET = "UTF-8";

    /**
     * The Scene7 publish server URL.
     */
    public static final String PUBLISH_SERVER = "publishServer";

    /**
     * The Scene7 preview server URL.
     */
    public static final String PREVIEW_SERVER = "previewServer";

    /**
     * The Scene7 preview server jwt for authorization.
     */
    public static final String PREVIEW_SERVER_TOKEN = "previewServerToken";

    /**
     * The Scene7 preview server jwt expiry.
     */
    public static final String PREVIEW_SERVER_TOKEN_EXPIRY = "previewServerTokenExpiry";

    /**
     * The Scene7 flash templates publish server URL.
     */
    public static final String FLASH_TEMPLATES_SERVER = "flashTemplatesServer";

    /**
     * The target path property, storing the path where Scene7 assets are downloaded into CQ DAM.
     */
    public static final String TARGET_PATH = "targetPath";

    /**
     * Stores the default configuration property flag
     */
    public static final String DEFAULT_CONFIG_FLAG = "defaultConfiguration";

    public static final String ADHOC_FOLDER = "adhocFolder";

    /**
     * The Scene7 FXG server URL.
     */
    @Deprecated
    public static final String FXG_SERVER = "fxgServer";

    /**
     * The Scene7 import queue name
     */
    public static final String IMPORT_QUEUE_NAME = "importQueueName";

    /**
     * The import queue topic
     */
    public static final String IMPORT_QUEUE_TOPIC = "importQueueTopic";

    /**
     * The companyName
     */
    public static final String COMPANY_NAME = "companyname";

    /**
     * The companyAlias, only supported in DMS7
     */
    public static final String COMPANY_ALIAS = "companyAlias";

    /**
     * Scene7 generated company handle ID for the company corresponding to the companyAlias
     * Only supported in DMS7
     */
    public static final String COMPANY_ALIAS_HANDLE = "companyAliasHandle";
}
