/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.mcm.api;

import java.util.Collection;
import java.util.HashMap;

import org.apache.sling.api.resource.Resource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.mcm.util.NormalizedResource;

/**
 * Helps when implementing {@link MCMPlugin}.
 * Provides the root component functionality with a list of components manageable
 * via {@link #addExperienceComponent(String, Factory)} and {@link #addTouchpointComponent(String, Factory)}. 
 * You can also add only one (Experience or touchpoint), or many.
 */
public abstract class MCMPluginAdapter implements MCMPlugin {

	private final static Logger log = LoggerFactory.getLogger(MCMPluginAdapter.class);
	
	public static interface Factory<T> {
		public T create(Resource r);
	}
	
	private HashMap<String, Factory<Experience>> experienceFactoryMap = new HashMap<String, Factory<Experience>>();
	
	private HashMap<String, Factory<Touchpoint>> touchpointFactoryMap = new HashMap<String, Factory<Touchpoint>>();
	
	public MCMPluginAdapter() {
		super();
	}
	
	/**
	 * Returns the comptypes that were registered with {@link #addTouchpointComponent(String, Factory)}. 
	 */
	public Collection<String> getTouchpointRootComponents() {
		return touchpointFactoryMap.keySet();
	}
	
	/**
	 * Returns the comptypes that were registered with {@link #addExperienceComponent(String, Factory)}.
	 */
	public Collection<String> getExperienceRootComponents() {
		return experienceFactoryMap.keySet();
	}
	
	/**
	 * Use this to add your types to the adapter.
	 * @param compType
	 * @param fac
	 */
	protected void addExperienceComponent(String compType, Factory<Experience> fac) {
		if (fac == null) {
			throw new NullPointerException("Factory null");
		}
		experienceFactoryMap.put(compType, fac);
	}
	
	/**
	 * Use this to add your types to the adapter.
	 * @param compType
	 * @param fac
	 */
	protected void addTouchpointComponent(String compType, Factory<Touchpoint> fac) {
		if (fac == null) {
			throw new NullPointerException("Factory null");
		}
		touchpointFactoryMap.put(compType, fac);
	}

	public Experience makeExperience(Resource adaptable) {
		Experience retval = null;
		String compTypeToSearch = extractCompType(adaptable);
		if (compTypeToSearch == null) {
			throw new RuntimeException("Given resource doesn't have a resource type: " 
					+ adaptable.getPath());
		}
		if (experienceFactoryMap.containsKey(compTypeToSearch)) {
			Factory<Experience> fac = experienceFactoryMap.get(compTypeToSearch);
			retval = fac.create(adaptable);
		} else {
			throw new RuntimeException("Unkown resourceType (misuse of the the class? We have: "
					+ experienceFactoryMap.toString() + "): " 
					+ compTypeToSearch);
		}
		return retval;
	}

	public Touchpoint makeTouchpoint(Resource adaptable) {
		if (adaptable == null) {
			throw new NullPointerException("Null passed as Resource adaptable.");
		}
		
		Touchpoint retval = null;
		String compTypeToSearch = extractCompType(adaptable);
		if (compTypeToSearch == null) {
			throw new RuntimeException("Given resource doesn't have a resource type: " 
					+ adaptable.getPath());
		}
		if (touchpointFactoryMap.containsKey(compTypeToSearch)) {
			Factory<Touchpoint> fac = touchpointFactoryMap.get(compTypeToSearch);
			retval = fac.create(adaptable);
		} else {
			throw new RuntimeException("Unkown resourceType (misuse of the the class? We have: "
					+ touchpointFactoryMap.toString() + "): " 
					+ compTypeToSearch);
		}
		return retval;
	}

	private String extractCompType(Resource adaptable) {
		NormalizedResource nr = new NormalizedResource();
		nr.setResource(adaptable);
		String retval = null;
		if (nr.getContentVals() != null) {
			retval = nr.getContentVals().get("sling:resourceType", String.class);
		}
		if (retval == null) {
			log.warn("Cannot make an object because resource {} doesn't have content node or resourceType.", adaptable);
		}
		return retval;
	}
	
	@Override
	public String toString() {
		return super.toString() + "/id:" + getPluginId();
	}
}
