/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.core.stats;

import java.util.Calendar;

import javax.jcr.Item;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.ValueFormatException;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.WCMMode;
import com.day.crx.statistics.Entry;
import com.day.crx.statistics.PathBuilder;

/**
 * <code>PageView</code> provides an {@link Entry} for storing {@link Page}
 * related statistics.
 *
 * @author asaar
 */
public class PageView extends Entry {
    /** Name of the property that contains the view count */
    public static final String VIEWS = "views";

    /** Name of the property that contains the rolling week count */
    public static final String ROLLING_WEEK_COUNT = "rollingWeekViews";

    /** Name of the property that contains the rolling month count */
    public static final String ROLLING_MONTH_COUNT = "rollingMonthViews";

    /** The page */
    private final Page page;

    /** The WCM mode */
    private final WCMMode mode;

    /**
     * Creates a new <code>PageView</code> instance.
     *
     * @param pathPrefix
     *            the location where the entry information will be stored
     * @param page
     *            the page that was viewed
     * @param mode
     *            the WCM mode that was used to view the page
     */
    public PageView(String pathPrefix, Page page, WCMMode mode) {
        super(pathPrefix);
        this.page = page;
        this.mode = mode;
    }

    /** {@inheritDoc} */
    protected PathBuilder getPathBuilder() {
        return new PageViewPathBuilder(page.getPath());
    }

    /**
     * Writes the statistics to the passed node.
     *
     * @param node
     *            the node where to write the statistics
     * @throws RepositoryException
     *             if an error occurs while writing
     */
    @Override
    public void write(Node node) throws RepositoryException {
        Node month = node.getParent();
        Node year = month.getParent();

        updateViews(node);
        updateViews(month);
        updateViews(year);

        updateCumulativeViews(node, ROLLING_WEEK_COUNT, 6);
        updateCumulativeViews(node, ROLLING_MONTH_COUNT, 29);
    }

    /**
     * Increments the view count on the given <code>node</code>.
     *
     * @param node
     *            the node where to increment the view count
     * @throws RepositoryException
     *             if an error occurs while writing to the node.
     */
    private void updateViews(Node node) throws RepositoryException {
        long viewCount = 0;
        if (node.hasProperty(VIEWS)) {
            viewCount = node.getProperty(VIEWS).getLong();
        }
        node.setProperty(VIEWS, ++viewCount);

        long modeViewCount = 0;
        String modeViewProp = VIEWS + "-" + mode;
        if (node.hasProperty(modeViewProp)) {
            modeViewCount = node.getProperty(modeViewProp).getLong();
        }
        node.setProperty(modeViewProp, ++modeViewCount);
    }

    /**
     * Updates a cumulative view count on the <code>node</code>.
     *
     * @param node
     *            the node where to update the cumulative view count
     * @param propertyName
     *            the name of the count property
     * @param numDays
     *            the number of days back in time that are cumulated
     * @throws RepositoryException
     *             if an error occurs while reading or updating.
     */
    private void updateCumulativeViews(Node node, String propertyName,
            int numDays) throws RepositoryException {
        long viewCount;
        if (node.hasProperty(propertyName)) {
            viewCount = node.getProperty(propertyName).getLong();
        } else {
            viewCount = getCumulativeCount(node, numDays, VIEWS);
        }
        node.setProperty(propertyName, ++viewCount);

        long modeViewCount;
        propertyName = propertyName + "-" + mode;
        if (node.hasProperty(propertyName)) {
            modeViewCount = node.getProperty(propertyName).getLong();
        } else {
            modeViewCount = getCumulativeCount(node, numDays, VIEWS + "-" + mode);
        }
        node.setProperty(propertyName, ++modeViewCount);
    }

    private long getCumulativeCount(Node node, int numDays, String propName)
            throws RepositoryException, ValueFormatException {
        long viewCount = 0;
        Session session = node.getSession();
        PathBuilder builder = getPathBuilder();
        Calendar date = Calendar.getInstance();
        date.setTimeInMillis(getTimestamp());
        PageView view = new PageView(getPathPrefix(), page, mode);
        StringBuffer buffer = new StringBuffer();
        for (int i = 0; i < numDays; i++) {
            // re-use buffer
            buffer.setLength(0);

            // go back one day
            date.add(Calendar.DAY_OF_MONTH, -1);
            view.setTimestamp(date.getTimeInMillis());
            builder.formatPath(view, buffer);
            String path = buffer.toString();
            try {
                Item item = session.getItem(path);
                if (item.isNode()) {
                    Node n = (Node) item;
                    if (n.hasProperty(propName)) {
                        viewCount += n.getProperty(propName).getLong();
                    }
                }
            } catch (PathNotFoundException e) {
                // no statistics found for that day
            }
        }
        return viewCount;
    }
}
