/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.Report;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.NodeIterator;
import javax.jcr.Node;
import javax.jcr.Value;
import javax.jcr.query.QueryManager;
import javax.jcr.query.Query;
import java.util.Iterator;
import java.util.Calendar;
import java.util.List;
import java.util.ArrayList;
import java.text.SimpleDateFormat;

import org.apache.jackrabbit.util.ISO9075;

/**
 * <code>MostPopularResultsReport</code> implements a report that returns the
 * most popular results in a given period.
 *
 * @author mreutegg
 */
public class MostPopularResultsReport extends Report {

    /**
     * Number of results to return at most. Default value: top 10 results.
     */
    private int size = 10;

    /**
     * Create a report for the month, as indicated by this date.
     */
    private long date = System.currentTimeMillis();

    /**
     * Creates a new {@link MostPopularResultsReport}.
     *
     * @param dataPath the path where the results statistics are stored.
     */
    public MostPopularResultsReport(String dataPath) {
        super(dataPath);
    }

    /**
     * {@inheritDoc}
     * Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code>, which indicates the month</li>
     * <li><code>Long</code> count (how may times a result was selected)</li>
     * <li><code>String[]</code> with recent queries that lead to this result.</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM");
        Calendar cal = Calendar.getInstance();
        cal.setTimeInMillis(getDate());
        QueryManager qm = session.getWorkspace().getQueryManager();
        StringBuffer stmt = new StringBuffer("/jcr:root");
        stmt.append(getDataPath());
        stmt.append("//");
        stmt.append(ISO9075.encode(ResultSelectedPathBuilder.STATS_NAME));
        stmt.append("/");
        stmt.append(ISO9075.encodePath(sdf.format(cal.getTime())));
        stmt.append(" order by @count descending");
        NodeIterator nodes = qm.createQuery(stmt.toString(), Query.XPATH).execute().getNodes();
        List data = new ArrayList();
        int num = getSize();
        while (nodes.hasNext() && num-- > 0) {
            Node n = nodes.nextNode();
            long count = n.getProperty(ResultSelected.COUNT).getLong();
            // get path of content node
            String path = n.getAncestor(n.getDepth() - 3).getPath();
            path = path.substring(getDataPath().length());
            data.add(new Object[]{path, new Long(count), getQueries(n)});
        }
        return data.iterator();
    }

    /**
     * Returns the recent queries as a String array.
     *
     * @param n the node where the recent queries are stored.
     * @return the recent queries.
     * @throws RepositoryException if an error occurs while reading from the
     *                             workspace.
     */
    private String[] getQueries(Node n) throws RepositoryException {
        if (n.hasProperty(ResultSelected.QUERIES)) {
            Value[] values = n.getProperty(ResultSelected.QUERIES).getValues();
            String[] queries = new String[values.length];
            for (int i = 0; i < queries.length; i++) {
                queries[i] = values[i].getString();
            }
            return queries;
        } else {
            return new String[0];
        }
    }

    //-----------------------------< properties >-------------------------------

    /**
     * @return the maximum number of results to return.
     */
    public int getSize() {
        return size;
    }

    /**
     * @param size the maximum number of results to return.
     */
    public void setSize(int size) {
        this.size = size;
    }

    /**
     * @param date the report covers the month as indicated by
     *             <code>date</code>.
     */ 
    public void setDate(long date) {
        this.date = date;
    }

    /**
     * @return the report covers the month as indicated by <code>date</code>.
     */
    public long getDate() {
        return date;
    }
}
