/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.JanitorReport;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.NodeIterator;
import javax.jcr.Node;
import javax.jcr.query.QueryManager;
import javax.jcr.query.Query;
import java.util.Iterator;
import java.util.Calendar;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.ArrayList;
import java.util.List;
import java.util.Comparator;

import org.apache.jackrabbit.util.ISO9075;

/**
 * <code>RemoveOldResultsReport</code> implements a report, which removes old
 * result statistics based on two threshold dates, one for monthly and another
 * one for daily statistics.
 *
 * @author mreutegg
 */
public class RemoveOldResultsReport extends JanitorReport {

    /**
     * Add OAK-specific "option(traversal ok)" to the statistics query so that no traversal warning is triggered
     */
    private boolean traversalOk = false;

    /**
     * Creates a new report.
     *
     * @param dataPath the path where result data is stored.
     * @param dryRun   whether the data should actually be removed.
     */
    public RemoveOldResultsReport(String dataPath, boolean dryRun) {
        super(dataPath, dryRun);
    }

    /**
     * Creates a new report.
     *
     * @param dataPath    the path where result data is stored.
     * @param dryRun      whether the data should actually be removed.
     * @param traversalOk set to true to put "traveral ok" option in the statistics query
     */
    public RemoveOldResultsReport(String dataPath, boolean dryRun, boolean traversalOk) {
        super(dataPath, dryRun);
        this.traversalOk = traversalOk;
    }
    /**
     * {@inheritDoc}
     * <p/>
     * Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code> of the node that was removed.</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        SortedSet removable = new TreeSet(new Comparator() {
            public int compare(Object o1, Object o2) {
                String[] p1 = split((String) o1);
                String[] p2 = split((String) o2);
                int c = p2[1].compareTo(p1[1]);
                if (c != 0) {
                    return c;
                } else {
                    return p1[0].compareTo(p2[0]);
                }
            }

            private String[] split(String path) {
                String[] parts = new String[2];
                int idx = path.indexOf("/" + ResultSelectedPathBuilder.STATS_NAME + "/");
                parts[0] = path.substring(0, idx);
                parts[1] = path.substring(idx + ResultSelectedPathBuilder.STATS_NAME.length() + 2);
                return parts;
            }
        });
        // get all .stats nodes
        QueryManager qm = session.getWorkspace().getQueryManager();
        StringBuffer stmt = new StringBuffer("/jcr:root");
        stmt.append(getDataPath()).append("//");
        stmt.append(ISO9075.encode(ResultSelectedPathBuilder.STATS_NAME));
        if (traversalOk) {
            stmt.append(" option(traversal ok)");
        }
        NodeIterator stats = qm.createQuery(stmt.toString(),
                Query.XPATH).execute().getNodes();
        Calendar cal = Calendar.getInstance();
        while (stats.hasNext()) {
            Node s = stats.nextNode();
            for (NodeIterator years = s.getNodes(); years.hasNext(); ) {
                // never remove years
                Node y = years.nextNode();
                cal.set(Calendar.YEAR, Integer.parseInt(y.getName()));
                for (NodeIterator months = y.getNodes(); months.hasNext(); ) {
                    Node m = months.nextNode();
                    cal.set(Calendar.MONTH, Integer.parseInt(m.getName()) - 1);
                    if (cal.getTimeInMillis() < getMonthThresholdDate()) {
                        removable.add(m.getPath());
                    } else {
                        // check days
                        for (NodeIterator days = m.getNodes(); days.hasNext(); ) {
                            Node d = days.nextNode();
                            cal.set(Calendar.DAY_OF_MONTH, Integer.parseInt(d.getName()));
                            if (cal.getTimeInMillis() < getDayThresholdDate()) {
                                removable.add(d.getPath());
                            }
                        }
                    }
                }
            }
        }
        List result = new ArrayList();
        for (Iterator it = removable.iterator(); it.hasNext(); ) {
            String path = (String) it.next();
            if (!isDryRun()) {
                session.getItem(path).remove();
            }
            result.add(new Object[]{path});
        }
        if (!isDryRun()) {
            session.save();
        }
        return result.iterator();
    }
}
