/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.workflow.core.event;

import static com.adobe.granite.workflow.event.WorkflowEvent.DELEGATEE;
import static com.adobe.granite.workflow.event.WorkflowEvent.RETURNED_TO;
import static com.adobe.granite.workflow.event.WorkflowEvent.EVENT_TYPE;
import static com.adobe.granite.workflow.event.WorkflowEvent.FROM_NODE_NAME;
import static com.adobe.granite.workflow.event.WorkflowEvent.JOB_FAILED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.MODEL_DELETED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.MODEL_DEPLOYED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.NODE_TRANSITION_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.PARENT_WORKFLOW_ID;
import static com.adobe.granite.workflow.event.WorkflowEvent.PROCESS_TIMEOUT_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.RESOURCE_COLLECTION_MODIFIED;
import static com.adobe.granite.workflow.event.WorkflowEvent.TIME_STAMP;
import static com.adobe.granite.workflow.event.WorkflowEvent.TO_NODE_NAME;
import static com.adobe.granite.workflow.event.WorkflowEvent.USER;
import static com.adobe.granite.workflow.event.WorkflowEvent.VARIABLE_NAME;
import static com.adobe.granite.workflow.event.WorkflowEvent.VARIABLE_UPDATE_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.VARIABLE_VALUE;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_ABORTED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_COMPLETED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_INSTANCE_ID;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_NAME;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_NODE;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_RESUMED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_STARTED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_SUSPENDED_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKFLOW_VERSION;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKITEM_DELEGATION_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKITEM_UNCLAIM_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORKITEM_CLAIM_EVENT;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORK_DATA;
import static com.adobe.granite.workflow.event.WorkflowEvent.WORK_ITEM;
import java.util.Date;
import java.util.Dictionary;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import javax.jcr.RepositoryException;

import com.adobe.granite.workflow.core.jcr.WorkflowManager;
import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.api.security.user.Authorizable;
import org.apache.sling.api.SlingException;
import org.osgi.framework.InvalidSyntaxException;
import org.osgi.framework.ServiceReference;
import org.osgi.service.component.ComponentContext;
import org.osgi.service.event.EventAdmin;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.granite.workflow.WorkflowException;
import com.adobe.granite.workflow.WorkflowSession;
import com.adobe.granite.workflow.core.exec.WorkflowImpl;
import com.adobe.granite.workflow.core.exec.WorkflowDataImpl;
import com.adobe.granite.workflow.core.util.WorkflowUtil;
import com.adobe.granite.workflow.event.WorkflowEvent;
import com.adobe.granite.workflow.exec.WorkItem;
import com.adobe.granite.workflow.exec.Workflow;
import com.adobe.granite.workflow.exec.WorkflowData;
import com.adobe.granite.workflow.job.AbsoluteTimeoutHandler;
import com.adobe.granite.workflow.job.AbsoluteTimeoutHandlerProxy;
import com.adobe.granite.workflow.job.ExternalProcessJob;
import com.adobe.granite.workflow.job.TimeoutJob;
import com.adobe.granite.workflow.job.WorkflowJob;
import com.adobe.granite.workflow.metadata.MetaDataMap;

/**
 * The <code>EventsPublisher</code> provides a utility for publishing workflow
 * related events.
 */
public class EventPublishUtil {
    /**
     * Logger instance for this class.
     */
    private static final Logger log = LoggerFactory.getLogger(EventPublishUtil.class);

    /**
     * The OSGI event admin used for sending events
     */
    private EventAdmin eventAdmin;

    public EventPublishUtil(EventAdmin eventAdmin) {
        this.eventAdmin = eventAdmin;
    }

    static private boolean isTransient(Workflow workflow) {
        Boolean isTransient = Boolean.FALSE;
        MetaDataMap metaMap = workflow.getWorkflowData().getMetaDataMap();
        isTransient =  metaMap.get("isTransient",  Boolean.class);
        log.trace("workflow.data.MetaDataMap.isTransient: {}", isTransient);
        return isTransient == null ? false : isTransient;
    }

    /**
     * Notifies all registered listeners about an event.
     *
     * @param props event properties
     */
    private void sendEvent(Dictionary<String, Object> props) {
        if (log.isDebugEnabled()) {
            log.debug("Sending workflow event of type " + props.get(EVENT_TYPE)
                    + ((props.get(WORKFLOW_INSTANCE_ID) != null) ? " for " + props.get(WORKFLOW_INSTANCE_ID) : ""));
        }
        eventAdmin.postEvent(new WorkflowEvent(props));
    }

    public void publishModelDeployedEvent(String id, String version, String user) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, MODEL_DEPLOYED_EVENT);
        properties.put(WORKFLOW_NAME, id);
        if (version != null) {
            properties.put(WORKFLOW_VERSION, version);
        } else {
            log.warn("No version specified"); // TODO: make sure that version
            // is specified...
        }
        sendEvent(properties);
    }

    public void publishModelDeletedEvent(String id, String user) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, MODEL_DELETED_EVENT);
        properties.put(WORKFLOW_NAME, id);
        sendEvent(properties);
    }

    public void publishWorkflowStartedEvent(Workflow instance, Workflow parentWorkflow, String user, WorkflowData data) {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKFLOW_STARTED_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        // properties.put(PARENT_WORKFLOW_ID, parentWorkflow.getId());
        properties.put(WORK_DATA, data);
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishWorkflowAbortedEvent(Workflow instance, String userId) {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, userId);
        properties.put(EVENT_TYPE, WORKFLOW_ABORTED_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishWorkflowSuspendedEvent(Workflow instance, String user) {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKFLOW_SUSPENDED_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishWorkflowResumedEvent(Workflow instance, String user) {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKFLOW_RESUMED_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishWorkflowCompletedEvent(Workflow instance, String user) throws WorkflowException {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKFLOW_COMPLETED_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());

        //  Avoid having to rev the API to get the parent instance id
        if (instance instanceof WorkflowImpl) {
            String parentId;
            WorkflowImpl wf = (WorkflowImpl) instance;
            log.debug("publishWorkflowCompletedEvent parent id: {}", wf.getParentInstanceId());
            parentId = wf.getParentInstanceId();
            if (StringUtils.isNotEmpty(parentId) && !StringUtils.equalsIgnoreCase(parentId, WorkflowManager.NO_PARENT)) {
                properties.put(PARENT_WORKFLOW_ID, parentId);
                log.debug("publishWorkflowCompletedEvent set PARENT_WORKFLOW_ID: {}", wf.getParentInstanceId());
            }
        }

        addPayloadInfo(instance, properties);
        sendEvent(properties);
    }

    public static final String WORKFLOW_PAYLOAD_MODIFIED_EVENT = "WorkflowPayloadModified";
    public static final String PROP_OLD_PAYLOAD_PATH = "oldPayloadPath";
    public static final String PROP_PAYLOAD_PATH = "payloadPath";

    public void publishWorkflowPayloadModified(Workflow instance, String oldPath, String user) throws WorkflowException {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKFLOW_PAYLOAD_MODIFIED_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());

        if (oldPath != null) {
            properties.put(PROP_OLD_PAYLOAD_PATH, oldPath);
        }

        addPayloadInfo(instance, properties);
        sendEvent(properties);
    }

    private static void addPayloadInfo(Workflow instance, Dictionary<String, Object> properties) {
        if (instance != null &&
                instance.getWorkflowData() != null &&
                WorkflowDataImpl.TYPE_JCR_PATH.equals(instance.getWorkflowData().getPayloadType())) {
            properties.put(PROP_PAYLOAD_PATH, instance.getWorkflowData().getPayload());
        }
    }

    public void publishNodeTransitionEvent(Workflow instance, String fromNodeName, String toNodeName,
                                           WorkItem workitem, String user) {
        if (isTransient(instance)) {
            return;
        }

        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, NODE_TRANSITION_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        properties.put(FROM_NODE_NAME, (fromNodeName  == null) ? "" : fromNodeName); // Handle workflow step with no title.
        properties.put(TO_NODE_NAME, (toNodeName == null) ? "" : toNodeName);
        properties.put(WORK_ITEM, workitem);
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishVariableUpdatedEvent(Workflow instance, String variableName, Object variableValue, String user) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, VARIABLE_UPDATE_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        properties.put(VARIABLE_NAME, variableName);
        properties.put(VARIABLE_VALUE, variableValue);
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishProcessTimeoutEvent(Workflow instance, String processName) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(EVENT_TYPE, PROCESS_TIMEOUT_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    @Deprecated
    public void publishJobEvent(Map workItemMap, Integer retryCount, int numOfParallelProcs, String jobId) {
        WorkflowJob job = new WorkflowJob(workItemMap);
        eventAdmin.postEvent(job.createJobEvent(retryCount, numOfParallelProcs, jobId));
    }

    @Deprecated
    public void publishExternalProcessJobEvent(Map workItemMap, Integer retryCount, String jobId) {
        ExternalProcessJob job = new ExternalProcessJob(workItemMap);
        eventAdmin.postEvent(job.createJobEvent(retryCount, jobId));
    }

    public void publishJobFailedEvent(WorkItem item, String message) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(EVENT_TYPE, JOB_FAILED_EVENT);
        properties.put(WORKFLOW_NODE, item.getNode().getId());
        properties.put(WORKFLOW_NAME, item.getWorkflow().getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, item.getWorkflow().getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, item.getWorkflow().getId());
        addPayloadInfo(item.getWorkflow(), properties);

        sendEvent(properties);
    }

    @Deprecated
    public Long publishTimeoutEvent(WorkItem item, WorkflowSession session) {
        log.debug("entering publishTimeoutEvent..");
        String handler = null;
        Long timeout = null;
        boolean addOffset = true;

        Iterator<String> keys = item.getNode().getMetaDataMap().keySet().iterator();
        while (keys.hasNext()) {
            String key = keys.next();
            if (key.equals("timeoutHandler")) {
                handler = item.getNode().getMetaDataMap().get("timeoutHandler", String.class);
            } else if (key.equals("timeoutMillis")) {
                timeout = item.getNode().getMetaDataMap().get("timeoutMillis", Long.class);
            }
        }

        // check for absolute timeout handler
        AbsoluteTimeoutHandler handlerImpl = handler != null ? getHandler(handler, session) : null;
        if (handlerImpl != null) {
            long handlerTimeout = handlerImpl.getTimeoutDate(item);
            if (handlerTimeout > 0) {
                // overwrite with handler metadata if set
                timeout = handlerTimeout;
                addOffset = false;
            }

            if (timeout > 0) {
                timeout = timeout / 1000; // convert into seconds
                log.debug("publishTimeoutEvent: Using AbsoluteTimeoutHandler. Timeout is: " + timeout + "s");
            } else {
                log.debug("publishTimeoutEvent: no time set");
                timeout = null;
            }
        }

        if ((handler != null) && (timeout != null)) {
            TimeoutJob job = new TimeoutJob(item, handler);
            eventAdmin.postEvent(job.createEvent(true, timeout, addOffset));
        }

        return timeout;
    }

    @Deprecated
    public void publishResetTimeoutEvent(WorkItem item) {
        String handler = null;
        Iterator<String> iter = item.getNode().getMetaDataMap().keySet().iterator();
        while (iter.hasNext()) {
            String key = iter.next();
            if (key.equals("timeoutHandler")) {
                handler = item.getNode().getMetaDataMap().get("timeoutHandler", String.class);
            }
        }
        if (handler != null) {
            TimeoutJob job = new TimeoutJob(item, handler);
            eventAdmin.postEvent(job.cancelEvent(true));
        }
    }

    public void publishDelegationEvent(Workflow instance, Authorizable participant, WorkItem item, String user) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKITEM_DELEGATION_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        try {
            properties.put(DELEGATEE, participant.getID());
        } catch (RepositoryException e) {
            throw new SlingException(e.getMessage(), e);
        }

        properties.put(WORK_ITEM, item);
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishUnclaimEvent(Workflow instance, Authorizable participant, WorkItem item, String user) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKITEM_UNCLAIM_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        try {
            properties.put(RETURNED_TO, participant.getID());
        } catch (RepositoryException e) {
            throw new SlingException(e.getMessage(), e);
        }

        properties.put(WORK_ITEM, item);
        addPayloadInfo(instance, properties);

        sendEvent(properties);
    }

    public void publishClaimEvent(Workflow instance, WorkItem item, String user) {
        final Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put(TIME_STAMP, new Date());
        properties.put(USER, user);
        properties.put(EVENT_TYPE, WORKITEM_CLAIM_EVENT);
        properties.put(WORKFLOW_NAME, instance.getWorkflowModel().getTitle());
        properties.put(WORKFLOW_VERSION, instance.getWorkflowModel().getVersion());
        properties.put(WORKFLOW_INSTANCE_ID, instance.getId());
        properties.put(WORK_ITEM, item);
        addPayloadInfo(instance, properties);
        sendEvent(properties);
    }

    public void publishResourceCollectionModificationEvent(String path) {
        Dictionary<String, Object> properties = new Hashtable<String, Object>();
        properties.put("Path", path);
        properties.put(EVENT_TYPE, RESOURCE_COLLECTION_MODIFIED);
        sendEvent(properties);
    }

    // ------------< helper >---------------------------------------------------
    private AbsoluteTimeoutHandler getHandler(String handler, WorkflowSession session) {
        ComponentContext context = (ComponentContext) WorkflowUtil.getServiceConfig(session, "componentContext");
        if (context != null && context.getBundleContext() != null) {
            try {
                ServiceReference refs[] = context.getBundleContext().getServiceReferences(
                        AbsoluteTimeoutHandler.class.getName(), null);
                ServiceReference ref = getServiceRef(refs, handler);
                if (ref != null) {
                    Object service = context.getBundleContext().getService(ref);
                    if (service instanceof AbsoluteTimeoutHandler) {
                        return (AbsoluteTimeoutHandler) service;
                    }
                }

                // attempt to load the handler through a proxy (into CQ)
                ServiceReference[] proxyRefs = context.getBundleContext().getServiceReferences(AbsoluteTimeoutHandlerProxy.class.getName(), null);
                if (proxyRefs != null) {
                    for( ServiceReference proxyRef : proxyRefs ) {
                        Object service = context.getBundleContext().getService(proxyRef);
                        if (service instanceof AbsoluteTimeoutHandlerProxy) {
                            AbsoluteTimeoutHandlerProxy absoluteTimeoutHandlerProxy = (AbsoluteTimeoutHandlerProxy) service;
                            AbsoluteTimeoutHandler handlerImpl = absoluteTimeoutHandlerProxy.findHandler(handler);
                            if (handlerImpl != null) {
                                return handlerImpl;
                            }
                        }
                    }
                }
            } catch (InvalidSyntaxException e) {
                // this is only thrown when the filter on getServiceReferences is invalid
                // -> since we don't provide a filter this should never happen.
                // that said: always log exceptions.
                log.debug("Failed to retrieve AbsoluteTimeoutHandler", e);
            }
        }
        return null;
    }

    private ServiceReference getServiceRef(ServiceReference[] refs, String handler) {
        for (ServiceReference ref : refs) {
            String componentName = (String) ref.getProperty("component.name");
            if (componentName.equals(handler)) {
                return ref;
            }
        }
        return null;
    }
}
