/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import com.adobe.xfa.ut.ExFull;


/**
 * EventPseudoModel describes the state of an event that is being fired.
 * Instances are available to scripts as the <code>$event</code> object. 
 * <p/>
 * Users of this class will not typically create instances of this class
 * directly, but will query the <code>AppModel</code> to find the existing
 * instance of the "$event" object.
 * @see AppModel#lookupPseudoModel(String)
 */
public class EventPseudoModel extends PseudoModel {
	
	/**
	 * Encapsulates the state of EventPseudoModel. 
	 * A caller will typically use instances of this class to save and restore
	 * the state of the EventPseudoModel when firing a nested event.
	 */
	public static class EventInfo {
		
		private String	msPrevText;
		private String	msPrevContentType;
		private String	msChange;
		private String	msNewText;
		private String	msNewContentType;
		private String	msFullText;
		private int		mnCommitKey;
		private boolean	mbKeyDown;
		private boolean	mbModifier;
		private int		meName;
		private int		mnSelEnd;
		private int		mnSelStart;
		private boolean	mbShift;
		private Obj		mTarget;
		private String	msSOAPFaultCode;
		private String	msSOAPFaultString;
		private boolean	mbReenter;
	
		public EventInfo() {
		}
		
		/**
		 * Returns the value before the field changes.
		 * @return the value before the field changes.
		 */
		public String getPrevText() { return msPrevText; }
		/**
		 * Sets the value before the field changes.
		 * @param sPrevText the value before the field changes.
		 */
		public void setPrevText(String sPrevText) { msPrevText = sPrevText; }
		
		/**
		 * Returns the contentType of prevText.
		 * <p/>
		 *  For example, if prevContentType="text/html", 
		 *  then prevText will contain an XHTML fragment.
		 * @return the contentType of prevText. 
		 */
		public String getPrevContentType() { return msPrevContentType; }
		/**
		 * Sets the contentType of prevText.
		 * @param sPrevContentType the contentType of prevText.
		 */
		public void setPrevContentType(String sPrevContentType) { msPrevContentType = sPrevContentType; }		

		/**
		 * Returns the text typed or pasted into the field.
		 * @return the text typed or pasted into the field.
		 */
		private String getChange() { return msChange; }
		/**
		 * Sets the text typed or pasted into the field.
		 * @param sChange the text typed or pasted into the field.
		 */
		public void setChange(String sChange) { msChange = sChange; }		

		/**
		 * Returns the value after the field change.
		 * @return the value after the field change.
		 */
		public String getNewText() { return msNewText; }
		/**
		 * Sets the value after the field change.
		 * @param sNewText the value after the field change.
		 */
		public void setNewText(String sNewText) { msNewText = sNewText; }

		/**
		 * Returns the contentType after the field change.
		 * @return the contentType after the field change.
		 */
		private String getNewContentType() { return msNewContentType; }
		/**
		 * Sets the contentType after the field change.
		 * @param sNewContentType the contentType after the field change.
		 */
		private void setNewContentType(String sNewContentType) { msNewContentType = sNewContentType; }

		/**
		 * Returns the full (untruncated) value of the field.
		 * <p/> 
		 * If the user pastes into a field, the field may truncate the pasted text.
		 * Content type is determined by $event.newContentType.
		 * @return the full (untruncated) value of the field.
		 */
		public String getFullText() { return msFullText; }
		/**
		 * Sets the full (untruncated) value of the field.
		 * <p/> 
		 * If the user pastes into a field, the field may truncate the pasted text.
		 * Content type is determined by $event.newContentType.
		 * @param sFullText the full (untruncated) value of the field.
		 */
		public void setFullText(String sFullText) { msFullText = sFullText; }

		/**
		 * Returns how a form field was committed.
		 * @return how a form field was committed.
		 */
		public int getCommitKey() { return mnCommitKey; }
		/**
		 * Sets how a form field value was committed.
		 * @param nCommitKey how a form field was committed.
		 */
		public void setCommitKey(int nCommitKey) { mnCommitKey = nCommitKey; }

		/**
		 * Returns whether the arrow key was used to make the selection.
		 * <p/>
		 * Available only for listbox and combobox.
		 * @return <code>true</code> if the arrow key was used to make the selection.
		 */
		public boolean getKeyDown() { return mbKeyDown; }
		/**
		 * Sets whether the arrow key was used to make the selection.
		 * <p/>
		 * Available only for listbox and combobox.
		 * @param bKeyDown <code>true</code> if the arrow key was used to make the selection.
		 */
		public void setKeyDown(boolean bKeyDown) { mbKeyDown = bKeyDown; }

		/**
		 * Returns whether the modifier key is down during the event.
		 * <p/>
		 * For the Microsoft Windows platform, the modifier key is CTRL.
		 * @return <code>true</code> if the modifier key is down during the event.
		 */
		public boolean getModifier() { return mbModifier; }
		/**
		 * Sets whether the modifier key is down during the event.
		 * <p/>
		 * For the Microsoft Windows platform, the modifier key is CTRL. 
		 * @param bModifier <code>true</code> if the modifier key is down during the event.  
		 */
		public void setModifier(boolean bModifier) { mbModifier = bModifier; }
		
		/**
		 * Returns the code that identifies the current event.
		 * <p/>
		 * The code must be one of the ScriptHandler.ACTIVITY_* values.
		 * @return the code that identifies the current event.
		 * @see ScriptHandler
		 */
		public int getName() { return meName; }
		/**
		 * Sets the code that identifies the current event.
		 * <p/>
		 * The code must be one of the ScriptHandler.ACTIVITY_* values.
		 * @param eName a code that defines the current event.
		 * @see ScriptHandler
		 */
		public void setName(int eName) { meName = eName; }

		/**
		 * Returns the end position of the current text selection during a change event.
		 * @return the end position of the current text selection.
		 */
		public int getSelEnd() { return mnSelEnd; }
		/**
		 * Sets the end position of the current text selection during a change event.
		 * @param nSelEnd the end position of the current text selection.
		 */
		public void setSelEnd(int nSelEnd) { mnSelEnd = nSelEnd; }

		/**
		 * Returns the start position of the current text selection during a change event.
		 * @return the start position of the current text selection.
		 */
		public int getSelStart() { return mnSelStart; }
		/**
		 * Sets the start position of the current text selection during a change event.
		 * @param nSelStart the start position of the current text selection during a change event.
		 */
		public void setSelStart(int nSelStart) { mnSelStart = nSelStart; }

		/**
		 * Returns whether the shift key is down during the event.
		 * @return <code>true</code> if the shift key is down during the event.
		 */
		public boolean getShift() { return mbShift; }
		/**
		 * Sets whether the shift key is down during the event.
		 * @param bShift <code>true</code> if the shift key is down during the event.
		 */
		public void setShift(boolean bShift) { mbShift = bShift; }

		/**
		 * Returns the target object that triggered the event.
		 * @return the target object that triggered the event.
		 */
		public Obj getTarget() { return mTarget; }
		/**
		 * Sets the target object that triggered the event.
		 * @param target the target object that triggered the event.
		 */
		public void setTarget(Obj target) { mTarget = target; }

		/**
		 * Returns the SOAP faultcode returned from executing a Web Service connection.
		 * @return the SOAP faultcode returned from executing a Weg Service connection.
		 */
		public String getSOAPFaultCode() { return msSOAPFaultCode; }
		/**
		 * Sets the SOAP faultstring returned from executing a Web Service connection.
		 * @param sSOAPFaultCode the SOAP faultstring returned from executing a Web Service connection.
		 */
		public void setSOAPFaultCode(String sSOAPFaultCode) { msSOAPFaultCode = sSOAPFaultCode; }

		/**
		 * Returns the SOAP faultstring returned from executing a Web Service connection.
		 * @return the SOAP faultstring returned from executing a Web Service connection.
		 */
		public String getSOAPFaultString() { return msSOAPFaultString; }
		/**
		 * Sets the SOAP faultstring returned from executing a Web Service connection.
		 * @param sSOAPFaultString the SOAP faultstring returned from executing a Web Service connection.
		 */
		public void setSOAPFaultString(String sSOAPFaultString) { msSOAPFaultString = sSOAPFaultString; }

		/**
		 * Returns whether this event is happening immediately after the user committed this field.
		 * @return <code>true</code> if this enter event is happening immediately after
		 * the user committed this field.
		 */
		public boolean getReenter() { return mbReenter; }	
		/**
		 * Sets whether this event is happening immediately after the user committed this field.
		 * Available only during the 'enter' event. 
		 * @param bReenter <code>true</code> if this enter event is happening immediately 
		 * after the user committed this field.
		 */
		public void setReenter(boolean bReenter) { mbReenter = bReenter; }
	}
	
	private final AppModel	mAppModel;
	private EventInfo		mEventInfo;
	
	private boolean	mbCancelAction;		// Set by a pre-* event to cancel a forthcoming action.
	private String	msCancelActionEvent;// Set by a pre-* event to know which post-* event to use to reset cancelAction.
	private	String	msCanceledAction;	// Record the most recently canceled action.
	
	/**
	 * @exclude from published api.
	 */
	public EventPseudoModel(AppModel appModel) {
	    mAppModel = appModel;
		reset();
	}
	
	/**
	 * Returns the value before the field changes.
	 * @return the value before the field changes.
	 */	
	public String	getPrevText() { return mEventInfo.getPrevText(); }
	/**
	 * Sets the value before the field changes.
	 * @param sPrevText the value before the field changes.
	 */	
	public void		setPrevText(String sPrevText) { mEventInfo.setPrevText(sPrevText); }

	/**
	 * Returns the contentType of prevText.
	 * <p/>
	 * For example, if prevContentType="text/html", 
	 * then prevText will contain an XHTML fragment.
	 * @return the contentType of prevText. 
	 */
	public String	getPrevContentType() { return mEventInfo.getPrevContentType(); }
	/**
	 * Sets the contentType of prevText.
	 * @param sPrevContentType the contentType of prevText.
	 */
	public void		setPrevContentType(String sPrevContentType) { mEventInfo.setPrevContentType(sPrevContentType); }

	/**
	 * Returns the text typed or pasted into the field.
	 * @return the text typed or pasted into the field.
	 */
	public String	getChange() { return mEventInfo.getChange(); }
	/**
	 * Sets the text typed or pasted into the field.
	 * @param sChange the text typed or pasted into the field.
	 */
	public void		setChange(String sChange) { mEventInfo.setChange(sChange); }

	/**
	 * Returns the value after the field change.
	 * @return the value after the field change.
	 */
	public String	getNewText() { return mEventInfo.getNewText(); }
	/**
	 * Sets the value after the field change.
	 * @param sNewText the value after the field change.
	 */	
	public void		setNewText(String sNewText) { mEventInfo.setNewText(sNewText); }

	/**
	 * Returns the contentType after the field change.
	 * @return the contentType after the field change.
	 */
	public String	getNewContentType() { return mEventInfo.getNewContentType(); }
	/**
	 * Sets the contentType after the field change.
	 * @param sNewContentType the contentType after the field change.
	 */	
	public void		setNewContentType(String sNewContentType) { mEventInfo.setNewContentType(sNewContentType); }

	/**
	 * Returns the full (untruncated) value of the field.
	 * <p/> 
	 * If the user pastes into a field, the field may truncate the pasted text.
	 * Content type is determined by $event.newContentType.
	 * @return the full (untruncated) value of the field.
	 */
	public String	getFullText() { return mEventInfo.getFullText(); }
	/**
	 * Sets the full (untruncated) value of the field.
	 * <p/> 
	 * If the user pastes into a field, the field may truncate the pasted text.
	 * Content type is determined by $event.newContentType.
	 * @param sFullText the full (untruncated) value of the field.
	 */
	public void		setFullText(String sFullText) { mEventInfo.setFullText(sFullText); }
	
	/**
	 * Returns how a form field was committed.
	 * @return how a form field was committed.
	 */
	public int		getCommitKey() { return mEventInfo.getCommitKey(); }
	/**
	 * Sets how a form field value was committed.
	 * @param nCommitKey how a form field was committed.
	 */
	public void		setCommitKey(int nCommitKey) { mEventInfo.setCommitKey(nCommitKey); }

	/**
	 * Returns whether the arrow key was used to make the selection.
	 * <p/>
	 * Available only for listbox and combobox.
	 * @return <code>true</code> if the arrow key was used to make the selection.
	 */
	public boolean	getKeyDown() { return mEventInfo.getKeyDown(); }
	/**
	 * Sets whether the arrow key was used to make the selection.
	 * <p/>
	 * Available only for listbox and combobox.
	 * @param bKeyDown <code>true</code> if the arrow key was used to make the selection.
	 */
	public void		setKeyDown(boolean bKeyDown) { mEventInfo.setKeyDown(bKeyDown); }

	/**
	 * Returns whether the modifier key is down during the event.
	 * <p/>
	 * For the Microsoft Windows platform, the modifier key is CTRL.
	 * @return <code>true</code> if the modifier key is down during the event.
	 */
	public boolean	getModifier() { return mEventInfo.getModifier(); }
	/**
	 * Sets whether the modifier key is down during the event.
	 * <p/>
	 * For the Microsoft Windows platform, the modifier key is CTRL. 
	 * @param bModifier <code>true</code> if the modifier key is down during the event.  
	 */
	public void		setModifier(boolean bModifier) { mEventInfo.setModifier(bModifier); }

	/**
	 * Returns the code that identifies the current event.
	 * <p/>
	 * The code must be one of the ScriptHandler.ACTIVITY_* values.
	 * @return the code that identifies the current event.
	 * @see ScriptHandler
	 */
	public int		getName() { return mEventInfo.getName(); }
	/**
	 * Sets the code that identifies the current event.
	 * <p/>
	 * The code must be one of the ScriptHandler.ACTIVITY_* values.
	 * @param eName a code that defines the current event.
	 * @see ScriptHandler
	 */
	public void		setName(int	eName) { mEventInfo.setName(eName); }
	
	/**
	 * Returns the name of the event as a String.
	 * @return the name of the event as a String.
	 */
	public String	getNameAsString() { return ScriptHandler.executeReasonToString(mEventInfo.getName()); }
	/**
	 * Sets the name of the event using a String.
	 * @param sName the name of the event.
	 */
	public void		setName(String sName) { mEventInfo.setName(ScriptHandler.stringToExecuteReason(sName)); }

	/**
	 * Returns the end position of the current text selection during a change event.
	 * @return the end position of the current text selection.
	 */
	public int		getSelEnd() { return mEventInfo.getSelEnd(); }
	/**
	 * Sets the end position of the current text selection during a change event.
	 * @param nSelEnd the end position of the current text selection.
	 */
	public void		setSelEnd(int nSelEnd) { mEventInfo.setSelEnd(nSelEnd); }

	/**
	 * Returns the start position of the current text selection during a change event.
	 * @return the start position of the current text selection.
	 */
	public int		getSelStart() { return mEventInfo.getSelStart(); }
	/**
	 * Sets the start position of the current text selection during a change event.
	 * @param nSelStart the start position of the current text selection during a change event.
	 */
	public void		setSelStart(int nSelStart) { mEventInfo.setSelStart(nSelStart); }

	/**
	 * Returns whether the shift key is down during the event.
	 * @return <code>true</code> if the shift key is down during the event.
	 */
	public boolean	getShift() { return mEventInfo.getShift(); }
	/**
	 * Sets whether the shift key is down during the event.
	 * @param bShift <code>true</code> if the shift key is down during the event.
	 */
	public void		setShift(boolean bShift) { mEventInfo.setShift(bShift); }

	/**
	 * Returns the target object that triggered the event.
	 * @return the target object that triggered the event.
	 */
	public Obj		getTarget() { return mEventInfo.getTarget(); }
	/**
	 * Sets the target object that triggered the event.
	 * @param target the target object that triggered the event.
	 */
	public void		setTarget(Obj target) { mEventInfo.setTarget(target); }

	/**
	 * Returns the SOAP faultcode returned from executing a Web Service connection.
	 * @return the SOAP faultcode returned from executing a Weg Service connection.
	 */
	public String	getSOAPFaultCode() { return mEventInfo.getSOAPFaultCode(); }
	/**
	 * Sets the SOAP faultcode returned from executing a Web Service connection.
	 * @param sSOAPFaultCode the SOAP faultCode. 
	 */
	public void		setSOAPFaultCode(String sSOAPFaultCode) { mEventInfo.setSOAPFaultCode(sSOAPFaultCode); }

	/**
	 * Returns the SOAP faultstring returned from executing a Web Service connection.
	 * @return the SOAP faultstring returned from executing a Web Service connection.
	 */
	public String	getSOAPFaultString() { return mEventInfo.getSOAPFaultString(); }
	/**
	 * Sets the SOAP faultstring returned from executing a Web Service connection.
	 * @param sSOAPFaultString the SOAP faultstring returned from executing a Web Service connection.
	 */
	public void		setSOAPFaultString(String sSOAPFaultString) { mEventInfo.setSOAPFaultString(sSOAPFaultString); }

	/**
	 * Returns whether this event is happening immediately after the user committed this field.
	 * @return <code>true</code> if this enter event is happening immediately after
	 * the user committed this field.
	 */
	public boolean	getReenter() { return mEventInfo.getReenter(); }
	/**
	 * Sets whether this event is happening immediately after the user committed this field.
	 * Available only during the 'enter' event. 
	 * @param bReenter <code>true</code> if this enter event is happening immediately 
	 * after the user committed this field.
	 */
	public void		setReenter(boolean bReenter) { mEventInfo.setReenter(bReenter); }
	
	/**
	 * Determines whether the current action event has been canceled.
	 * @return <code>true</code> if the current action event has been canceled.
	 */
	public boolean	getCancelAction() { return mbCancelAction; }
	
	/**
	 * Returns the name of the canceled action.
	 * @return the name of the canceled action.
	 */
	public String	getCanceledAction() { return msCanceledAction; }
	
	/**
	 * @exclude from published api.
	 */
	public void setCancelAction(boolean bCancelAction, String sSource /* = null */) { 
		// determine the source of setting cancelAction
		String sCurrentEvent = sSource;
		if (sCurrentEvent == null) {
			EventManager eventManager = mAppModel.getEventManager();
			if (eventManager != null)
				sCurrentEvent = eventManager.getCurrentEvent();
		}

		// record the event where cancelAction goes from false to true
		if (! mbCancelAction && bCancelAction) {
			
			// cancelAction can only be set in predetermined pre-* events (preExecute, preOpen, prePrint, preSign, preSubmit)
			if (isPreActionEvent(sCurrentEvent)) {
				msCancelActionEvent = sCurrentEvent;
				mbCancelAction = true;
			}
		}
		// ensure proper event is used to reset cancelAction from true to false
		else if (mbCancelAction && ! bCancelAction) {
			
			// cancelAction can only be reset in predetermined post-* events (postExecute, postOpen, postPrint, postSign, postSubmit)
			// cancelAction can also be reset in the same event that set it (i.e. if it was set in preSubmit, it could be reset in preSubmit)
			if (msCancelActionEvent.equals(STRS.PREEXECUTE)) {
				if (sCurrentEvent.equals(STRS.POSTEXECUTE) || sCurrentEvent.equals(STRS.PREEXECUTE)) {
					msCancelActionEvent = null;
					mbCancelAction = false;
				}
			}
			else if (msCancelActionEvent.equals(STRS.PREOPEN)) {
				if (sCurrentEvent.equals(STRS.POSTOPEN) || sCurrentEvent.equals(STRS.PREOPEN)) {
					msCancelActionEvent = null;
					mbCancelAction = false;
				}
			}
			else if (msCancelActionEvent.equals(STRS.PREPRINT)) {
				if (sCurrentEvent.equals(STRS.POSTPRINT) || sCurrentEvent.equals(STRS.PREPRINT)) {
					msCancelActionEvent = null;
					mbCancelAction= false;
				}
			}
			else if (msCancelActionEvent.equals(STRS.PRESIGN)) {
				if (sCurrentEvent.equals(STRS.POSTSIGN) || sCurrentEvent.equals(STRS.PRESIGN)) {
					msCancelActionEvent = null;
					mbCancelAction = false;
				}
			}
			else if (msCancelActionEvent.equals(STRS.PRESUBMIT)) {
				if (sCurrentEvent.equals(STRS.POSTSUBMIT) || sCurrentEvent.equals(STRS.PRESUBMIT)) {
					msCancelActionEvent = null;
					mbCancelAction = false;
				}
			}
		} 
	}

	/**
	 * Sets the event state to the state contained within an EventInfo object.
	 * @param eventInfo the new state
	 */
	public void setEventInfo(EventInfo eventInfo) {
		mEventInfo = eventInfo;
	}	
	
	/**
	 * Returns the current event state object. 
	 * @return an EventInfo object that represents the state of the current event.
	 */
	public EventInfo getEventInfo() {
		return mEventInfo;
	}

	/**
	 * Resets the event state to all-empty values.
	 */
	public void	reset() {
		mEventInfo = new EventInfo();
	}
	
	/**
	 * Fires the event defined by the current event state.
	 * The caller must ensure that the event's name, target and
	 * any other properties defined for that event are set before
	 * calling this method.
	 */
	public void emit() {
		EventManager eventManager = mAppModel.getEventManager();

		String sActivity = getNameAsString();
		int nEventID = eventManager.getEventID(sActivity);

		Obj obj = getTarget();
		eventManager.eventOccurred(nEventID, obj);
	}
	
	/**
	 * Determines whether the specified action has been canceled.
	 * <p/>
	 * If the action was canceled, this method as the side effect of
	 * setting the name of the canceled action.
	 * @param sAction the name of the action. Must be one of:
	 * execute, open, print, sign or submit.
	 * @see #getCanceledAction()
	 */
	public boolean cancelAction(String sAction) {
		assert isAction(sAction);
		
		boolean	bResult = cancelOperation(sAction);
		if (bResult) {
			if (isAction(sAction))
				msCanceledAction = sAction;
		}
		return bResult;
	}
	
	/**
	 * Determines whether the specified event has been canceled.
	 * @param sEvent the name of the event
	 * @return <code>true</code> if the event should be executed.
	 * @exclude from published api.
	 */
	public boolean cancelEvent(String sEvent) {
		assert !isAction(sEvent);
		
		return cancelOperation(sEvent);
	}
	
	/**
	 * @exclude from published api.
	 */
	public boolean isAction(String sOperation) {
		if (null == sOperation) return false;
		return sOperation.equals(STRS.EXECUTE) || sOperation.equals(STRS.OPEN) || sOperation.equals(STRS.PRINT) ||
			   sOperation.equals(STRS.SIGN)    || sOperation.equals(STRS.SUBMIT);
	}
	
	/**
	 * @exclude from published api.
	 */
	boolean isPreActionEvent(String sEvent) {
		if (null == sEvent) return false;		
		return sEvent.equals(STRS.PREEXECUTE) || sEvent.equals(STRS.PREOPEN) || sEvent.equals(STRS.PREPRINT) ||
			   sEvent.equals(STRS.PRESIGN)	  || sEvent.equals(STRS.PRESUBMIT);
	}
	
	/**
	 * @exclude from published api.
	 */
	boolean isPostActionEvent(String sEvent) {
		if (null == sEvent) return false;		
		return sEvent.equals(STRS.POSTEXECUTE) || sEvent.equals(STRS.POSTOPEN) || sEvent.equals(STRS.POSTPRINT) ||
			   sEvent.equals(STRS.POSTSIGN)    || sEvent.equals(STRS.POSTSUBMIT);
	}
	
	/**
	 * @exclude from published api.
	 */
	protected boolean cancelOperation(String sOperation) {
		// if cancelAction is not set, operation definitely isn't canceled
		if (! getCancelAction())
			return false;

		// if cancelAction is set, cancel all operations except the one that can reset cancelAction
		// the operation that can reset cancelAction is the post-* event that matches the pre-* event
		// that set cancelAction
		if (msCancelActionEvent.equals(STRS.PREEXECUTE)) {
			if (sOperation.equals(STRS.POSTEXECUTE))
				return false;
		}
		else if (msCancelActionEvent.equals(STRS.PREOPEN)) {
			if (sOperation.equals(STRS.POSTOPEN))
				return false;
		}
		else if (msCancelActionEvent.equals(STRS.PREPRINT)) {
			if (sOperation.equals(STRS.POSTPRINT))
				return false;
		}
		else if (msCancelActionEvent.equals(STRS.PRESIGN)) {
			if (sOperation.equals(STRS.POSTSIGN))
				return false;
		}
		else if (msCancelActionEvent.equals(STRS.PRESUBMIT)) {
			if (sOperation.equals(STRS.POSTSUBMIT))
				return true;
		}

		return true;
	}
	
	/**
	 * @exclude from published api.
	 */
	public String getClassAtom() {
		return "eventPseudoModel";
	}

	/**
	 * @exclude from published api.
	 */
	public String getClassName() {
		return "eventPseudoModel";
	}

	/**
	 * @exclude from published api.
	 */
	public ScriptTable getScriptTable() {
		return EventPseudoModelScript.getScriptTable();
	}
	
	/**
	 * @see Obj#sendMessenge(ExFull, int)
	 * @exclude from published api.
	 */
	public void sendMessenge(ExFull error, int eSeverity) {
		if (mAppModel != null)
			mAppModel.addErrorList(error, eSeverity, null);
	}
	
	/**
	 * @see Obj#validateUsage(int, int, boolean)
	 * @exclude from published api.
	 */
	public boolean validateUsage(int nVersion, int nAvailability, boolean bUpdateVersion) {	
		if (mAppModel != null)
			return mAppModel.validateUsage(nVersion, nAvailability, bUpdateVersion);

		return super.validateUsage(nVersion, nAvailability, bUpdateVersion);
	}
	
	/**
	 * @see Obj#validateUsageFailedIsFatal(int, int)
	 * @exclude from published api.
	 */
	public boolean validateUsageFailedIsFatal(int nVersion, int nAvailability) {	
		if (mAppModel != null)
			return mAppModel.validateUsageFailedIsFatal(nVersion, nAvailability);

		return super.validateUsageFailedIsFatal(nVersion, nAvailability);
	}
}