/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

/**
 * A class to represents plain XML attributes, i.e., attributes
 * that do not fit into an XFA schema.
 * <p>
 * Generic attributes are more expensive than XFA attributes, because
 * they must include all possible properties.
 */

public final class GenericAttribute extends Attribute {

	private final String mNS;
	
	/**
	 * Instantiates an attribute with the given attribute parameters.
	 * @param NS the namespace for this attribute.
	 * @param localName the local name for this attribute.
	 * @param qName the qualified name for this attribute .
	 * @param value the string to use to create the new attribute.  
	 */
	public GenericAttribute(String NS,
							String localName,
							String qName,
							String value) {
		this(NS, localName, qName, value, true);
	}

	/**
	 * Instantiates an attribute with the given value.
	 * @param value the attribute value.
	 *
	 * @exclude from published api.
	 */
	protected GenericAttribute(String name, String value) {
		super(name, value);
		mNS = null;
	}
	
	/**
	 * Instantiates an attribute with the given attribute parameters.
	 * @param NS the namespace for this attribute.
	 * @param localName the local name for this attribute.
	 * @param qName the qualified name for this attribute .
	 * @param value the string to use to create the new attribute.
	 * @param internSymbols indicates whether the symbols in other parameters need to be interned.  
	 */
	GenericAttribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		super(NS, localName, qName, value, internSymbols);
		
		if (internSymbols && NS != null) {
			NS = NS.intern();
		}
		
		mNS = NS;
	}

	/**
	 * Determines if this attribute is a schema attribute.
	 * @return returns false.
	 *
	 * @exclude from published api.
	 */
	public boolean isSchemaAttr() {
		return false;
	}

	/**
	 * Gets this attribute's namespace.
	 * @return the namespace.
	 */
	public String getNS() {
		return mNS;
	}

	/**
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 *
	 * @exclude from published api.
	 */
	public Attribute newAttribute(String NS,
									String localName,
										String qName,
											String value) {
		return newAttribute(NS, localName, qName, value, true); 
	}
	
	/**
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean)
	 *
	 * @exclude from published api.
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		return new GenericAttribute(NS, localName, qName, value, internSymbols);
	}

	/**
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String)
	 *
	 * @exclude from published api.
	 */
	public Attribute newAttribute(String value) {
		return newAttribute(getNS(), getLocalName(), getQName(), value, false);
	}

}
