/* ****************************************************************************
 *
 *	File: PDFFontSet
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.fontset.impl;

import java.nio.charset.UnsupportedCharsetException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;
import java.util.Set;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.FontEngineException;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.fontmanagement.FontLoader;
import com.adobe.fontengine.fontmanagement.FontResolutionPriority;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontResolver;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontSearchAttributes;
import com.adobe.fontengine.fontmanagement.postscript.PSNameResolver;
import com.adobe.fontengine.fontmanagement.postscript.PostscriptFontDescription;
import com.adobe.fontengine.inlineformatting.FallbackFontSet;
import com.adobe.fontengine.inlineformatting.PDF16RichTextFormatter;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20FontSet;
import com.adobe.fontengine.inlineformatting.css20.CSS20GenericFontFamily;
import com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFFontException;
import com.adobe.internal.pdftoolkit.core.fontset.PDFFontSet;
import com.adobe.internal.pdftoolkit.core.types.ASName;


/**
 *Provides access to an aggregated font set, including CSS20FontSet, PSFontSet, and FallbackFontSet. 
 * @author mplunket
 *
 */
public class PDFFontSetImpl implements PDFFontSet
{
	/**
	 * Mangle the font names in the same way that Acrobat does.
	 *
	 */
	private static final class AcrobatPlatformNameMangler implements FamilyNameNormalizer
	{
		/**
		 * 
		 */
		private static final long serialVersionUID = 1L;

		/**
		 * 
		 */
		private AcrobatPlatformNameMangler() 
		{
			super();
		}

		public String normalize(String familyName) 
		{
			// remove space and hyphen
			StringBuilder strBuilder = new StringBuilder();
			char[] chars = familyName.toCharArray();
			for(int i=0;i<chars.length;i++){
				if(chars[i] == ' ' || chars[i] == '-')
					continue;
				strBuilder.append(chars[i]);
			}
			return strBuilder.toString();
		}
	}

	/* Serialization signature is explicitly set and should be 
	 * incremented on each release to prevent compatibility.
	 */
	private static final long serialVersionUID = 2L;

	private final CSS20FontSet cssFontSet;	
	private final PSNameResolver psFontSet;
	private final FallbackFontSet fallbackFontSet;
	private final PlatformFontResolver platformFontSet;
	
	private boolean ignoreFontLoadingErrors;

	public PDFFontSetImpl() 
	throws PDFFontException 
	{
		this(FontResolutionPriority.INTELLIGENT_LAST);
	}

	/**
	 * Default constructor that provides no fonts.
	 */
	public PDFFontSetImpl(FontResolutionPriority priority) 
	throws PDFFontException
	{
		this.fallbackFontSet = new FallbackFontSet();

		this.cssFontSet = PDF16RichTextFormatter.getFontSetInstance();
		this.cssFontSet.setResolutionPriority(priority);
		this.cssFontSet.setFallbackFonts(this.fallbackFontSet);

		this.psFontSet = FontLoader.getPSNameResolverInstance();
		this.psFontSet.setResolutionPriority(priority);

		this.platformFontSet = FontLoader.getPlatformFontResolverInstance(new AcrobatPlatformNameMangler());
		this.platformFontSet.setResolutionPriority(priority);
	}

	public PDFFontSetImpl(PDFFontSetImpl original)
	throws PDFFontException
	{
		if (original == null)
		{
			// Unfortunately this code from the default constructor MUST be repeated.
			// The call to another constructor must be the first statement in a constructor
			// and we can't do that if we want to have the check for null above.
			// Also, we can't split the common code out as the data members are final and so
			// can only be set inside a constructor.  ARGH!!!
			this.fallbackFontSet = new FallbackFontSet();

			this.cssFontSet = PDF16RichTextFormatter.getFontSetInstance();
			this.cssFontSet.setResolutionPriority(FontResolutionPriority.INTELLIGENT);
			this.cssFontSet.setFallbackFonts(this.fallbackFontSet);

			this.psFontSet = FontLoader.getPSNameResolverInstance();
			this.psFontSet.setResolutionPriority(FontResolutionPriority.INTELLIGENT);	

			this.platformFontSet = FontLoader.getPlatformFontResolverInstance(new AcrobatPlatformNameMangler());
			this.platformFontSet.setResolutionPriority(FontResolutionPriority.INTELLIGENT);
		} else {
			this.cssFontSet = PDF16RichTextFormatter.getFontSetInstance(original.cssFontSet);
			this.psFontSet = FontLoader.getPSNameResolverInstance(original.psFontSet);
			this.fallbackFontSet = new FallbackFontSet(original.fallbackFontSet);
			this.platformFontSet = FontLoader.getPlatformFontResolverInstance(original.platformFontSet);

			// Must set the cssFontSet to use the new cloned fallback font set
			// because AFE doesn't clone the fallback fontset
			this.cssFontSet.setFallbackFonts(this.fallbackFontSet);
			this.ignoreFontLoadingErrors = original.ignoreFontLoadingErrors;
			
			if(! original.resolvedNamesMap.isEmpty())
			{
				this.resolvedNamesMap.putAll(original.resolvedNamesMap);
			}
		}
	}

	public boolean hasRootFallback() throws PDFFontException
	{
		// TODO ULocale.ROOT should be replaced with Locale.ROOT once AFE's
		// dependency on AGL is removed.
		Iterator iter = getFallbackFontSet().getFallbackFonts(ULocale.ROOT);
		return iter.hasNext();
	}

	/**
	 * This maps all the possible names of the font with it's AFE font object.
	 */
	private HashMap<String, Font> resolvedNamesMap = new HashMap<String, Font>();
	public void addFont(Font font, Platform platForm, ULocale locale) 
	throws PDFFontException
	{
		platForm = platForm == null ? Platform.WINDOWS : platForm;
		locale = locale == null ? ULocale.getDefault() : locale;
		try
		{
			this.cssFontSet.addFont(font);
			this.psFontSet.addFont(font);
			try {
				this.platformFontSet.addFont(font);
			}catch (UnsupportedFontException e)
			{
				// not all font types are supported for platform font name
			}
			// Add all the possible names we could have for this font.
			Set<String> resolvedNames = FontNameResolver.doNameMangling(font, platForm, locale);
			Iterator<String> itr = resolvedNames.iterator();
			while(itr.hasNext()){
				resolvedNamesMap.put(itr.next(), font);
			}
		}
		catch (UnsupportedCharsetException e) {
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("Charset for font is not supported.", e);
			}
		} 
		catch (UnsupportedFontException e) {
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("Font is not supported.", e);
			}
		} catch (InvalidFontException e) {
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("The Font is invalid.", e);
			}
		} catch (FontLoadingException e) {
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("Problems in loading the font.", e);
			}
		}	
	}

	public void addFont(Font font, PostscriptFontDescription[] psDescriptions, CSS20FontDescription[] cssDescriptions) 
	throws PDFFontException 
	{
		if ((psDescriptions == null || psDescriptions.length == 0)
				|| (cssDescriptions == null || cssDescriptions.length == 0))
		{
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("Invalid font descriptions.");		
			}
		}

		try
		{
			if (psDescriptions != null)
			{
				for (int i = 0; i < psDescriptions.length; i++)
				{
					this.psFontSet.addFont(psDescriptions[i], font);
				}
			}

			if (cssDescriptions != null)
			{
				for (int i = 0; i < cssDescriptions.length; i++)
				{
					this.cssFontSet.addFont(cssDescriptions[i], font);
				}
			}
		} 
		catch (UnsupportedCharsetException e) {
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("Font could not be added to PDFFontSet.", e);
			}
		} 
		catch (FontEngineException e)	{
			if(!ignoreFontLoadingErrors)
			{
				throw new PDFFontException("Font could not be added to PDFFontSet.", e);
			}
		}		
	}

	public void addFont(Font[] fonts, Platform platForm, ULocale locale) 
	throws PDFFontException
	{
		for (int i = 0; i < fonts.length; i++)
		{
			this.addFont(fonts[i], platForm, locale);					
		}
	}

	public void addFallbackFont(Locale locale, Font font)
	throws PDFFontException
	{
		// TODO ULocale.forLocale(locale) should be replaced with locale
		// once AFE's dependency on AGL is removed.
		this.fallbackFontSet.addFallbackFont(ULocale.forLocale(locale), font);
	}

	public void addFallbackFont(Locale locale, Font[] fonts)
	throws PDFFontException
	{
		for (int i = 0; i < fonts.length; i++)
		{
			this.addFallbackFont(locale, fonts[i]);
		}		
	}

	public void setGenericFontFamilyName(CSS20GenericFontFamily family, String[] replacements)
	throws PDFFontException
	{
		this.cssFontSet.setGenericFont(family, replacements);
	}

	/**
	 * Get the CSS20 fonts.
	 * @return  an object that provides access to a font set. 
	 * @throws PDFFontException 
	 */
	public CSS20FontSet getCSS20FontSet() 
	throws PDFFontException 
	{
		return this.cssFontSet;
	}

	/**
	 * Get the PS fonts.
	 * @return  an object that provides access to a font set. 
	 * @throws PDFFontException 
	 */
	public PSNameResolver getPSFontSet() 
	throws PDFFontException 
	{
		return this.psFontSet;
	}	

	/**
	 * Get the fallback fonts.
	 * @return  an object that provides access to a font set. 
	 * @throws PDFFontException 
	 */
	public FallbackFontSet getFallbackFontSet() 
	throws PDFFontException 
	{
		return this.fallbackFontSet;
	}

	/**
	 * Get the PS fonts.
	 * @param substitute If true and the exact fontDescription is not found, a fallback font can be returned.
	 * @return  a font matching the description provided or a fallback appropriate for the locale. 
	 * @throws PDFFontException 
	 */
	public Font getPSFont(PostscriptFontDescription fontDescription, Locale locale, boolean substitute) 
	throws PDFFontException 
	{
		Font font = this.psFontSet.findFont(fontDescription);
		String psName = fontDescription.getPSName();
		if (font == null)
		{
			PlatformFontSearchAttributes searchAttributes =
				new PlatformFontSearchAttributes(psName);
			font = this.platformFontSet.findFont(searchAttributes);
		}
		// font is still null then let's try to map this with available resolved names.
		if(font == null)
			font = this.resolvedNamesMap.get(psName);
		if (substitute && font == null)
		{
			
			font = getFallbackFont(locale);
		}
		
		if(font == null)
		{
			String[] standardFontNames = CoreFontUtils.getStandardFontNames(ASName.create(psName));
			
			if(standardFontNames != null)
			{
				//Checking for each synonym
				for(int i=0;i<standardFontNames.length && font == null;i++)
				{
					font = this.getPSFontSet().findFont(new PostscriptFontDescription(standardFontNames[i]));
					if (font == null)
					{
						PlatformFontSearchAttributes searchAttributes =
							new PlatformFontSearchAttributes(standardFontNames[i]);
						font = this.platformFontSet.findFont(searchAttributes);
					}
				}
			}
		}
		
		
		//System.out.println("\nSearched for = " + fontDescription);
		//System.out.println("Found = " + font);
		return font;
	}

	/**
	 * Gets the preferred fallback font for locale supplied
	 * @param locale locale
	 * @return fallback font
	 */
	public Font getFallbackFont(Locale locale) {
		Iterator iter = this.fallbackFontSet.getFallbackFonts(ULocale.forLocale(locale));
		while (iter.hasNext())
		{
			Font font = (Font) iter.next();
			if (font != null)
			{
				return font;
			}
		}
		return null;
	}	

	public Font getPSFont(String fontName, Locale locale,	boolean substitute) 
	throws PDFFontException 
	{
		return this.getPSFont(new PostscriptFontDescription(fontName), locale, substitute);
	}

	public void setResolutionPriority(FontResolutionPriority priority)
	{
		this.cssFontSet.setResolutionPriority(priority);
		this.psFontSet.setResolutionPriority(priority);
		this.platformFontSet.setResolutionPriority(priority);
	}

	/**
	 * A debugging string containing information about the contents
	 * of the <code>PDFFontSetImpl</code> instance.
	 */
	@Override
	public String toString()
	{
		StringBuilder message = new StringBuilder(140);

		message.append("==== PDFFontSet ====\n\n=== CSS Font Set ===\n");
		message.append(this.cssFontSet.toString());

		message.append("\n\n=== PS Font Set ===\n");
		message.append(this.psFontSet.toString());

		message.append("\n\n=== Platform Font Set ===\n");
		message.append(this.platformFontSet.toString());

		message.append("\n\n=== Fallback Font Set ===\n");
		message.append(this.fallbackFontSet.toString());

		return message.toString();
	}
	
	public void addFont(Font font) throws PDFFontException {
		this.addFont(font, Platform.WINDOWS, ULocale.getDefault());
	}

	public void addFont(Font[] fonts)
			throws PDFFontException {
		this.addFont(fonts, Platform.WINDOWS, ULocale.getDefault());
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.fontset.PDFFontSet#setIgnoreFontLoadingErrors(boolean)
	 */
	public void setIgnoreFontLoadingErrors(boolean ignoreFontLoadingErrors) {
		this.ignoreFontLoadingErrors = ignoreFontLoadingErrors;
	}
}
