/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.reporting;

import com.day.cq.search.QueryBuilder;
import org.apache.sling.jcr.api.SlingRepository;
import javax.jcr.Session;
import java.util.ResourceBundle;

/**
 * This class represents the context for creating a report.
 */
// todo find a better, more OSGi-ish solution
public class Context {

    /**
     * The repository
     */
    private final SlingRepository repository;

    /**
     * The query builder instance to use (if applicable)
     */
    private final QueryBuilder queryBuilder;

    /**
     * The provider for report components
     */
    private final ComponentProvider componentService;

    /**
     * The resource bundle for text retrieval
     */
    private final ResourceBundle resourceBundle;

    /**
     * The resolver cache to use for value resolving
     */
    private final PathResolvingCache cache;

    /**
     * The aggregate manager to be used
     */
    private final AggregateManager aggregateManager;

    /**
     * The preconfigured session to be used
     */
    private final Session reportSession;

    /**
     * The user ID to be used if no preconfigured session is available
     */
    private String reportUserId;

    /**
     * Column to use for sorting (optional; overrides the sort parameters of the report)
     */
    private Column overriddenSortColumn;

    /**
     * Sorting direction (optional; overrides the sort parameters of the report)
     */
    private Sorting.Direction overriddenSortDir;


    /**
     * Creates a new context for creating a report.
     *
     * @param repository The repository
     * @param queryBuilder The query builder
     * @param componentService The provider for report components
     * @param resourceBundle The CQ resource bundle for text retrieval
     * @param cache The resolver cache to use
     * @param reportSession The preconfigured session to use
     */
    public Context(SlingRepository repository, QueryBuilder queryBuilder,
                         ComponentProvider componentService,
                         ResourceBundle resourceBundle, PathResolvingCache cache,
                         Session reportSession) {
        this(repository, queryBuilder, componentService, resourceBundle, cache,
                reportSession, null, null);
    }

    /**
     * Creates a new context for creating a report.
     *
     * @param repository The repository
     * @param queryBuilder The query builder
     * @param componentService The provider for report components
     * @param resourceBundle The CQ resource bundle for text retrieval
     * @param cache The resolver cache to use
     * @param reportUserId The ID of the user to be used for accessing the repository
     */
    public Context(SlingRepository repository, QueryBuilder queryBuilder,
                         ComponentProvider componentService,
                         ResourceBundle resourceBundle, PathResolvingCache cache,
                         String reportUserId) {
        this(repository, queryBuilder, componentService, resourceBundle, cache,
                null, reportUserId, null);
    }

    /**
     * Constructor for overriding implementations.
     */
    protected Context(SlingRepository repository, QueryBuilder queryBuilder,
                            ComponentProvider componentService,
                            ResourceBundle resourceBundle, PathResolvingCache cache,
                            Session reportSession, String reportUserId,
                            AggregateManager aggregateManager) {
        this.repository = repository;
        this.queryBuilder = queryBuilder;
        this.componentService = componentService;
        this.resourceBundle = resourceBundle;
        this.cache = cache;
        if (aggregateManager == null) {
            this.aggregateManager = new AggregateManager(this);
        } else {
            this.aggregateManager = aggregateManager;
        }
        this.reportSession = reportSession;
        this.reportUserId = reportUserId;
        this.overriddenSortColumn = null;
        this.overriddenSortDir = null;
    }

    /**
     * Gets the repository.
     *
     * @return The repository
     */
    public SlingRepository getRepository() {
        return this.repository;
    }

    /**
     * Gets the query builder.
     *
     * @return The query builder
     */
    public QueryBuilder getQueryBuilder() {
        return this.queryBuilder;
    }

    /**
     * Gets a provider for report components.
     *
     * @return The provider for report components
     */
    public ComponentProvider getComponentService() {
        return this.componentService;
    }

    /**
     * Gets the resource bundle for text retrieval.
     *
     * @return The resource bundle
     */
    public ResourceBundle getResourceBundle() {
        return this.resourceBundle;
    }

    /**
     * Gets the resolver cache to be used for creating the report.</p>
     *
     * @return The resolver cache to be used
     */
    public PathResolvingCache getCache() {
        return this.cache;
    }

    /**
     * Gets the aggregate manager to be used.
     *
     * @return The aggregate manager
     */
    public AggregateManager getAggregateManager() {
        return this.aggregateManager;
    }

    /**
     * Gets the {@link Session} to execute the report.
     *
     * @return The {@link Session}; <code>null</code> if a separate session with the
     *         user specified by {@link #getReportUserId} should be used
     */
    public Session getReportSession() {
        return this.reportSession;
    }

    /**
     * Gets the user that has to be used to execute the report.
     *
     * @return The ID of the user; <code>null</code> if the "preconfigured" session provided
     *         by {@link #getReportSession} should be used
     */
    public String getReportUserId() {
        return this.reportUserId;
    }

    /**
     * Gets the overridden sort column of the report.
     *
     * @return The overridden sort colummn of the report; <code>null</code> to keep the
     *         report's original sorting column(s)
     */
    public Column getOverriddenSortColumn() {
        return this.overriddenSortColumn;
    }

    /**
     * Sets the overridden sort column of the report.
     *
     * @param overriddenSortColumn The overridden sort colummn of the report;
     *                             <code>null</code> to keep the report's original sorting
     *                             column(s)
     */
    public void setOverriddenSortColumn(Column overriddenSortColumn) {
        this.overriddenSortColumn = overriddenSortColumn;
    }

    /**
     * Gets the overridden sort direction of the report.
     *
     * @return The overridden sort direction of the report; <code>null</code> to keep the
     *         report's original sorting direction
     */
    public Sorting.Direction getOverriddenSortDir() {
        return this.overriddenSortDir;
    }

    /**
     * Sets the overridden sort direction of the report.
     *
     * @param overriddenSortDir The overridden direction of the report; <code>null</code>
     *                          to keep the report's original sorting direction
     */
    public void setOverriddenSortDir(Sorting.Direction overriddenSortDir) {
        this.overriddenSortDir = overriddenSortDir;
    }

}
