/*
 * File: PDFSimpleFont.java
 * 
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is
 *      obtained from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.font.pdffont;

import java.io.IOException;
import java.io.OutputStream;
import java.util.HashSet;
import java.util.Set;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.font.CatalogDescription;
import com.adobe.fontengine.font.CodePage;
import com.adobe.fontengine.font.CoolTypeScript;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontData;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.InvalidGlyphException;
import com.adobe.fontengine.font.LineMetrics;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.PDFFontDescription;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.ROS;
import com.adobe.fontengine.font.Rect;
import com.adobe.fontengine.font.SWFFont4Description;
import com.adobe.fontengine.font.SWFFontDescription;
import com.adobe.fontengine.font.Scaler;
import com.adobe.fontengine.font.ScanConverter;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetSimpleTrueType;
import com.adobe.fontengine.font.SubsetSimpleType1;
import com.adobe.fontengine.font.UnderlineMetrics;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.XDCFontDescription;
import com.adobe.fontengine.font.postscript.GlyphNamesAccessor;
import com.adobe.fontengine.font.postscript.ScriptHeuristics;
import com.adobe.fontengine.font.postscript.UnicodeCmap;
import com.adobe.fontengine.fontmanagement.CacheSupportInfo;
import com.adobe.fontengine.fontmanagement.PDFSimpleFontValuesAccessor;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontDescription;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontDescription;
import com.adobe.fontengine.fontmanagement.postscript.PostscriptFontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSStretchValue;

final public class PDFSimpleFont extends FontData {

	//---------------------------------------------------------------- members ---
	private final String fontFamily;
	private final String postscriptName;

	private final Rect fontBBox;

	private final double capHeight;
	private final double xHeight;

	private final double ascent;
	private final double descent;
	private final double leading;
	private final double stemV;
	private final double italicAngle;
	private final int flags;
	private final int weight;
	private final CSSStretchValue stretch;

	private final int numGlyphs;

	private final UnicodeCmap cmap;

	// Array of advance widths; by gid
	private final double[] gid2width;
	private final String[] gid2name;

	//----------------------------------------------------------- constructors ---

	public PDFSimpleFont (PDFSimpleFontValuesAccessor dataAccessor) 
	throws InvalidFontException, UnsupportedFontException {
		super (null);
		this.fontFamily = dataAccessor.getFontFamily ();
		this.postscriptName = dataAccessor.getPostscriptName ();
		this.fontBBox = dataAccessor.getFontBBox ();
		this.capHeight = dataAccessor.getCapHeight ();
		this.ascent = dataAccessor.getAscent ();
		this.descent = dataAccessor.getDescent ();
		this.leading = dataAccessor.getLeading ();
		this.stemV = dataAccessor.getStemV ();
		this.italicAngle = dataAccessor.getItalicAngle ();
		this.flags = dataAccessor.getFlags ();
		this.weight = dataAccessor.getFontWeight ();
		this.stretch = dataAccessor.getFontStretch ();
		this.xHeight = dataAccessor.getXHeight();

		int[] gid2characterCode = new int [257];
		this.gid2name = new String [257];

		int lastGid = 0; // notdef
		gid2characterCode[0] = 0;
		gid2name[0] = ".notdef";

		for (int characterCode = 0; characterCode < 256; characterCode++) {
			String name = dataAccessor.getGlyphName (characterCode);
			if (name != null) {
				lastGid++;
				gid2characterCode [lastGid] = characterCode;
				gid2name [lastGid] = name; }}
		this.numGlyphs = lastGid + 1;    


		this.cmap = UnicodeCmap.computeCmapFromGlyphNames 
		(numGlyphs, 
				false, 
				new GlyphNamesAccessor  ()  {
			public String getAGlyphName (int gid) throws  InvalidFontException, UnsupportedFontException { 
				return gid2name [gid]; }}); 

		this.gid2width = new double [numGlyphs];
		for (int gid = 0; gid < numGlyphs; gid++) {
			gid2width [gid] = dataAccessor.getGlyphWidth (gid2characterCode [gid]); }

	}

	//--------------------------------------------------------- general access ---

	/** Tell whether a font is monospaced. */
	public boolean getCoolTypeProportionalRomanFromFontProperties()
	{
		return (flags & 0x1) != 0;
	}

	/** Tell whether the font is symbolic.
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public boolean isSymbolic ()
	throws UnsupportedFontException, InvalidFontException {
		return (flags & 0x4) != 0;
	}

	public int getNumGlyphs ()
	throws InvalidFontException, UnsupportedFontException {
		return numGlyphs;
	}

	//------------------------------------------------------------- unitsPerEm ---

	public double getUnitsPerEmX ()
	throws UnsupportedFontException, InvalidFontException {
		return 1000;
	}

	public double getUnitsPerEmY ()
	throws UnsupportedFontException, InvalidFontException {
		return 1000;
	}

	//----------------------------------------------------------------- bboxes ---

	protected Rect getCoolTypeRawFontBBox () 
	throws InvalidFontException, UnsupportedFontException {
		return fontBBox;
	}

	public Rect getFontBBox () 
	throws InvalidFontException, UnsupportedFontException {
		return fontBBox;
	}
	//----------------------------------------------------------------- script ---

	public CoolTypeScript getCoolTypeScript ()
	throws UnsupportedFontException, InvalidFontException {

		return ScriptHeuristics.getCoolTypeScript (postscriptName,
				getNumGlyphs (),
				new GlyphNamesAccessor  () {
			public String getAGlyphName (int gid) throws  InvalidFontException, UnsupportedFontException { 
				return gid2name [gid]; }});
	}

	//-------------------------------------------------------------- ideoEmBox ---
	private Rect getCoolTypeIdeoEmBoxFromCapHeight ()
	throws InvalidFontException, UnsupportedFontException {

		if (Double.isNaN (capHeight)) {
			return null; }

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		double ymin = - (unitsPerEmY - capHeight) / 2.0d;
		return new Rect (0, ymin, unitsPerEmX,  ymin + unitsPerEmY);
	}

	public Rect getCoolTypeIdeoEmBox ()
	throws UnsupportedFontException, InvalidFontException {
		Rect r;

		if ((r = getCoolTypeIdeoEmBoxFromCapHeight ()) != null) {
			return r; }

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		return new Rect (0, -0.120 * unitsPerEmY, unitsPerEmX, 0.880 * unitsPerEmY);
	}

	//----------------------------------------------------------------- icfBox ---
	public Rect getCoolTypeIcfBox ()
	throws InvalidFontException, UnsupportedFontException {

		return getCoolTypeIcfBoxFromIdeoEmBox (getCoolTypeIdeoEmBox ());
	}

	//----------------------------------------------------------- line metrics ---
	public LineMetrics getCoolTypeLineMetrics ()
	throws UnsupportedFontException, InvalidFontException {
		return new LineMetrics (ascent, descent, leading);
	}

	//------------------------------------------------------ underline metrics ---
	public UnderlineMetrics getCoolTypeUnderlineMetrics ()
	throws UnsupportedFontException, InvalidFontException {

		return new UnderlineMetrics (-150, 50);
	}

	//------------------------------------------------------------------- cmap ---

	public int getGlyphForChar (int unicodeScalarValue)
	throws InvalidFontException, UnsupportedFontException {
		return cmap.getGlyphForChar (unicodeScalarValue);
	}

	//-------------------------------------------------- individual glyph data ---
	public Rect getGlyphBBox (int gid)
	throws UnsupportedFontException, InvalidFontException {
		// we do not have individual glyph bboxes, the best
		// we can get is the font bbox.
		return fontBBox;
	}

	public void getGlyphOutline (int gid, OutlineConsumer consumer)
	throws InvalidFontException, UnsupportedFontException {
		throw new UnsupportedFontException ("cannot get outline for PDFSimpleFont");
	}

	public double getHorizontalAdvance (int gid)
	throws InvalidGlyphException, UnsupportedFontException, InvalidFontException {
		return gid2width [gid];
	}

	//----------------------------------------------------------------------------
	public Scaler getScaler (ScanConverter c)
	throws InvalidFontException, UnsupportedFontException {
		throw new UnsupportedFontException ("getOutlineAccessor not supported for PDFSimpleFont");
	}

	//--------------------------------------------------------- PDF generation ---
	public Permission getEmbeddingPermission (boolean wasEmbedded)
	throws InvalidFontException, UnsupportedFontException {
		return Permission.RESTRICTED;
	}

	class PDFSimpleFontXDCFontDescription extends XDCFontDescription {
		public double getAdvance (int gid)
		throws InvalidFontException, UnsupportedFontException {
			return gid2width [gid];
		}

		public String getBase14Name () {
			return null;
		}

		public double getCapHeight ()
		throws UnsupportedFontException, InvalidFontException {
			return capHeight;
		}

		public double getXHeight()
		throws UnsupportedFontException, InvalidFontException {
			return xHeight;
		}

		public Rect getFontBBox ()
		throws InvalidFontException, UnsupportedFontException {
			return fontBBox;
		}

		public String getFontFamily ()
		throws InvalidFontException, UnsupportedFontException {
			return fontFamily;
		}

		public int getGlyphCid (int gid)
		throws UnsupportedFontException, InvalidFontException {
			return -1;
		}

		public String getGlyphName (int gid)
		throws InvalidFontException, UnsupportedFontException {
			return gid2name [gid];
		}

		public double getItalicAngle () throws InvalidFontException,
		UnsupportedFontException {
			return italicAngle;
		}

		public int getNumGlyphs ()
		throws InvalidFontException, UnsupportedFontException {
			return numGlyphs;
		}

		public String getPostscriptName ()
		throws InvalidFontException, UnsupportedFontException {
			return postscriptName;
		}

		public ROS getROS ()
		throws UnsupportedFontException, InvalidFontException {
			return null;
		}

		public double getStemV ()
		throws UnsupportedFontException, InvalidFontException {
			return stemV;
		}

		public boolean pdfFontIsTrueType ()
		throws InvalidFontException, UnsupportedFontException {
			return false;
		}

		public boolean isSerifFont() {
			return (flags & (1<<1)) != 0;
		}

		public boolean isSmallCapFont() {
			return (flags & (1<<17)) != 0;
		}

		public boolean isAllCapFont() throws InvalidFontException, UnsupportedFontException {
			return (flags & (1<<16)) != 0;
		}

		public int getCIDCount()
		{
			return -1;
		}

		public void subsetAndStream(Subset subset, OutputStream out, boolean preserveROS)
		throws UnsupportedFontException
		{
			throw new UnsupportedFontException("subsetAndStream not meaningful for PDFSimpleFont");
		}

		public void subsetAndStream(SubsetSimpleType1 t1Subset, OutputStream out)
		throws UnsupportedFontException
		{
			throw new UnsupportedFontException("subsetAndStream not meaningful for PDFSimpleFont");
		}

		public void subsetAndStream(SubsetSimpleTrueType ttSubset, OutputStream out)
		throws UnsupportedFontException
		{
			throw new UnsupportedFontException("subsetAndStream not meaningful for PDFSimpleFont");
		}

		public CodePage[] getXDCCodePages()
		throws InvalidFontException, UnsupportedFontException
		{
			if (isSymbolic())
				return new CodePage[0];
			CodePage theCodePage = null;
			for (int i = 0; i < numGlyphs; i++) {
				if (gid2name[i].equals("ecircumflex")) {
					if (theCodePage == CodePage.ROMAN2)
						return new CodePage[0];
					theCodePage = CodePage.ROMAN1;
				} else if (gid2name[i].equals("Ccaron") || gid2name[i].equals("ncaron")) {
					if (theCodePage == CodePage.ROMAN1)
						return new CodePage[0];
					theCodePage = CodePage.ROMAN2;
				}
			}
			if (theCodePage == null)
				return new CodePage[0];
			CodePage[] retVal = new CodePage[1];
			retVal[0] = theCodePage;
			return retVal;
		}

		public void stream(OutputStream out, boolean openTypeFontsAllowed) throws UnsupportedFontException {
			throw new UnsupportedFontException("stream not meaningful for PDFSimpleFont");
		}
	}

	public PDFFontDescription getPDFFontDescription(Font font)
	{
		return new PDFSimpleFontXDCFontDescription();
	}

	public XDCFontDescription getXDCFontDescription(Font font)
	{
		return new PDFSimpleFontXDCFontDescription();
	}

	public Subset createSubset ()
	throws InvalidFontException, UnsupportedFontException {
		throw new UnsupportedFontException 
		("createSubset not meaningful for PDFSimpleFont");
	}

	public void subsetAndStream (Subset subset, OutputStream out,
			boolean preserveROS)
	throws InvalidFontException, UnsupportedFontException, IOException {
		throw new UnsupportedFontException 
		("subsetAndStream not meaningful for PDFSimpleFont");
	}

	public CacheSupportInfo getCacheSupportInfo()
		throws InvalidFontException, UnsupportedFontException
	{
		return new CacheSupportInfo(getClass().getSimpleName(), getNumGlyphs(), false);
	}

	//--------------------------------------------------------- PS properties ---
	public PostscriptFontDescription[] getPostscriptFontDescription ()
	throws InvalidFontException, UnsupportedFontException {
		return new PostscriptFontDescription[] {new PostscriptFontDescription (postscriptName)};
	}

	//--------------------------------------------------------- CSS properties ---
	protected Set getCSSFamilyNames ()
	throws InvalidFontException, UnsupportedFontException {
		Set s = new HashSet ();
		if (fontFamily != null) {
			s.add (fontFamily); }
		return s;
	}

	protected String getPreferredCSSFamilyName()
	throws InvalidFontException, UnsupportedFontException
	{
		return fontFamily;
	}

	protected CSSStretchValue getCSSStretchValue ()
	throws InvalidFontException, UnsupportedFontException {
		return stretch;
	}
	protected int getCSSWeight ()
	throws InvalidFontException, UnsupportedFontException {
		return weight;
	}

	protected boolean isCSSStyleItalic ()
	throws InvalidFontException, UnsupportedFontException {
		return italicAngle != 0 || (flags & (1<<6)) != 0;
	}

	protected boolean isCSSStyleNormal ()
	throws InvalidFontException, UnsupportedFontException {
		return ! isCSSStyleItalic ();
	}

	protected boolean isCSSStyleOblique ()
	throws InvalidFontException, UnsupportedFontException {
		return false;
	}

	protected boolean isCSSVariantNormal ()
	throws InvalidFontException, UnsupportedFontException {
		return ! isCSSVariantSmallCaps ();
	}

	protected boolean isCSSVariantSmallCaps ()
	throws InvalidFontException, UnsupportedFontException {
		return (flags & (1<<17)) != 0;
	}

	//---------------------------------------------------------- FXG Properties ---
	public FXGFontDescription[] getFXGFontDescription(Platform platform,	ULocale locale) 
	throws InvalidFontException, UnsupportedFontException 
	{
		return new FXGFontDescription[0];
	}

	//---------------------------------------------------------- Platform Properties ---
	public PlatformFontDescription[] getPlatformFontDescription(Platform platform, ULocale locale)
	throws InvalidFontException, UnsupportedFontException
	{
		return new PlatformFontDescription[0];
	}

	//----------------------------------------------------------------------------  
	public CatalogDescription getSelectionDescription ()
	throws InvalidFontException, UnsupportedFontException {
		throw new UnsupportedFontException 
		("getSelectionDescription not supported for PDFSimpleFont");
	}

	public SWFFontDescription getSWFFontDescription(boolean wasEmbedded) throws UnsupportedFontException, InvalidFontException {
		// Unsupported. We don't have outlines, so we can't embed these in SWF.
		return null;
	}

	public SWFFont4Description getSWFFont4Description(boolean wasEmbedded) throws UnsupportedFontException, InvalidFontException {
		return null;
	}

}

