/*
 * File: ASCIIHexInputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.InputStream;

/**
 * ASCIIHexInputStream
 *  Reads data coded as Hex (2 hex bytes per byte of data)
 *
 *	Important: This filter assumes the input stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
public class ASCIIHexInputStream extends DecodeInputStream
{
	// Variables
	private static final byte WS = 85;	/* white space */
	private static final byte ED = 87;	/* >, EOD */
	private static final byte IL = 88;	/* illegal character */

	static private byte ascHxDecodeTable[];

	static
	{
		ascHxDecodeTable = new byte[128];

		/* Set up decode table */
		for (int i = 0; i < 128; i++) {
			switch ((char)i) {
			case ' ':
			case '\t':
			case '\n':
			case '\r':
			case '\f':
			case '\b':
				ascHxDecodeTable[i] = WS;    // white space
				break;

			case '>':
				ascHxDecodeTable[i] = ED;  // >, is EOD
				break;

			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				ascHxDecodeTable[i] = (byte)(i - '0');
				break;

			case 'A':
			case 'a':
				ascHxDecodeTable[i] = 10;
				break;

			case 'B':
			case 'b':
				ascHxDecodeTable[i] = 11;
				break;

			case 'C':
			case 'c':
				ascHxDecodeTable[i] = 12;
				break;

			case 'D':
			case 'd':
				ascHxDecodeTable[i] = 13;
				break;

			case 'E':
			case 'e':
				ascHxDecodeTable[i] = 14;
				break;

			case 'F':
			case 'f':
				ascHxDecodeTable[i] = 15;
				break;

			default:
				ascHxDecodeTable[i] = IL;
				break;
			}
		}
	}

	// Constructors
	/**
	 * Creates an input stream filter.
	 * @param in	the input stream
	 */
	public ASCIIHexInputStream(InputStream in, int inSize, int outSize, FilterParams diparams)
	{
		super(in, inSize, outSize, 1, diparams);
	}

	public ASCIIHexInputStream(InputStream  in, FilterParams p)
	{
		super(in, 1, p);
	}

	public ASCIIHexInputStream(InputStream  in)
	{
		this(in, null);
	}

	@Override
	public void fill()
	{
		int value;
		int newChar, digitVal, dp;

		fillLoop:
		while (outCount <= (outBuf.length - 1) && !pendingEOF && pendingException == null) {
			value = 0;
			accumulate:
			for (dp = 4; dp >= 0;) {
				if ( inCount <= inPos ) {
					if (!fillInputBuffer()) {
						pendingEOF = true;
						if (dp == 4)
							break fillLoop;
						break accumulate;
					}
				}
				newChar = inBuf[inPos++] & 0x7F;
				digitVal = ascHxDecodeTable[newChar];

				if ( digitVal < 16 ) {
					value += digitVal << dp;
					dp -= 4;
				}
				else switch (digitVal) {
				case WS: /* whitespace */
					break;

				case ED: /* '>',  EOD */
					pendingEOF = true;
					if (dp == 4)
						break fillLoop;
					break accumulate;

				default: /* illegal character, or EOF */
					pendingException =
						new ASCIIHexFilterException("AscHx illegal char");
					break accumulate;
				}
			} /* accumulate: */

			outBuf[outCount++] = (byte)value;
		} /* fillLoop */
	}
}
