/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

/**
 * Generator contains information about the generator tag contained in
 * many of our XFA files.  
 *
 * The known generator tags will be populated into an
 * enum, for convenient reference.  In our XML files, the generator tag lives in
 * a processing instruction of the form:
 * &lt;?xfa generator="xxxx" APIVersion="x.x.buildNumber.x"&gt;
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class Generator {

	/**
	 * enum describing known generator tags
	 */
	public static final int XFAGen_Unknown			= 0;
	public static final int XFAGen_FF99V250_01		= 1;
	public static final int XFAGen_FF99V310			= 2;
	public static final int XFAGen_JDV530_01		= 3;
	public static final int XFAGen_FF99V40			= 4;
	public static final int XFAGen_FF99V50			= 5;

	public static final int XFAGen_AdobeDesignerV60	= 50;
	public static final int XFAGen_TemplateDesignerV60d = 51;
	public static final int XFAGen_AdobeDesignerV60_SAP = 52;
	public static final int XFAGen_AdobeDesignerV70	= 53;
	public static final int XFAGen_AdobeDesignerV70_SAP	= 54;
	public static final int XFAGen_AdobeDesignerV71	= 55;
	public static final int XFAGen_AdobeDesignerV71_SAP	= 56;
	public static final int XFAGen_AdobeDesignerV80 = 57;
	public static final int XFAGen_AdobeDesignerV80_SAP = 58;

	public static final int XFAGen_XDCV_1_0			= 100;
	public static final int XFAGen_XDCV_1_1			= 101;
	public static final int XFAGen_XDCV_1_3			= 102;
	public static final int XFAGen_XDCV_2_0			= 103;
	
	public static final int XFAGen_2_0				= 150;
	public static final int XFAGen_2_4				= 154;

	private int	meTag = XFAGen_Unknown;
	private String msName;
	private String msAPIVersion = "";
    private int mnBuildNumber;
	
	/**
	 * Default constructor for Generator. 
	 */
	public Generator(String sGenerator, String sAPIVersion) {
		msName = sGenerator;
		msAPIVersion = sAPIVersion;
		mnBuildNumber = 0;
		setEnum();
	}

	public Generator(ProcessingInstruction pi) {
		if (pi.getName() != XFA.XFA)
			return;
			
		String[] parameters = pi.getData().split("[\" '=\t]");
		for (int i=0; i<parameters.length; i++) {
			if (parameters[i].equals(STRS.GENERATOR)) {
				i++;
				while (i<parameters.length) {
					if (parameters[i].length() > 0) {
						msName = parameters[i];
						i++;
						break;
					}
					i++;
				}
			}

			while ((i < parameters.length) && (parameters[i].length() == 0))
				i++;
			
			if ((i < parameters.length) && parameters[i].equals(STRS.APIVERSION) && (i + 1 < parameters.length)) {
				i++;
				while (i<parameters.length) {
					if (parameters[i].length() > 0) {
						msAPIVersion = parameters[i];
						break;
					}
					i++;
				}
			}
		}
		setEnum();
	}
	
	private void setEnum() {
		String sGenerator = msName;
		if (sGenerator.equals("FF99V250_01"))
			meTag = XFAGen_FF99V250_01;

		else if (sGenerator.equals("FF99V310"))
			meTag = XFAGen_FF99V310;

		else if (sGenerator.equals("JDV530_01"))
			meTag = XFAGen_JDV530_01;

		else if (sGenerator.equals("FF99V40"))
			meTag = XFAGen_FF99V40;

		else if (sGenerator.equals("FF99V50"))
			meTag = XFAGen_FF99V50;

		else if (sGenerator.equals("XDCEDIT_V1.0"))
			meTag = XFAGen_XDCV_1_0;

		else if (sGenerator.equals("XDCEDIT_V1.1"))
			meTag = XFAGen_XDCV_1_1;

		else if (sGenerator.equals("XDCEDIT_V1.3"))
			meTag = XFAGen_XDCV_1_3;

		else if (sGenerator.equals("XDCEDIT_V2.0"))
			meTag = XFAGen_XDCV_2_0;

		else if (sGenerator.equals("AdobeDesigner_V6.0"))
			meTag = XFAGen_AdobeDesignerV60;

		else if (sGenerator.equals("Template_Designer_V6.0d"))
			meTag = XFAGen_TemplateDesignerV60d;

		else if (sGenerator.equals("AdobeDesigner_V6.0_SAP"))
			meTag = XFAGen_AdobeDesignerV60_SAP;

		else if (sGenerator.equals("AdobeDesigner_V7.0"))
			meTag = XFAGen_AdobeDesignerV70;

		else if (sGenerator.equals("AdobeDesigner_V7.0_SAP"))
			meTag = XFAGen_AdobeDesignerV70_SAP;

		else if (sGenerator.equals("AdobeDesigner_V7.1"))
			meTag = XFAGen_AdobeDesignerV71;

		else if (sGenerator.equals("AdobeDesigner_V7.1_SAP"))
			meTag = XFAGen_AdobeDesignerV71_SAP;

		else if (sGenerator.equals("AdobeLiveCycleDesigner_V8.0"))
			meTag = XFAGen_AdobeDesignerV80;

		else if (sGenerator.equals("AdobeLiveCycleDesigner_V8.0_SAP"))
			meTag = XFAGen_AdobeDesignerV80_SAP;

		else if (sGenerator.equals("XFA2_0"))
			meTag = XFAGen_2_0;

		else if (sGenerator.equals("XFA2_4"))
			meTag = XFAGen_2_4;

		else
			meTag = XFAGen_Unknown;

		// x.x.buildNumber.x
		
		// Find first dot
	    int nFoundAt = msAPIVersion.indexOf(".");
	    if (nFoundAt > 0) {
	    	nFoundAt++;
	    	// Find second dot
	    	nFoundAt = msAPIVersion.indexOf(".", nFoundAt);
	    }

	    String sBuildNumber = null;
	    if (nFoundAt > 0) {
	    	sBuildNumber = msAPIVersion.substring(nFoundAt + 1);
	    
	    	// Strip off the last dot and number
	    	nFoundAt = sBuildNumber.indexOf(".");
	    	if (nFoundAt > 0)
	    		sBuildNumber = sBuildNumber.substring(0, nFoundAt);
	    }
		
		if (sBuildNumber != null && sBuildNumber.length() > 0) {
			try {
				mnBuildNumber = Integer.parseInt(sBuildNumber);
			} catch (NumberFormatException e) {
			}
		}
	}


	/**
	 * return the generator tag name
	 * @return the string representing the generator tag
	 */
	public String getName() {
		return msName;
	}

	/**
	 * return the generator version
	 * @return the string representing the generator version
	 */
	public String getAPIVersion() {
		return msAPIVersion;
	}

	/**
	 * return the generator build number
	 * @return the integer representing the build number
	 */
	public int getBuildNumber() {
		return mnBuildNumber;
	}
	
	/**
	 * return the generator tag enum.
	 * @return Return the enum value corresponding to this generator tag.
	 * if it's not a recognized name, it will return XFAGenerator_Unknown.
	 */
	public int generator() {
		return meTag;
	}
}
