/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.replication;

import java.util.Calendar;
import java.util.List;
import java.util.Set;

/**
 * <code>ReplicationQueue</code>...
 */
public interface ReplicationQueue {

    /**
     * Returns the name of this queue
     * @return the name of this queue
     */
    String getName();

    /**
     * Returns an unmodifiable list of all entries in this queue.
     * @return list of entries
     */
    List<Entry> entries();

    /**
     * Returns an unmodifiable list of entries for the given path.
     * @param path replication path
     * @return list of entries
     */
    List<Entry> entries(String path);

    /**
     * Returns the entry that matches the given path and was scheduled after
     * the given time.
     * @param path the path
     * @param after the schedule time or <code>null</code>
     * @return the entry or <code>null</code>
     */
    Entry getEntry(String path, Calendar after);

    /**
     * Clears this queue
     */
    void clear();

    /**
     * Clears the entries with the id's specified in the set.
     * Note: this operation might change the sequence of replication events.
     * especially then they depend on each other, eg: delete/create can
     * result in unexpected behaviours.
     *
     * @param ids the ids of the entries to remove
     */
    void clear(Set<String> ids);

    /**
     * Returns <code>true</code> if this queue is blocked by an undeliverable entry.
     * @return <code>true</code> if blocked.
     *
     * @deprecated since 5.4. use {@link #getStatus()} instead
     */
    boolean isBlocked();

    /**
     * Returns <code>true</code> if this queue is paused.
     * @return <code>true</code> if paused.
     *
     * @since 5.4
     */
    boolean isPaused();

    /**
     * Controls if this queue should be paused. Please note that the "paused"
     * state is not persisted (yet), i.e. will be in running state after a restart.
     *
     * @param paused <code>true</code> to pause
     *
     * @since 5.4
     */
    void setPaused(boolean paused);

    /**
     * Returns the time when the last entry was processed.
     * @return the last process time (in ms).
     *
     * @deprecated since 5.4. use {@link #getStatus()} instead
     */
    long lastProcessTime();

    /**
     * Returns the time period (in milliseconds) when the next retry is performed. this
     * only returns a valid time if the queue is blocked.
     * @return the time for next retry
     *
     * @deprecated since 5.4. use {@link #getStatus()} instead
     */
    long nextRetryPeriod();

    /**
     * Returns the queue status
     * @return the queue status
     *
     * @since 5.4
     */
    Status getStatus();

    /**
     * Forces a retry attempt on a blocked entry. this has only an effect, if
     * the queue is blocked.
     */
    void forceRetry();

    /**
     * Informational status of this queue
     */
    interface Status {

        /**
         * Gets the time when this status was generated.
         * @return the time this status was generated.
         */
        long getStatusTime();

        /**
         * Returns the time when the next retry is performed if the queue is
         * blocked or 0 otherwise.
         * @return the time for next retry
         */
        long getNextRetryTime();

        /**
         * Gets the time since the queue is processing an entry or 0 if the queue
         * is currently blocked or idle.
         * @return the time since the queue is processing an entry
         */
        long getProcessingSince();

        /**
         * Returns the time when the last entry was processed.
         * @return the last process time (in ms).
         */
        long getLastProcessTime();

    }

    /**
     * Defines an entry of the replication queue
     */
    interface Entry {

        /**
         * Returns the entry id
         * @return the entry id
         */
        String getId();

        /**
         * Empty array of entries
         */
        Entry[] EMPTY_ARRAY = new Entry[0];

        /**
         * Returns 'my' queue
         * @return the queue of this entry
         */
        ReplicationQueue getQueue();

        /**
         * Returns the replication action
         * @return the replication action
         */
        ReplicationAction getAction();

        /**
         * Returns the replication content or <code>null</code> if this entry
         * was delivered.
         * @return the replication content
         */
        ReplicationContentFacade getContent();

        /**
         * Returns the queue position of this entry
         * @return the queue position
         */
        int getQueuePosition();

        /**
         * Returns the last time this entry was tried to process
         * @return the last process time
         */
        Calendar getLastProcessTime();

        /**
         * Returns the number of times this entry was tried to process
         * @return number of processes
         */
        int getNumProcessed();
    }
}