/*
 *
 *     File: OpenTypeFont.java
 *
 *
 *     ADOBE CONFIDENTIAL
 *     ___________________
 *
 *     Copyright 2004-2006 Adobe Systems Incorporated
 *     All Rights Reserved.
 *
 *     NOTICE: All information contained herein is, and remains the property of
 *     Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *     and technical concepts contained herein are proprietary to Adobe Systems
 *     Incorporated and its suppliers and may be covered by U.S. and Foreign
 *     Patents, patents in process, and are protected by trade secret or
 *     copyright law. Dissemination of this information or reproduction of this
 *     material is strictly forbidden unless prior written permission is obtained
 *     from Adobe Systems Incorporated.
 *
 */

/*
 * Adobe Patent and/or Adobe Patent Pending invention included within this file:
 *
 * Adobe patent application tracking # P376,
 * entitled 'Method for calculating CJK emboxes in fonts',
 * invented by Nathaniel McCully
 * Issued US Patent 7,071,941 on July 4, 2006.
 *
 * Adobe patent application tracking # P376,
 * entitled 'A LINE COMPOSITION CONTROLLABLE DTP SYSTEM, A LINE
 * COMPOSITION CONTROLLING METHOD, A LINE COMPOSITION CONTROL 
 * PROGRAM AND A RECORDING MEDIUM STORING THE SAME',
 * invented by Nathaniel McCully
 * Issued Japanese Patent 3708828 on August 12, 2005.
 *
 * Adobe patent application tracking # P377,
 * entitled 'LINE PREEMPT CONTROLLABLE DTP SYSTEM, A LINE
 * PREEMPT CONTROL METHOD, A LINE PREEMPT CONTROL PROGRAM
 * AND A RECORDING MEDIUM STORING THE SAME'
 * invented by Nathaniel McCully
 * Issued Japanese Patent 3598070 on September 17, 2004.
 */

package com.adobe.fontengine.font.opentype;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.font.CScan;
import com.adobe.fontengine.font.CatalogDescription;
import com.adobe.fontengine.font.CodePage;
import com.adobe.fontengine.font.CoolTypeScript;
import com.adobe.fontengine.font.EmbeddingPermission;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.LineMetrics;
import com.adobe.fontengine.font.OutlineConsumer;
import com.adobe.fontengine.font.PDFFontDescription;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.ROS;
import com.adobe.fontengine.font.Rect;
import com.adobe.fontengine.font.SWFFont4Description;
import com.adobe.fontengine.font.SWFFontDescription;
import com.adobe.fontengine.font.Scaler;
import com.adobe.fontengine.font.ScanConverter;
import com.adobe.fontengine.font.StemFinder;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.SubsetDefaultImpl;
import com.adobe.fontengine.font.SubsetSimpleTrueType;
import com.adobe.fontengine.font.SubsetSimpleType1;
import com.adobe.fontengine.font.TTScan;
import com.adobe.fontengine.font.UnderlineMetrics;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.XDCFontDescription;
import com.adobe.fontengine.font.cff.CFFFont;
import com.adobe.fontengine.font.cff.CFFScaler;
import com.adobe.fontengine.font.cff.CIDKeyedFont;
import com.adobe.fontengine.font.cff.NameKeyedFont;
import com.adobe.fontengine.font.cff.NameKeyedSubset;
import com.adobe.fontengine.font.opentype.Cmap.CmapSelector;
import com.adobe.fontengine.font.opentype.Cmap.MS_EncodingID;
import com.adobe.fontengine.font.opentype.Cmap.PlatformID;
import com.adobe.fontengine.font.opentype.Name.MacintoshEncodingId;
import com.adobe.fontengine.font.opentype.Name.MicrosoftEncodingId;
import com.adobe.fontengine.font.opentype.Name.MicrosoftLCID;
import com.adobe.fontengine.font.opentype.Name.PlatformId;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;
import com.adobe.fontengine.fontmanagement.CacheSupportInfo;
import com.adobe.fontengine.fontmanagement.Platform;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontDescription;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontDescription;
import com.adobe.fontengine.fontmanagement.postscript.PostscriptFontDescription;
import com.adobe.fontengine.inlineformatting.css20.CSS20Attribute.CSSStretchValue;

/** Gives access to an OpenType font */

final public class OpenTypeFont extends com.adobe.fontengine.font.FontData {

	//---------------------------------------------------------------- members ---

	/** The 'head' table, if present, null otherwise */
	public final Head head;

	/** The 'hhea' table, if present, null otherwise */
	public final Hhea hhea;

	/** The 'hmtx' table, if present, null otherwise */
	public final Hmtx hmtx;

	/** The 'maxp' table, if present, null otherwise */
	public final Maxp maxp;

	/** The 'name' table, if present, null otherwise */
	public final Name name;

	/** The 'OS/2' table, if present, null otherwise */
	public final Os2 os2;

	/** The 'cmap' table, if present, null otherwise */
	public final Cmap cmap;

	/** The 'post' table, if present, null otherwise */
	public final Post post;

	/** The 'GPOS' table, if present, null otherwise */
	public final Gpos gpos;

	/** The 'GSUB' table, if present, null otherwise */
	public final Gsub gsub;

	/** The 'GDEF' table if present, null otherwise */
	public final Gdef gdef;

	/** The 'BASE' table if present, null otherwise */
	public final Base base;

	/** The 'glyf' table, if present, null otherwise */
	public final Glyf glyf;

	/** The 'CFF' table, if present, null otherwise */
	public final Cff cff;

	/** The 'vhea' table, if present, null otherwise */
	public final Vhea vhea;

	/** The 'vmtx' table, if present, null otherwise */
	public final Vmtx vmtx;

	/** The 'cvt' table, if present, null otherwise */
	public final Cvt cvt;

	/** The 'fpgm' table, if present, null otherwise */
	public final Fpgm fpgm;

	/** The 'prep' table, if present, null otherwise */
	public final Prep prep;

	/** The 'kern' table, if present, null otherwise */
	public final Kern kern;

	/** The 'gasp' table, if present, null otherwise */
	public final Gasp gasp;

	/** The 'LTSH' table if present, null otherwise */
	public final Ltsh ltsh;

	/** The 'VORG' table if present, null otherwise */
	public final Vorg vorg;

	/** The Apple 'fond' table if present, null otherwise */
	public final Fond fond;

	/** The inverted cmap - access to invertedcmap and invertedcmapHasBeenComputed 
	 * is synchronized by invertedcmapMutex */
	final private Object invertedcmapMutex = new Object ();
	private boolean invertedcmapHasBeenComputed = false;
	private int[] invertedcmap;

	private final String base14CSSName;
	private final String base14PSName;

	private final OTXDCFontDescription xdcDescription;

	static interface ScriptHeuristicChars {
		static final int ARABIC = '\u0622';
		static final int CYRILLIC1 = '\u042f';
		static final int CYRILLIC2 = '\u0451';
		static final int EASTERNEUROPEAN1 = '\u010c';
		static final int EASTERNEUROPEAN2 = '\u0148';
		static final int EASTERNEUROPEAN3 = '\u0173';
		static final int GREEK = '\u03cb';
		static final int HEBREW = '\u05d0';
		static final int JAPANESE = '\u30a2';
		static final int ROMAN1 = '\u0148'; 
		static final int ROMAN2 = '\u0173'; 
		static final int ROMAN3 = '\u00ea';
		static final int THAI = '\u0e01';
		static final int VIETNAMESE = '\u20ab';
	}

	private static final boolean testing_subsetAndStreamForSWFEmbedding = false;
	//----------------------------------------------------------- constructors ---

	public OpenTypeFont (java.util.Map /*tag, Table*/ tables, byte[] digest, String base14CSSName, String base14PSName) 
	throws InvalidFontException, UnsupportedFontException
	{  
		super (digest);

		this.base14CSSName = base14CSSName;
		this.base14PSName = base14PSName;

		head = (Head) tables.get (new Integer (Tag.table_head)); 

		hhea = (Hhea) tables.get (new Integer (Tag.table_hhea));
		maxp = (Maxp) tables.get (new Integer (Tag.table_maxp));
		name = (Name) tables.get (new Integer (Tag.table_name));
		os2  = (Os2)  tables.get (new Integer (Tag.table_os2));
		cmap = (Cmap) tables.get (new Integer (Tag.table_cmap));   
		post = (Post) tables.get (new Integer (Tag.table_post));

		cff  = (Cff)  tables.get (new Integer (Tag.table_CFF));

		gpos = (Gpos) tables.get (new Integer (Tag.table_GPOS));   
		gsub = (Gsub) tables.get (new Integer (Tag.table_GSUB));
		gdef = (Gdef) tables.get (new Integer (Tag.table_GDEF));
		base = (Base) tables.get (new Integer (Tag.table_BASE));

		vhea = (Vhea) tables.get (new Integer (Tag.table_vhea));
		cvt  = (Cvt)  tables.get (new Integer (Tag.table_cvt));
		fpgm = (Fpgm) tables.get (new Integer (Tag.table_fpgm));
		prep = (Prep) tables.get (new Integer (Tag.table_prep));
		kern = (Kern) tables.get (new Integer (Tag.table_kern));
		gasp = (Gasp) tables.get (new Integer (Tag.table_gasp));
		ltsh = (Ltsh) tables.get (new Integer (Tag.table_LTSH));
		vorg = (Vorg) tables.get (new Integer (Tag.table_VORG));
		fond = (Fond) tables.get (new Integer (Tag.table_fond));

		{ HmtxRaw hmtxRaw = (HmtxRaw) tables.get (new Integer (Tag.table_hmtx));
		if (hmtxRaw != null && hhea != null) {
			hmtx = new Hmtx (hmtxRaw, hhea.getNumberOfHMetrics ()); }
		else {
			hmtx = null; }}

		{ VmtxRaw vmtxRaw = (VmtxRaw) tables.get (new Integer (Tag.table_vmtx));
		if (vmtxRaw != null && vhea != null) {
			vmtx = new Vmtx (vmtxRaw, vhea.getNumberOfVMetrics ()); }
		else {
			vmtx = null; }}

		{ GlyfRaw glyfRaw = (GlyfRaw) tables.get (new Integer (Tag.table_glyf));
		LocaRaw locaRaw = (LocaRaw) tables.get (new Integer (Tag.table_loca));
		if (locaRaw != null && head != null) {
			glyf = new Glyf (glyfRaw, locaRaw, head.getIndexToLocFormat ()); }
		else {
			glyf = null; }}

		xdcDescription = new OTXDCFontDescription();
	}

	//--------------------------------------------------------- general access ---

	/** Tell whether the font is symbolic.
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public boolean isSymbolic ()
	throws UnsupportedFontException, InvalidFontException {
		if (cmap != null) {
			return cmap.isSymbolic (); }
		else {
			return true; }
	}

	public int getNumGlyphs () throws InvalidFontException, UnsupportedFontException {
		if (maxp != null)
			return maxp.getNumGlyphs ();
		else if (getCFFFont() != null) {
			return getCFFFont().getNumGlyphs(); }
		else  	
			throw new InvalidFontException("required table (maxp) is missing");
	}


	//------------------------------------------------------------- unitsPerEm ---

	public double getUnitsPerEmX ()
	throws UnsupportedFontException, InvalidFontException {
		if (head != null) {
			return head.getUnitsPerEm (); }
		else if (getCFFFont() != null) {
			return getCFFFont().getUnitsPerEmX(); }
		else
			throw new InvalidFontException("No way to know units per em");
	}

	public double getUnitsPerEmY ()
	throws UnsupportedFontException, InvalidFontException {
		if (head != null) {
			return head.getUnitsPerEm (); }
		else if (getCFFFont() != null) {
			return getCFFFont().getUnitsPerEmY(); }
		else
			throw new InvalidFontException("No way to know units per em"); 
	}

	//---------------------------------------------------- CoolType unitsPerEm ---

	public double getCoolTypeUnitsPerEm ()
	throws UnsupportedFontException, InvalidFontException {

		if (head != null) {
			return head.getUnitsPerEm (); }
		else if (glyf != null) {
			return 2048.0d; }
		else {
			return 1000.0d; }
	}

	//----------------------------------------------------------------- bboxes ---

	public Rect getFontBBox () 
	throws InvalidFontException, UnsupportedFontException {
		if (head != null)
			return head.getFontBBox ();
		else if (getCFFFont() != null) {
			return getCFFFont().getFontBBox(); }
		else
			throw new InvalidFontException("No bbox in font");

	}

	public Rect getCoolTypeRawFontBBox () 
	throws InvalidFontException, UnsupportedFontException {
		if (cff != null) {
			return cff.getFontBBox (); }
		else {
			return getFontBBox (); }
	}


	//----------------------------------------------------------------- script ---

	private class ScriptFromCmapSelector implements CmapSelector {
		int macCount = 0;
		int msCount = 0;
		CoolTypeScript macScript = null;
		CoolTypeScript msScript = null;

		public void cmapFound(int platformID, int platformEncoding, int index) 
		throws InvalidFontException, UnsupportedFontException {
			if (platformID == Cmap.PlatformID.MACINTOSH) {
				if (platformEncoding == Cmap.MacEncodingID.JAPANESE) {
					macCount++;
					macScript = CoolTypeScript.JAPANESE; }
				else if (platformEncoding == Cmap.MacEncodingID.CHINESE_TRADITIONAL) {
					macCount++;
					macScript = CoolTypeScript.TRADITIONAL_CHINESE; }
				else if (platformEncoding == Cmap.MacEncodingID.KOREAN) {
					macCount++;
					macScript = CoolTypeScript.KOREAN; }
				else if (platformEncoding == Cmap.MacEncodingID.HEBREW) {
					macCount++;
					macScript = CoolTypeScript.HEBREW; }
				else if (platformEncoding == Cmap.MacEncodingID.ARABIC) {
					macCount++;
					macScript = CoolTypeScript.ARABIC; }
				else if (platformEncoding == Cmap.MacEncodingID.CHINESE_SIMPLIFIED) {
					macCount++;
					macScript = CoolTypeScript.SIMPLIFIED_CHINESE; } }

			else if (platformID == Cmap.PlatformID.MICROSOFT) {
				if (platformEncoding == Cmap.MS_EncodingID.SHIFTJIS) {
					msCount++;
					msScript = CoolTypeScript.JAPANESE; }
				else if (platformEncoding == Cmap.MS_EncodingID.BIG5) {
					msCount++;
					msScript = CoolTypeScript.TRADITIONAL_CHINESE; }
				else if (platformEncoding == Cmap.MS_EncodingID.PRC) {
					msCount++;
					msScript = CoolTypeScript.SIMPLIFIED_CHINESE; }
				else if (platformEncoding == Cmap.MS_EncodingID.JOHAB) {
					msCount++;
					msScript = CoolTypeScript.KOREAN; }
				else if (platformEncoding == Cmap.MS_EncodingID.WANSUNG) {
					msCount++;
					msScript = CoolTypeScript.KOREAN; }}}

		CoolTypeScript getScript() {
			if (macCount <= 1 && msCount <= 1) {
				if (msCount == 0) {
					return macScript;
				} else if (macCount == 0 || macScript == msScript){
					return msScript; }}

			return null;
		}
	}



	private CoolTypeScript computeCoolTypeScriptFromCmapSubtables ()
	throws InvalidFontException, UnsupportedFontException {
		/// ATMCore.cpp:ATMCGetWritingScript

		if (cmap == null) {
			return null; }

		ScriptFromCmapSelector selector = new ScriptFromCmapSelector();
		cmap.enumerateCmaps(selector);

		return selector.getScript();
	}

	private CoolTypeScript computeCoolTypeScriptFromOs2CodePage(CoolTypeScript foundScript, int numFound)
	throws InvalidFontException, UnsupportedFontException {
		/// ATMCore.cpp:GetWritingScriptFromCodePageRange, part 2
		if (os2 != null) {

			if (os2.supportsCodePage (Os2.CodePage.CP_1250_LATIN_2_EE)) {
				numFound++;
				if (foundScript == null){
					foundScript = CoolTypeScript.ROMAN; }}

			if (   os2.supportsCodePage (Os2.CodePage.CP_1251_CYRILLIC)
					|| os2.supportsCodePage (Os2.CodePage.CP_866_MSDOS_RUSSIAN)
					|| os2.supportsCodePage (Os2.CodePage.CP_855_IBM_CYRILLIC)) {
				numFound++;
				if (foundScript == null) {
					foundScript = CoolTypeScript.CYRILLIC; }}


			if (   os2.supportsCodePage (Os2.CodePage.CP_1253_GREEK)
					|| os2.supportsCodePage (Os2.CodePage.CP_869_IBM_GREEK)
					|| os2.supportsCodePage (Os2.CodePage.CP_737_GREEK)) {
				numFound++;
				if (foundScript == null) {
					foundScript = CoolTypeScript.ROMAN; }}

			if (   os2.supportsCodePage (Os2.CodePage.CP_1255_HEBREW)
					|| os2.supportsCodePage (Os2.CodePage.CP_862_HEBREW)) {
				numFound++;
				if (foundScript == null) {
					foundScript = CoolTypeScript.HEBREW; }}

			if (   os2.supportsCodePage (Os2.CodePage.CP_1256_ARABIC)
					|| os2.supportsCodePage (Os2.CodePage.CP_864_ARABIC)
					|| os2.supportsCodePage (Os2.CodePage.CP_708_ARABIC)) {
				numFound++;
				if (foundScript == null) {
					foundScript = CoolTypeScript.ARABIC; }}

			if (os2.supportsCodePage (Os2.CodePage.CP_874_THAI)) {
				numFound++;
				if (foundScript == null) {
					foundScript = CoolTypeScript.THAI; }}

			if (   os2.supportsCodePage (Os2.CodePage.CP_1252_LATIN_1)  
					|| os2.supportsCodePage (Os2.CodePage.CP_863_MSDOS_CANADIAN)
					|| os2.supportsCodePage (Os2.CodePage.CP_861_MSDOS_ICELANDIC)
					|| os2.supportsCodePage (Os2.CodePage.CP_860_MSDOS_PORTUGUESE)
					|| os2.supportsCodePage (Os2.CodePage.CP_852_LATIN_2)
					|| os2.supportsCodePage (Os2.CodePage.CP_850_WE_LATIN_1)
					|| os2.supportsCodePage (Os2.CodePage.CP_437_US)) {
				numFound++;
				if (foundScript == null) {
					foundScript = CoolTypeScript.ROMAN; }}}

		if (foundScript != null && numFound > 1) {
			int index = getCoolTypeUnicodeCmapIndex();
			if (index != -1 && !computeCoolTypeCmapSupportsScript(foundScript, index)) {
				return null;
			}
		}

		return foundScript;
	}

	private CoolTypeScript computeCoolTypeScriptFromOs2CodePageNotNameKeyed ()
	throws InvalidFontException, UnsupportedFontException {
		/// ATMCore.cpp:GetWritingScriptFromCodePageRange, part 1

		if (os2 == null) {
			return null; }

		CoolTypeScript foundScript = null;
		int numFound = 0;

		if (   os2.supportsCodePage (Os2.CodePage.CP_949_KOREAN_WANSUNG)
				|| os2.supportsCodePage (Os2.CodePage.CP_1361_KOREAN_JOHAB)) {
			numFound++;
			if (foundScript == null) {
				foundScript = CoolTypeScript.KOREAN; }}

		if (os2.supportsCodePage (Os2.CodePage.CP_932_JAPANESE)) {
			numFound++;
			if (foundScript == null) {
				foundScript = CoolTypeScript.JAPANESE; }}

		if (os2.supportsCodePage (Os2.CodePage.CP_936_CHINESE_SIMPLIFIED)) {
			numFound++;
			if (foundScript == null) {
				foundScript = CoolTypeScript.SIMPLIFIED_CHINESE; }}

		if (os2.supportsCodePage (Os2.CodePage.CP_950_CHINESE_TRADITIONAL)) {
			numFound++;
			if (foundScript == null) {
				foundScript = CoolTypeScript.TRADITIONAL_CHINESE; }}

		return computeCoolTypeScriptFromOs2CodePage(foundScript, numFound);
	}

	private CoolTypeScript computeCoolTypeScriptFromOs2UnicodeRange ()
	throws InvalidFontException {
		/// ATMCore.cpp:ATMCGetWritingScript

		if (os2 == null) {
			return null; }

		if (   os2.supportsUnicodeRange (Os2.UnicodeRange.HANGUL_COMPATIBILITY_JAMO)
				|| os2.supportsUnicodeRange (Os2.UnicodeRange.HANGUL_SYLLABLES)) {
			// here CoolType tests bits 57 and 58, with the belief they are
			// HANGUL_SUPPLEMENT_A and HANGUL_SUPPLEMENT_B, but those do not exist as such.
			return CoolTypeScript.KOREAN; }

		if (os2.supportsUnicodeRange (Os2.UnicodeRange.BOPOMOFO)) {
			int count = 0;
			if (os2.supportsUnicodeRange (Os2.UnicodeRange.CJK_UNIFIED_IDEOGRAPHS)) {
				count++; }
			if (os2.supportsUnicodeRange (Os2.UnicodeRange.CJK_COMPATIBILITY_IDEOGRAPHS)) {
				count++; }
			if (os2.supportsUnicodeRange (Os2.UnicodeRange.CJK_COMPATIBILITY_FORMS)) {
				count++; }
			if (count >= 2) {
				return CoolTypeScript.TRADITIONAL_CHINESE; }
			else {
				return CoolTypeScript.SIMPLIFIED_CHINESE; }}

		if (   os2.supportsUnicodeRange (Os2.UnicodeRange.CJK_SYMBOLS_AND_PUNCTUATION)
				|| os2.supportsUnicodeRange (Os2.UnicodeRange.CJK_UNIFIED_IDEOGRAPHS)
				|| os2.supportsUnicodeRange (Os2.UnicodeRange.HIRAGANA)
				|| os2.supportsUnicodeRange (Os2.UnicodeRange.KATAKANA)) {
			return CoolTypeScript.JAPANESE; }

		return null; 
	}

	private int getCoolTypeUnicodeCmapIndex() throws InvalidFontException
	{
		int[][] subtables 
		= {{Cmap.PlatformID.MICROSOFT, Cmap.MS_EncodingID.UNICODE_BMP},
				{Cmap.PlatformID.UNICODE, Cmap.UnicodeEncodingID.ID_2_0_FULL},
				{Cmap.PlatformID.UNICODE, Cmap.UnicodeEncodingID.ID_2_0_BMP},
				{Cmap.PlatformID.UNICODE, Cmap.UnicodeEncodingID.ID_10646_1993},    
				{Cmap.PlatformID.UNICODE, Cmap.UnicodeEncodingID.ID_1_1},
				{Cmap.PlatformID.UNICODE, Cmap.UnicodeEncodingID.ID_1_0}};
		int index = -1;

		for (int i = 0; i < subtables.length; i++) {
			index = cmap.getCmapSubtableIndex (subtables [i][0], subtables [i][1]);
			if (index != -1) {
				break;}}

		return index;
	}

	private boolean computeCoolTypeCmapSupportsScript(CoolTypeScript script, int index) 
	throws UnsupportedFontException, InvalidFontException {
		// ATMCore.cpp:TestUnicodecmapSupportsScript

		if (script == CoolTypeScript.JAPANESE) {
			return cmap.char2glyph(ScriptHeuristicChars.JAPANESE, index) > 0;

		} else if (script == CoolTypeScript.ARABIC) {
			return cmap.char2glyph(ScriptHeuristicChars.ARABIC, index) > 0;  

		} else if (script == CoolTypeScript.HEBREW) {
			return cmap.char2glyph(ScriptHeuristicChars.HEBREW, index) > 0;

		} else if (script == CoolTypeScript.EAST_EUROPEAN_ROMAN) {
			return cmap.char2glyph(ScriptHeuristicChars.EASTERNEUROPEAN1, index) > 0 || 
			cmap.char2glyph(ScriptHeuristicChars.EASTERNEUROPEAN2, index) > 0 ||
			cmap.char2glyph(ScriptHeuristicChars.EASTERNEUROPEAN3, index) > 0;

		} else if (script == CoolTypeScript.CYRILLIC) {
			return cmap.char2glyph(ScriptHeuristicChars.CYRILLIC1, index) > 0 ||
			cmap.char2glyph(ScriptHeuristicChars.CYRILLIC2, index) > 0;

		} else if (script == CoolTypeScript.GREEK) {
			return cmap.char2glyph(ScriptHeuristicChars.GREEK, index) > 0;

		} else if (script == CoolTypeScript.ROMAN) {
			return cmap.char2glyph(ScriptHeuristicChars.ROMAN3, index) > 0;

		} else if (script == CoolTypeScript.THAI) {
			return cmap.char2glyph(ScriptHeuristicChars.THAI, index) > 0;

		} else if (script == CoolTypeScript.VIETNAMESE) {
			return cmap.char2glyph(ScriptHeuristicChars.VIETNAMESE, index) > 0;

		}

		return true;
	}

	private CoolTypeScript computeCoolTypeScriptFromUnicodeCmap () 
	throws UnsupportedFontException, InvalidFontException {
		// ATMCore.cpp:GetWritingScriptFromUniCMap

		if (cmap == null) {
			return null; }

		int index = getCoolTypeUnicodeCmapIndex();

		if (index == -1) {
			return null; }

		if (cmap.char2glyph (ScriptHeuristicChars.JAPANESE, index) != 0) {
			return CoolTypeScript.JAPANESE; }

		if (cmap.char2glyph (ScriptHeuristicChars.ARABIC, index) != 0) {
			return CoolTypeScript.ARABIC; }

		if (cmap.char2glyph (ScriptHeuristicChars.HEBREW, index) != 0) {
			return CoolTypeScript.HEBREW; }

		// CoolType comments:
		//  - LS 6/2000. Some old Bitstream fonts that should be kCTRomanScript
		//    include the Ccaron character but not ncaron. Do a special check for
		//    ncaron here to eliminate them.
		//
		//  - Special check for Roman fonts (like Hoefler-Italic, file name 
		//    "Hoefler.ttf") that support both Roman charset and EastEuropean 
		//    charset. If this font supports EE, checks to see if it also has
		//    ecircumflex. If so, regard it as Roman font, not EE font.

		// The CoolType code is a bit weird; notice that
		// the test on U+010C is irrelevant. If it is mapped, then we go on
		// testing U+0148; if that one is not mapped, then writingScript is
		// not set. Then, the next test again on U+0148! I suppose that this was
		// not the intended logic, and that adding U+010C to the || test just
		// below was the intent.

		if (   cmap.char2glyph (ScriptHeuristicChars.ROMAN1, index) != 0
				|| cmap.char2glyph (ScriptHeuristicChars.ROMAN2, index) != 0) {
			if (cmap.char2glyph (ScriptHeuristicChars.ROMAN3, index) != 0) {
				return CoolTypeScript.ROMAN; }
			else {
				return CoolTypeScript.EAST_EUROPEAN_ROMAN; }}

		if (   cmap.char2glyph (ScriptHeuristicChars.CYRILLIC1, index) != 0
				|| cmap.char2glyph (ScriptHeuristicChars.CYRILLIC2, index) != 0) {
			return CoolTypeScript.CYRILLIC; }

		if (cmap.char2glyph (ScriptHeuristicChars.GREEK, index) != 0) {
			return CoolTypeScript.GREEK; }

		if (cmap.char2glyph (ScriptHeuristicChars.ROMAN3, index) != 0) {
			return CoolTypeScript.ROMAN; }

		if (cmap.char2glyph (ScriptHeuristicChars.THAI, index) != 0) {
			return CoolTypeScript.THAI; }

		if (cmap.char2glyph (ScriptHeuristicChars.VIETNAMESE, index) != 0) {
			return CoolTypeScript.VIETNAMESE; }

		return null;
	}

	private CoolTypeScript computeCoolTypeScriptFromOs2fsSelection ()
	throws InvalidFontException {
		// ATMCore.cpp:ATMCGetWritingScript

		// CoolType Comment:
		//   Check for Windows 3.1 Arabic fonts in the uppper byte of fsSelection
		//   (undocumented in OT spec 1.4!)

		if (os2 == null) {
			return null; }

		int x = os2.getSelection () >> 8;

		if (x == 0xb2 || x == 0xb3) {
			return CoolTypeScript.ARABIC; }

		if (x == 0xb1) {
			return CoolTypeScript.HEBREW; }

		return null;
	}

	private CoolTypeScript computeCoolTypeScriptFromNameEntriesLanguages ()
	throws InvalidFontException {
		// ATMCore.cpp:ATMCGetWritingScript

		// CoolType Comment:
		//   NIS Japanese fonts don't have anything of the above information
		// [= heuristics applied before this one]

		if (os2 == null || name == null) {
			return null; }

		long v = os2.getVendor ();

		if (v == 0x4c455452 /* LETR */ || v == 0x44594e41 /* DYNA */ || v == 0x48414e53 /* HANS */) {
			if (   name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.KOREAN_KOREA.getLanguageCode(), Name.PredefinedNames.FULL_FONT_NAME)
					|| name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.KOREAN_KOREA.getLanguageCode(), Name.PredefinedNames.COPYRIGHT_NOTICE)) {
				return CoolTypeScript.KOREAN; }

			if (   name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.CHINESE_TAIWAN.getLanguageCode(), Name.PredefinedNames.FULL_FONT_NAME)
					|| name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.CHINESE_TAIWAN.getLanguageCode(), Name.PredefinedNames.COPYRIGHT_NOTICE)) {
				return CoolTypeScript.TRADITIONAL_CHINESE; }

			if (   name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.CHINESE_PRC.getLanguageCode(), Name.PredefinedNames.FULL_FONT_NAME)
					|| name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.CHINESE_PRC.getLanguageCode(), Name.PredefinedNames.COPYRIGHT_NOTICE)) {
				return CoolTypeScript.SIMPLIFIED_CHINESE; }

			if (   name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.JAPANESE_JAPAN.getLanguageCode(), Name.PredefinedNames.FULL_FONT_NAME)
					|| name.hasName (Name.PlatformId.MICROSOFT, -1, Name.MicrosoftLCID.JAPANESE_JAPAN.getLanguageCode(), Name.PredefinedNames.COPYRIGHT_NOTICE)) {
				return CoolTypeScript.JAPANESE; }}

		return null;
	}


	public CoolTypeScript getCoolTypeScript () 
	throws UnsupportedFontException, InvalidFontException {

		CoolTypeScript ctScript;

		ROS ros = null;
		if (cff != null) {
			ros = cff.getROS (); }

		// On MacOS, CoolType tests the MacOS reported script

		if ((ctScript = CoolTypeScript.fromWellKnownROS (ros)) != null) {
			return ctScript; }

		// From CoolType:
		//  It is important to do the cmap check before attempting to check
		//  the OS/2 table.  There are many non-Unicode, Microsoft CJK fonts
		//  (for example a whole bunch from Cannon) which have the wrong bits
		//  set in their OS/2 table.  If we use that data we end up thinking
		//  that the fonts are Thai instead of Japanese.  The logic this way
		//  says that if the font has a non-Unicode, non-Roman, non-Macintosh
		//  cmap then we use the writing script implied by that cmap to set
		//  the script for the font.  Even if this information disagrees with
		//  what is in the OS/2 table

		if ((ctScript = computeCoolTypeScriptFromCmapSubtables ()) != null) {
			return ctScript; }

		if (glyf != null || this.cff.getCFFFont() instanceof com.adobe.fontengine.font.cff.CIDKeyedFont) {
			if ((ctScript = computeCoolTypeScriptFromOs2CodePageNotNameKeyed ()) != null) {
				return ctScript; } }
		else {
			if ((ctScript = computeCoolTypeScriptFromOs2CodePage(null, 0)) != null) {
				return ctScript; }}

		if ((ctScript = computeCoolTypeScriptFromOs2UnicodeRange ()) != null) {
			return ctScript; }

		if ((ctScript = computeCoolTypeScriptFromOs2fsSelection ()) != null) {
			return ctScript; }

		if ((ctScript = computeCoolTypeScriptFromNameEntriesLanguages ()) != null) {
			return ctScript; }

		if ((ctScript = computeCoolTypeScriptFromUnicodeCmap ()) != null) {
			return ctScript; }

		// On Windows, CoolType tests the Window's reported charset

		if ((ctScript = CoolTypeScript.fromAnyROS (ros)) != null) {
			return ctScript; }

		return CoolTypeScript.ROMAN;  
	}    

	//------------------------------------------------------------- cap height ---

	public double getCoolTypeCapHeight ()
	throws InvalidFontException, UnsupportedFontException {

		if (os2 != null) {
			int capHeight = os2.getCapHeight ();
			// if the capHeight in OS/2 is 0, we behave as if it where not present
			if (capHeight != Integer.MAX_VALUE && capHeight != 0) {
				return capHeight; }}

		return getCoolTypeCapHeightFromGlyphs ();
	}

	//-------------------------------------------------------------- ideoEmBox ---

	private Rect computeCoolTypeIdeoEmBoxFromBaseTable ()
	throws UnsupportedFontException, InvalidFontException {
		if (base == null) {
			return null; }

		int ideoH = base.getBaselinePosition (Orientation.HORIZONTAL, 
				Tag.script_DFLT, Tag.baseline_ideo); 
		if (ideoH == Integer.MAX_VALUE)  {
			return null; }

		int romnH = base.getBaselinePosition (Orientation.HORIZONTAL, 
				Tag.script_DFLT, Tag.baseline_romn);
		if (romnH != Integer.MAX_VALUE && romnH < ideoH) {
			return null; }

		double ymin = ideoH;

		int idtpH = base.getBaselinePosition (Orientation.HORIZONTAL, 
				Tag.script_DFLT, Tag.baseline_idtp);
		double ymax = (idtpH != Integer.MAX_VALUE) ? idtpH : (ymin + getUnitsPerEmY ());

		double xmin = 0.0d;

		int idtpV = base.getBaselinePosition (Orientation.VERTICAL, 
				Tag.script_DFLT, Tag.baseline_idtp);
		double xmax = (idtpV != Integer.MAX_VALUE) ? idtpV : (xmin + getUnitsPerEmX ());

		return new Rect (xmin, ymin, xmax, ymax);
	}

	private boolean validateOs2OrHheaIdeoEmBox (int ascender, int descender) 
	throws UnsupportedFontException, InvalidFontException {

		/* Validate the embox metrics obtained from the OS/2 or hhea
		 * table.  This generally handles cases like Osaka and Osaka-Mono
		 * which don't in fact record their correct embox (220,-36) in
		 * their hhea, but instead (256,-64) and (219,-49)
		 * respectively. */

		for (int i = 0; i < typicalCharactersForIdeoEmBoxComputation.length; i++) {
			int gid = getCoolTypeGlyphForChar (typicalCharactersForIdeoEmBoxComputation [i]);
			if (gid != 0) {
				Rect r = getGlyphBBox (gid);

				double diff = Math.abs ((ascender-r.ymax) - (r.ymin-descender));
				double unitsPerEmY = getUnitsPerEmY ();

				return (   descender <= r.ymin && r.ymax <= ascender  /* within ascender/descender */
						&& diff <= 0.003d * unitsPerEmY /* centered */); }}

		return false;
	}

	private Rect computeCoolTypeIdeoEmBoxFromOs2OrHhea ()
	throws UnsupportedFontException, InvalidFontException {

		int ascender = -1; 
		int descender = 0;

		if (os2 != null) {
			ascender = os2.getTypoAscender ();
			descender = os2.getTypoDescender (); }
		if (ascender <= descender) {
			ascender = hhea.getAscender ();
			descender = hhea.getDescender (); 
			if (ascender <= descender) {
				return null; }}

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		if (   cff != null 
				|| ascender - descender == unitsPerEmY
				|| validateOs2OrHheaIdeoEmBox (ascender, descender)) {
			return new Rect (0, descender, unitsPerEmX, ascender); }

		return null;
	}

	/** Compute the IdeoEmBox from the CapHeight.
	 */
	private Rect computeCoolTypeIdeoEmBoxFromCapHeight ()
	throws InvalidFontException, UnsupportedFontException {

		double capHeight = getCoolTypeCapHeight ();
		if (Double.isNaN (capHeight)) {
			return null; }

		if (base != null) {
			int romnH = base.getBaselinePosition (Orientation.HORIZONTAL, 
					Tag.script_DFLT, Tag.baseline_romn);
			if (romnH != Integer.MAX_VALUE) {
				capHeight -= romnH; }}

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		double ymin = - (unitsPerEmY - capHeight) / 2.0d;
		return new Rect (0, ymin, unitsPerEmX, ymin + unitsPerEmY);
	}

	/** Compute the IdeoEmBox.
	 * Various heuristics are applied until the IdeoEmBox can be determined.
	 */
	public Rect getCoolTypeIdeoEmBox () 
	throws InvalidFontException, UnsupportedFontException {

		Rect r;

		if ((r = computeCoolTypeIdeoEmBoxFromBaseTable ()) != null) {
			return r; }

		if (useCoolTypeCJKHeuristics ()) {

			if ((r = computeCoolTypeIdeoEmBoxFromOs2OrHhea ()) != null) {
				return r; }

			if ((r = getCoolTypeIdeoEmBoxFromFullBoxCharacter ()) != null) {
				return r; }

			if ((r = getCoolTypeIdeoEmBoxFromTypicalCharacter ()) != null) {
				return r; }}

		else {
			if ((r = computeCoolTypeIdeoEmBoxFromCapHeight ()) != null) {
				return r; }}

		double unitsPerEmX = getUnitsPerEmX ();
		double unitsPerEmY = getUnitsPerEmY ();
		return new Rect (0, -0.120 * unitsPerEmY, unitsPerEmX, 0.880 * unitsPerEmY);
	}

	//----------------------------------------------------------------- icfBox ---

	/** Compute the IcfBox from the BASE table.
	 * This heuristic applies iff the horizontal 'icfb' baseline is present. 
	 * Other icf* baselines are used if present.
	 */  
	private Rect computeCoolTypeIcfBoxFromBaseTable (Rect ideoEmBox) 
	throws InvalidFontException, UnsupportedFontException {
		if (base == null) {
			return null; }

		int icfbH = base.getBaselinePosition (Orientation.HORIZONTAL, 
				Tag.script_DFLT, Tag.baseline_icfb);
		if (icfbH == Integer.MAX_VALUE) {
			return null; }

		int icfbV = base.getBaselinePosition (Orientation.VERTICAL, 
				Tag.script_DFLT, Tag.baseline_icfb);
		int icftH = base.getBaselinePosition (Orientation.HORIZONTAL, 
				Tag.script_DFLT, Tag.baseline_icft);
		int icftV = base.getBaselinePosition (Orientation.VERTICAL, 
				Tag.script_DFLT, Tag.baseline_icft);

		double ymin = icfbH;
		double ymax = (icftH != Integer.MAX_VALUE) ?
				icftH : ideoEmBox.ymax - (ymin - ideoEmBox.ymin);
		double xmin = (icfbV != Integer.MAX_VALUE) ?
				icfbV : ymin - ideoEmBox.ymin;
		double xmax = (icftV != Integer.MAX_VALUE) ?
				icftV : ideoEmBox.xmax - (xmin - ideoEmBox.xmin);

		if (   useCoolTypeCJKHeuristics () 
				&& (   ymin < ideoEmBox.ymin || ideoEmBox.ymax < ymax
						|| xmin < ideoEmBox.xmin || ideoEmBox.xmax < xmax)) {
			return null; }

		return new Rect (xmin, ymin, xmax, ymax);
	}


	/** Compute the IcfBox.
	 * Various heuristics are applied until the IcfBox can be determined.
	 */
	public Rect getCoolTypeIcfBox ()
	throws InvalidFontException, UnsupportedFontException {

		Rect ideoEmBox = getCoolTypeIdeoEmBox ();

		Rect r;

		if ((r = computeCoolTypeIcfBoxFromBaseTable (ideoEmBox)) != null) {
			return r; }

		if (useCoolTypeCJKHeuristics ()) {
			if ((r = getCoolTypeIcfBoxFromTypicalCharacter (ideoEmBox)) != null) {
				return r; }}

		return getCoolTypeIcfBoxFromIdeoEmBox (ideoEmBox);
	}

	//----------------------------------------------- line metrics computation ---

	private boolean useOs2 () throws InvalidFontException {
		if (os2 == null)
			return false;

		int a = os2.getTypoAscender ();
		int d = os2.getTypoDescender ();
		if (d > 0 && head != null && head.getYMin () <= 0) {
			// some fonts, e.g. Arial-Black 2.35, have the 
			// wrong sign in OS/2.typoDescender
			d = - d; }
		return d < a;
	}

	private double computeCoolTypeLineGapForCJK () 
	throws UnsupportedFontException, InvalidFontException{

		if (! ( cff != null && head != null && head.isConverted ()) && useOs2 ()) {
			return Math.abs (os2.getTypoLineGap ()); }
		else if (cff == null) {
			return Math.abs (hhea.getLineGap ()); }
		else {
			return getUnitsPerEmY () / 2; }
	}

	private LineMetrics computeCoolTypeLineMetricsFromTypicalCharacters (double linegap)
	throws UnsupportedFontException, InvalidFontException {

		int dGid = getCoolTypeGlyphForChar ('d');
		int pGid = getCoolTypeGlyphForChar ('p');
		if (dGid != 0 && pGid != 0) {
			double ascender = getGlyphBBox (dGid).ymax;
			double descender = getGlyphBBox (pGid).ymin;
			if (descender < ascender) {
				return new LineMetrics (ascender, descender, linegap); }}

		return null; 
	}

	private LineMetrics computeCoolTypeLineMetricsFromICFBox (double linegap)
	throws UnsupportedFontException, InvalidFontException {

		Rect icfBox = getCoolTypeIcfBox ();
		return new LineMetrics (icfBox.ymax, 
				icfBox.ymax - getUnitsPerEmY (), 
				linegap); 
	}

	private LineMetrics computeCoolTypeLineMetricsFromOs2 ()
	throws UnsupportedFontException, InvalidFontException {

		if (os2 == null)
			return null;

		int ascender = os2.getTypoAscender ();
		int descender = os2.getTypoDescender ();
		if (descender > 0 && head != null && head.getYMin () <= 0) {
			// some fonts, e.g. Arial-Black 2.35, have the wrong sign 
			// in OS/2.typoDescender
			descender = - descender; }
		if (descender < ascender) {
			return new LineMetrics (ascender, descender, 
					Math.abs (os2.getTypoLineGap ())); }

		return null;
	}

	private LineMetrics computeCoolTypeLineMetricsFromHhea ()
	throws UnsupportedFontException, InvalidFontException {

		int ascender = hhea.getAscender ();
		int descender = hhea.getDescender ();
		if (descender < ascender) { 
			return new LineMetrics (ascender, descender, 
					Math.abs (hhea.getLineGap ())); }

		return null;
	}


	/** Emulates the CoolType API CTFontDict:GetHorizontalMetrics CoolType API.
	 * 
	 * <p>The metrics are expressed in the design space of the font, 
	 * i.e. they need to be converted through the metrics matrix.
	 * 
	 * <p>This methods never returns null. 
	 * 
	 * <p>See also the {@link #getLineMetrics()} method.
	 * 
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public LineMetrics getCoolTypeLineMetrics ()
	throws UnsupportedFontException, InvalidFontException {

		LineMetrics lm;

		if (useCoolTypeCJKHeuristics ()) {
			double linegap = computeCoolTypeLineGapForCJK ();

			if ((lm = computeCoolTypeLineMetricsFromTypicalCharacters (linegap)) != null) {
				return lm; }

			return computeCoolTypeLineMetricsFromICFBox (linegap); }

		if (! (cff != null && head != null && head.isConverted ())) {
			if ((lm = computeCoolTypeLineMetricsFromOs2 ()) != null) {
				return lm; }}

		if (glyf != null) {
			if ((lm = computeCoolTypeLineMetricsFromHhea ()) != null) {
				return lm; }}

		return getCoolTypeLineMetricsFromFontBbox ();
	}


	/** Return the line metrics for this font. 
	 * 
	 * <p>The metrics are expressed in the design space of the font, 
	 * i.e. they need to be converted through the metrics matrix.
	 * 
	 * <p>Some font formats do not support the notion of line metrics,
	 * and in those cases, this method returns null.
	 * 
	 * <p>See also the {@link #getCoolTypeLineMetrics()} method.
	 * 
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public LineMetrics getLineMetrics ()
	throws UnsupportedFontException, InvalidFontException {
		if (os2 != null) {
			return os2.getLineMetrics (); }
		else {
			return null; }
	}

	//------------------------------------------------------ underline metrics ---

	public UnderlineMetrics getCoolTypeUnderlineMetrics ()
	throws UnsupportedFontException, InvalidFontException {

		double unitsPerEmY = getUnitsPerEmY ();

		if (cff != null) {
			return cff.getCoolTypeUnderlineMetrics (getCoolTypeUnitsPerEm (), unitsPerEmY); }

		if (post != null) {
			int p = post.getUnderlinePosition ();
			int u = post.getUnderlineThickness ();
			if (p != 0 || u != 0) {
				return new UnderlineMetrics (p, Math.abs (u)); }}

		if (glyf != null && useCoolTypeCJKHeuristics ()) {
			int gid = getCoolTypeGlyphForChar (0x672C); 
			if (gid != 0) {
				Rect r = getGlyphBBox (gid);
				return new UnderlineMetrics (r.ymin - 0.050d * unitsPerEmY / 2.0d, 
						0.050d * unitsPerEmY); }}

		if (useCoolTypeCJKHeuristics ()) {
			return new UnderlineMetrics (-0.100d * unitsPerEmY, 0.050d * unitsPerEmY); }

		return new UnderlineMetrics (-0.150d * unitsPerEmY, 0.050d * unitsPerEmY);
	}

	//------------------------------------------------------- proportional roman ---

	public boolean getCoolTypeProportionalRomanFromFontProperties ()
	throws InvalidFontException {
		if (os2 != null) {
			switch (os2.panoseIndicatesProportional ()) {
			case 0: {
				return false; }
			case 1: {
				return true; }}}

		if (cff != null) {
			return cff.getCFFFont().getCoolTypeProportionalRomanFromFontProperties (); }

		return false;
	}

	//--------------------------------------------------------------- baseline ---

	public int computeRomanBaselineH () throws InvalidFontException {
		if (base != null) {
			int bRomnH = base.getBaselinePosition (Orientation.HORIZONTAL,
					Tag.script_DFLT, Tag.baseline_romn);
			if (bRomnH == Integer.MAX_VALUE) {
				return 0; }

			int bIdeoH = base.getBaselinePosition (Orientation.HORIZONTAL,
					Tag.script_DFLT, Tag.baseline_ideo);
			if (bIdeoH == Integer.MAX_VALUE || bRomnH >= bIdeoH) {
				return bRomnH; }}

		return 0;
	}

	public double computeRomanBaselineV () 
	throws UnsupportedFontException, InvalidFontException {
		if (base != null) {
			int bRomnV = base.getBaselinePosition (Orientation.VERTICAL, 
					Tag.script_DFLT, Tag.baseline_romn);
			if (bRomnV != Integer.MAX_VALUE) {
				return bRomnV; }}

		Rect ideoEmBox = getCoolTypeIdeoEmBox ();
		int romanBaselineH = computeRomanBaselineH ();
		return -ideoEmBox.ymin - romanBaselineH + ideoEmBox.xmin;
	}


	//------------------------------------------------------------------- cmap ---

	/** {@inheritDoc} */
	public int getGlyphForChar (int usv) 
	throws InvalidFontException, UnsupportedFontException {

		if (cmap == null) {
			return 0; }

		return cmap.unicodeChar2glyph (usv);
	}

	public int getCoolTypeGlyphForChar (int usv)
	throws InvalidFontException, UnsupportedFontException {

		if (cmap == null) {
			return 0; }

		return cmap.coolTypeUnicodeChar2glyph (usv);
	}

	/** Find a character mapped to some glyph.
	 * @param gid the target glyph
	 * @return one of the characters mapped to the target glyph 
	 *    via the Microsoft/Unicode_BMP cmap if there is one,
	 *    -1 if none
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	public int getCharForGlyph (int gid)
	throws UnsupportedFontException, InvalidFontException {
		synchronized (invertedcmapMutex) {
			if (! invertedcmapHasBeenComputed) {
				if (cmap == null)
					return -1;
				int index = cmap.getCmapSubtableIndex (PlatformID.MICROSOFT, 
						MS_EncodingID.UNICODE_BMP);
				if (index != -1) {
					invertedcmap = cmap.glyph2char (getNumGlyphs (), index); }
				else {
					invertedcmap = null; }
				invertedcmapHasBeenComputed = true; }

			if (invertedcmap != null) {
				return invertedcmap [gid]; }
			else {
				return -1; }}
	}

	//-------------------------------------------------- individual glyph data ---

	public double getHorizontalAdvance (int gid) 
	throws InvalidFontException, UnsupportedFontException {
		if (hmtx != null)
			return hmtx.getHorizontalAdvance (gid);
		else if (getCFFFont() != null)
			return getCFFFont().getHorizontalAdvance(gid);
		else
			throw new InvalidFontException("No way to know horizontal advance");

	}

	public void getGlyphOutline (int gid, OutlineConsumer consumer, int pathType)
	throws UnsupportedFontException, InvalidFontException {
		if (cff != null) {
			cff.getOutline (gid, consumer); }
		else if (glyf != null ) {
			TTParser parser = new TTParser ();
			parser.parse (this, gid, consumer, pathType); }
        else {
            throw new InvalidFontException("Either cff or glyf must be present"); }
	}

	public void getGlyphOutline (int gid, OutlineConsumer consumer)
	throws UnsupportedFontException, InvalidFontException {
		if (cff != null) {
			cff.getOutline (gid, consumer); }
		else if (glyf != null ) {
			TTParser parser = new TTParser ();
			parser.parse (this, gid, consumer); }
        else {
            throw new InvalidFontException("Either cff or glyf must be present"); }
    }

	public Rect getGlyphBBox (int gid) 
	throws UnsupportedFontException, InvalidFontException {

		if (cff != null) {
			Rect result = cff.getGlyphBoundingBoxInMetricSpace (gid); 
			if (Math.abs(getUnitsPerEmX() - cff.getCFFFont().getUnitsPerEmX()) < 0.1 && 
					Math.abs(getUnitsPerEmY() - cff.getCFFFont().getUnitsPerEmY()) < 0.1) 
				return result;

			double cffUnitsPerEmX = cff.getCFFFont().getUnitsPerEmX();
			double cffUnitsPerEmY = cff.getCFFFont().getUnitsPerEmY();

			return new Rect(result.xmin * getUnitsPerEmX()/cffUnitsPerEmX,
					result.ymin * getUnitsPerEmY()/cffUnitsPerEmY,
					result.xmax * getUnitsPerEmX()/cffUnitsPerEmX,
					result.ymax * getUnitsPerEmY()/cffUnitsPerEmY);

		}
		else if (glyf != null) {
			return glyf.getGlyphBoundingBox (gid); }
        else {
            throw new InvalidFontException("Either cff or glyf must be present."); }
  
	}

	//----------------------------------------------------------------------------

	public Scaler getScaler (ScanConverter c)
	throws InvalidFontException, UnsupportedFontException {

		if (glyf != null) {
			if (c == null) {
				c = new TTScan (); }
			return new TTScaler (this, c); }
		else {
			if (c == null) {
				c = new CScan (false, 1.0d, true); }
			return new CFFScaler (cff.cffFont, c); }
	}

	//--------------------------------------------------------- pdf generation ---

	/** Get the name of glyph <code>glyphID</code>.
	 */
	public String getGlyphName (int glyphID)
	throws UnsupportedFontException, InvalidFontException {

		String s = null;

		if (glyphID == 0)
			return ".notdef";

		if (post != null) {
			s = post.getGlyphName (glyphID); }

		if (s == null && cff != null) {
			s = cff.getGlyphName (glyphID); }

		if (s == null) {
			int usv = getCharForGlyph(glyphID);
			if (usv != -1)
				s = GlyphNames.resolveUSVToAGNCName(usv); }

		if (s == null) {
			return "g" + glyphID; }

		return s;
	}

	public ROS getROS () {
		if (cff != null) {
			return cff.getROS (); }

		return null;
	}

	public int getCIDCount()
	{
		if (cff != null && cff.getCFFFont() instanceof CIDKeyedFont)
			return ((CIDKeyedFont)cff.getCFFFont()).getCIDCount();
		return -1;
	}

	public int getGlyphCid(int glyphID)
	throws InvalidFontException, UnsupportedFontException {

		if (cff != null) {
			return cff.getGlyphCid (glyphID); }

		return -1; 
	}


	static abstract class SingleNameSelector implements Name.NameSelector {

		protected String currentSelection;
		protected int currentPlatformID;
		protected int currentEncoding;
		protected int currentLanguage;

		SingleNameSelector () {
			currentSelection = null;
			currentPlatformID = -1;
		}

		protected boolean pickMSLanguage (int language) {
			if (   currentLanguage != MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode() 
					&& language == MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode()) {
				return true; }

			return false;
		}

		protected boolean pickMacName(int encoding) {
			if (   encoding == MacintoshEncodingId.ROMAN
					&& currentEncoding != MacintoshEncodingId.ROMAN) {
				return true; }

			return false;
		}
	}

	static class SingleFamilyName extends SingleNameSelector {

		protected boolean pickMSName(int encoding, int language) {
			if (currentEncoding == encoding) {
				return pickMSLanguage(language); }

			// keep the first MS full unicode name we find
			if (currentEncoding == MicrosoftEncodingId.UTF16_BE_FULL) {
				return false; }

			if (encoding == MicrosoftEncodingId.UTF16_BE_FULL) {
				return true; }

			if (currentEncoding == MicrosoftEncodingId.UTF16_BE_BMP_ONLY) {
				return false; }

			if (encoding == MicrosoftEncodingId.UTF16_BE_BMP_ONLY) {
				return true; }

			return false;
		}

		public boolean nameFound(String name, int nameID, int platformID, int encoding, int language) {

			if (name == null || name.length() == 0 || name.charAt(0) == 0)
				return false;

			if (currentPlatformID == -1) {
				currentSelection = name;
				currentPlatformID = platformID;
				currentEncoding = encoding;
				currentLanguage = language;
				return false; }

			// prefer microsoft names over macintosh names
			if (   currentPlatformID == PlatformId.MICROSOFT 
					&& platformID == PlatformId.MACINTOSH) {
				return false; }

			if (   platformID == PlatformId.MICROSOFT 
					&& currentPlatformID == PlatformId.MACINTOSH) {
				currentSelection = name;
				currentPlatformID = platformID;
				currentEncoding = encoding;
				currentLanguage = language;
				return false; }

			// both are microsoft
			if (currentPlatformID == PlatformId.MICROSOFT) {
				if (pickMSName(encoding, language)) {
					currentSelection = name;
					currentPlatformID = platformID;
					currentEncoding = encoding;
					currentLanguage = language; }
				return false; }

			if (pickMacName(encoding)) {
				currentSelection = name;
				currentPlatformID = platformID;
				currentEncoding = encoding;
				currentLanguage = language; }

			return false;
		}

		String getSelection () {
			return currentSelection;
		}
	}


	static class SinglePostscriptName extends SingleNameSelector {

		public boolean nameFound(String name, int nameID, int platformID, int encoding, int language) {
			// choice 1 = microsoft american
			// choice 2 = macintosh roman
			// choice 3 = first microsoft
			// choice 4 = first macintosh

			if (name == null || name.length() == 0 || name.charAt(0) == 0)
				return false;

			if (currentPlatformID == -1) {
				currentSelection = name;
				currentPlatformID = platformID;
				currentEncoding = encoding;
				currentLanguage = language;
				return false; }

			// prefer microsoft american names over macintosh names
			// prefer microsoft names over nonRoman macintosh names
			// prefer macintosh roman names over microsoft non-american names
			if (   (   currentPlatformID == PlatformId.MICROSOFT 
					&& platformID == PlatformId.MACINTOSH
					&& (   currentLanguage == MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode()
							|| encoding != MacintoshEncodingId.ROMAN)) 
							|| (   currentPlatformID == PlatformId.MACINTOSH
									&& platformID == PlatformId.MICROSOFT
									&& language != MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode()
									&& currentEncoding == MacintoshEncodingId.ROMAN)) {
				return false; }

			else if (currentPlatformID != platformID) {
				currentSelection = name;
				currentPlatformID = platformID;
				currentEncoding = encoding;
				currentLanguage = language;
				return false; }

			// both are microsoft
			if (currentPlatformID == PlatformId.MICROSOFT) {
				if (pickMSLanguage(language)) {
					currentSelection = name;
					currentPlatformID = platformID;
					currentEncoding = encoding;
					currentLanguage = language; }
				return false; }

			if (pickMacName(encoding)) {
				currentSelection = name;
				currentPlatformID = platformID;
				currentEncoding = encoding;
				currentLanguage = language; }
			return false;
		}

		String getSelection() {
			return currentSelection;
		}
	}

	private class OTXDCFontDescription 
	extends com.adobe.fontengine.font.XDCFontDescription {

		public int getGlyphCid(int glyphID)
		throws UnsupportedFontException, InvalidFontException {
			return OpenTypeFont.this.getGlyphCid(glyphID);
		}

		public String getPostscriptName() 
		throws InvalidFontException, UnsupportedFontException {

			if (getBase14Name() != null)
				return getBase14Name();
			if (name != null)
			{
				SinglePostscriptName s = new SinglePostscriptName();
				name.enumerateNames(s, Name.PredefinedNames.POSTSCRIPT_NAME);
				if (s.getSelection() != null)
					return s.getSelection();
			}
			if (cff!= null)
			{
				PostscriptFontDescription[] descs = cff.getCFFFont().getPostscriptFontDescription();
				if (descs.length != 0)
					return descs[0].getPSName();
			}

			return null;
		}

		public String getFontFamily()
		throws InvalidFontException, UnsupportedFontException {
			if (name == null)
				return null;
			SingleFamilyName s = new SingleFamilyName();
			name.enumerateNames(s, name.selectFamilyNameId());
			return s.getSelection();
		}

		public int getNumGlyphs ()
		throws UnsupportedFontException, InvalidFontException {
			return OpenTypeFont.this.getNumGlyphs ();
		}

		public double getAdvance(int glyphID)
		throws InvalidFontException, UnsupportedFontException {
			return getHorizontalAdvance (glyphID) * 1000.0d / getUnitsPerEmX ();
		}

		public Rect getFontBBox() 
		throws InvalidFontException, UnsupportedFontException {
			if (head == null) {
				return null;}

			Rect b = head.getFontBBox ();
			int unitsPerEm = head.getUnitsPerEm ();
			return new Rect (b.xmin * 1000.0d / unitsPerEm,
					b.ymin * 1000.0d / unitsPerEm,
					b.xmax * 1000.0d / unitsPerEm,
					b.ymax * 1000.0d / unitsPerEm);
		}

		public double getCapHeight () 
		throws UnsupportedFontException, InvalidFontException {
			double capHeight = getCoolTypeCapHeight ();
			if (Double.isNaN (capHeight)) {
				return 0; }
			else {
				return capHeight * (1000.0d / getUnitsPerEmY ()); }
		}

		public double getXHeight() throws UnsupportedFontException, InvalidFontException
		{
			double xHeight = getCoolTypeXHeight();
			if (Double.isNaN(xHeight))
				return 0;
			else
				return xHeight * (1000.0d/getUnitsPerEmY());

		}

		public double getItalicAngle() 
		throws InvalidFontException {
			double angle = 0;

			if (hhea == null)
				return 0;

			int rise = hhea.getRise();
			int run = hhea.getRun();

			if (rise != 0 && run != 0) {
				angle = (Math.atan( -run / (double)rise ) / Math.PI) * 180;
				if ( run < 0 ) {
					angle += 180; }}

			return angle; 
		}

		public ROS getROS() 
		throws UnsupportedFontException, InvalidFontException { 

			if (cff != null) {
				return OpenTypeFont.this.getROS(); }
			else {
				return null; }
		}

		public boolean pdfFontIsTrueType() { 
			return cff == null;
		}

		private double calculateTTStemV () 
		throws UnsupportedFontException, InvalidFontException {
			StemFinder finder = new StemFinder(true, true);
			TTParser parser = new TTParser();
			int gidh = getGlyphForChar('h');
			int gidl = getGlyphForChar('l');
			int gid1 = getGlyphForChar('1');
			int gid4 = getGlyphForChar('4');
			double stemComputed = 0;
			double italicAngle = getItalicAngle();

			if (gidl != 0) {
				finder.reset();
				parser.parse(OpenTypeFont.this, gidl, finder);
				stemComputed = finder.getComputedStem(getHorizontalAdvance(gidl), 
						italicAngle); }

			if (gidh != 0) {
				double control;
				finder.reset();
				parser.parse(OpenTypeFont.this, gidh, finder);
				control = finder.getComputedStem(getHorizontalAdvance(gidh), italicAngle);
				if (   stemComputed == 0 
						|| (   control < stemComputed 
								&& control != 0 
								&& (stemComputed - control) > 20
								&& (stemComputed - control) < 70)) {
					stemComputed = control; }}

			if (stemComputed == 0 && gid1 != 0) {
				finder.reset();
				parser.parse(OpenTypeFont.this, gid1, finder);
				stemComputed = finder.getComputedStem(getHorizontalAdvance(gid1), 
						italicAngle);

				if (stemComputed == 0 && gid4 != 0) {
					finder.reset();
					parser.parse(OpenTypeFont.this, gid4, finder);
					stemComputed = finder.getComputedStem(getHorizontalAdvance(gid4), 
							italicAngle); } 
				else if (gid4 != 0) {
					double control;
					finder.reset();
					parser.parse(OpenTypeFont.this, gid4, finder);
					control = finder.getComputedStem(getHorizontalAdvance(gid4),
							italicAngle);
					if (control != 0 && (stemComputed - control) > 50) {
						stemComputed = control; }}}

			else if (gid1 != 0) {
				double control;
				finder.reset();
				parser.parse(OpenTypeFont.this, gid1, finder);
				control = finder.getComputedStem(getHorizontalAdvance(gid1),
						italicAngle);
				if (control != 0 && (stemComputed - control) > 50) {
					stemComputed = control; }}

			return stemComputed;
		}

		public double getStemV() 
		throws InvalidFontException, UnsupportedFontException{

			if (cff != null) {
				int glyphID;
				glyphID = getGlyphForChar ('l');

				if (glyphID == 0) {
					glyphID = getGlyphForChar ('I'); }

				if (glyphID == 0) {
					return 0; }

				return cff.getCFFFont().getStemVForGlyph(glyphID); }

			else {
				return calculateTTStemV(); }
		}

		public String getBase14Name() {
			return base14PSName;
		}

		public String getGlyphName(int gid) 
		throws InvalidFontException, UnsupportedFontException {
			return OpenTypeFont.this.getGlyphName(gid); 
		}

		public boolean isSerifFont() throws InvalidFontException, UnsupportedFontException {
			int gid1, gid2;
			gid1 = OpenTypeFont.this.getGlyphForChar('l');
			gid2 = OpenTypeFont.this.getGlyphForChar('I');
			return OpenTypeFont.this.isSerifFont(gid1, gid2, getItalicAngle());
		}

		public boolean isSmallCapFont() 
		throws InvalidFontException, UnsupportedFontException {
			if (!isAllCapFont()) {
				int gid1 = OpenTypeFont.this.getGlyphForChar('h');
				double xHeight;

				int gid2 = OpenTypeFont.this.getGlyphForChar('x');
				if (gid2 != 0) {
					xHeight = OpenTypeFont.this.getGlyphBBox(gid2).ymax; }
				else if (os2 != null) {
					xHeight = os2.getxHeight(); } 
				else {
					return false; }

				return OpenTypeFont.this.isSmallCapFont(gid1, xHeight); }

			return false;
		}

		public boolean isAllCapFont() throws InvalidFontException, UnsupportedFontException {
			int gid1 = OpenTypeFont.this.getGlyphForChar('K');
			int gid2 = OpenTypeFont.this.getGlyphForChar('k');
			return OpenTypeFont.this.isAllCapFont(gid1, gid2);
		}

		public int getCIDCount()
		{
			return OpenTypeFont.this.getCIDCount();
		}

		public void subsetAndStream(Subset subset, OutputStream out, boolean preserveROS)
		throws InvalidFontException, UnsupportedFontException, IOException
		{
			OpenTypeFont.this.subsetAndStream(subset, out, preserveROS);
		}

		public void subsetAndStream(SubsetSimpleType1 t1Subset, OutputStream out)
		throws InvalidFontException, UnsupportedFontException, IOException
		{
			if (cff == null || !(cff.getCFFFont() instanceof NameKeyedFont))
				throw new UnsupportedFontException("Not a name-keyed font");
			NameKeyedFont nk = (NameKeyedFont)cff.getCFFFont();
			NameKeyedSubset subset = (NameKeyedSubset)nk.createSubset();
			subset.addGlyphs(nk, t1Subset.getGlyphNames());
			nk.subsetAndStream(subset, out, (os2 != null) ? new Integer(os2.getRawFSType()) : null, true, null);
		}

		public void subsetAndStream(SubsetSimpleTrueType ttSubset, OutputStream out)
		throws InvalidFontException, UnsupportedFontException, IOException
		{
			if (cff != null)
				throw new UnsupportedFontException("Not a TrueType font");
			if (cmap == null)
				throw new UnsupportedFontException("Font contains no cmaps");
			int platformID = ttSubset.getPlatformID();
			int platSpecID = ttSubset.getPlatformSpecificID();
			int index = cmap.offsetToIndex(cmap.probe(platformID, platSpecID));
			if (index < 0)
				throw new InvalidFontException("Invalid cmap ID");
			Subset subset = createSubset();
			int[] cps = ttSubset.getCodePoints();
			for (int i = 0; i < cps.length; i++) {
				int gid = cmap.char2glyph(cps[i], index);
				if (gid >= getNumGlyphs())
					throw new InvalidFontException("GID greater than glyph count");
				if (gid <= 0)
					throw new InvalidFontException("Font does not contain required codepoint");
				subset.getSubsetGid(gid);
			}
			String[] postNames = ttSubset.getPostNames();
			if (postNames != null) {
				for (int i = 0; i < postNames.length; i++) {
					String glyphName = postNames[i];
					if (glyphName != null) {
						int gid = 0;
						if (post != null)
							gid = post.glyphName2gid(glyphName);
						if (gid >= getNumGlyphs())
							throw new InvalidFontException("GID greater than glyph count");
						if (gid <= 0 && !glyphName.equals(".notdef"))
							throw new InvalidFontException("Font does not contain required codepoint");
						subset.getSubsetGid(gid);
					}
				}
			}
			OpenTypeFont.this.subsetAndStream(subset, ttSubset, out, false);
		}

		public CodePage[] getXDCCodePages()
		throws InvalidFontException, UnsupportedFontException
		{
			HashSet codePageSet = new HashSet();
			CoolTypeScript script = getCoolTypeScript();
			if (script == CoolTypeScript.ROMAN) {
				if (cmap != null) {
					int index = getCoolTypeUnicodeCmapIndex();
					if (index != -1) {
						if (cmap.char2glyph('A', index) != 0)
							codePageSet.add(CodePage.ROMAN1);
					}
				}
			} else if (script == CoolTypeScript.EAST_EUROPEAN_ROMAN)
				codePageSet.add(CodePage.ROMAN2);
			else if (script == CoolTypeScript.JAPANESE)
				codePageSet.add(CodePage.JAPANESE);
			else if (script == CoolTypeScript.KOREAN)
				codePageSet.add(CodePage.KOREAN);
			else if (script == CoolTypeScript.SIMPLIFIED_CHINESE)
				codePageSet.add(CodePage.SIMPLIFIED_CHINESE);
			else if (script == CoolTypeScript.TRADITIONAL_CHINESE)
				codePageSet.add(CodePage.TRADITIONAL_CHINESE);
			if (os2 != null && os2.getTableVersion() > 0) {
				if (os2.supportsCodePage(Os2.CodePage.CP_1252_LATIN_1))
					codePageSet.add(CodePage.ROMAN1);
				if (os2.supportsCodePage(Os2.CodePage.CP_1250_LATIN_2_EE))
					codePageSet.add(CodePage.ROMAN2);
				if (os2.supportsCodePage(Os2.CodePage.CP_932_JAPANESE))
					codePageSet.add(CodePage.JAPANESE);
				if (os2.supportsCodePage(Os2.CodePage.CP_949_KOREAN_WANSUNG))
					codePageSet.add(CodePage.KOREAN);
				if (os2.supportsCodePage(Os2.CodePage.CP_936_CHINESE_SIMPLIFIED))
					codePageSet.add(CodePage.SIMPLIFIED_CHINESE);
				if (os2.supportsCodePage(Os2.CodePage.CP_950_CHINESE_TRADITIONAL))
					codePageSet.add(CodePage.TRADITIONAL_CHINESE);
			} else {
				if (cff != null) {
					CodePage[] cffList = cff.getCFFFont().getXDCFontDescription(null).getXDCCodePages();
					for (int i = 0; i < cffList.length; i++) {
						codePageSet.add(cffList[i]);
					}
				}
				if (cmap != null) {
					int index = getCoolTypeUnicodeCmapIndex();
					if (index >= 0) {
						if (cmap.char2glyph(ScriptHeuristicChars.ROMAN3, index) != 0)
							codePageSet.add(CodePage.ROMAN1);
						if (cmap.char2glyph(ScriptHeuristicChars.EASTERNEUROPEAN1, index) != 0 || cmap.char2glyph(ScriptHeuristicChars.EASTERNEUROPEAN2, index) != 0)
							codePageSet.add(CodePage.ROMAN2);
						if (cmap.char2glyph(ScriptHeuristicChars.JAPANESE, index) != 0)
							codePageSet.add(CodePage.JAPANESE);
					}
				}
			}
			CodePage[] retVal = new CodePage[codePageSet.size()];
			Iterator iter = codePageSet.iterator();
			int index = 0;
			while (iter.hasNext())
				retVal[index++] = (CodePage)iter.next();
			return retVal;
		}

		public void stream(OutputStream out, boolean openTypeFontsAllowed) 
		throws InvalidFontException, UnsupportedFontException, IOException {
			if (openTypeFontsAllowed || cff == null)
			{
				OpenTypeFont.this.streamSFNTForPDFEditting(out);
			} else 
			{
				cff.getCFFFont().stream(out, (os2 != null) ? new Integer(os2.getRawFSType()) : null);
			}

		}
	}

	public SWFFontDescription getSWFFontDescription (boolean wasEmbedded) 
	throws UnsupportedFontException, InvalidFontException {
		return new OTFSWFFont3Description(this, wasEmbedded);
	}

	public SWFFont4Description getSWFFont4Description(boolean wasEmbedded) 
	throws UnsupportedFontException, InvalidFontException {
		return new OTFSWFFont4Description(this, wasEmbedded);
	}

	public PDFFontDescription getPDFFontDescription(Font font) {
		return xdcDescription;
	}

	public XDCFontDescription getXDCFontDescription(Font font) {
		return xdcDescription;
	}

	public Permission getEmbeddingPermission (boolean wasEmbedded)
	throws InvalidFontException, UnsupportedFontException {

		if (os2 == null) {
			if (cff != null) {
				return cff.getEmbeddingPermission (wasEmbedded); }
			else {
				return EmbeddingPermission.getTrueTypeDefaultPermission (); }}
		else {
			return os2.getEmbeddingPermission (); }
	}

	public Subset createSubset ()
	throws UnsupportedFontException, InvalidFontException {
		return new OTSubset (this, cff != null);
	}

	public void subsetAndStream(Subset subset, OutputStream out, boolean preserveROS)
	throws InvalidFontException, UnsupportedFontException, IOException
	{
		subsetAndStream(subset, null, out, preserveROS);
	}

	private void streamTablesForEditting(OutputStream out, Map tables) 
	throws InvalidFontException, UnsupportedFontException, IOException
	{
		if (hmtx != null) 
			hmtx.stream(tables);

		if (hhea != null) 
			hhea.stream(tables);

		if (os2 != null) {
			os2.stream(tables);
		}

		if (gsub != null) {
			gsub.stream(tables);
		}

		if (gpos != null) {
			gpos.stream(tables);
		}

		if (gdef != null) {
			gdef.stream(tables);
		}

		if (base != null) {
			base.stream(tables);
		}

		if (cff != null) {
			cff.stream(tables, null);  // we are keeping the OS/2. The fsType will be there, not in the cff.
		}

	}

	void streamSFNTForPDFEditting(OutputStream out) throws InvalidFontException, UnsupportedFontException, IOException
	{
		Map tables = new TreeMap ();

		streamTablesForEditting(out, tables);


		if (name != null) {
			/*
			 * Commenting the below code as of now, 
			 * since it returns namesIDS as null for most fonts
			int[] nameIds;
			OpticalSizeData data = getOpticalSizeData();
			if (data == null){
				nameIds = null;
			} else {
				nameIds = new int[]{data.nameId};
			}
			*/
			name.stream(tables, true, new int []{Name.PredefinedNames.FONT_FAMILY, Name.PredefinedNames.POSTSCRIPT_NAME}); }

		if (cmap != null) {
			cmap.stream(tables); }

		if (maxp != null) {
			maxp.stream(tables);
		}

		if (glyf != null) {
			glyf.stream(tables);
		} 

		if (cvt != null) {
			cvt.stream(tables); }

		if (fpgm != null) {
			fpgm.stream(tables); }

		if (prep != null) {
			prep.stream(tables); }

		if (post != null) {
			post.stream(tables); }

		OTByteArrayBuilder headData = null;
		if (head != null)
		{
			headData = head.stream(tables);
		}

		finalizeStreamData(headData, tables, out, glyf != null);
	}

	void streamSFNTForSWFEditting(OutputStream out) throws UnsupportedFontException, InvalidFontException, IOException
	{
		Map tables = new TreeMap ();

		streamTablesForEditting(out, tables);


		if (name != null) {
			int[] nameIds;
			OpticalSizeData data = getOpticalSizeData();
			if (data == null){
				nameIds = null;
			} else {
				nameIds = new int[]{data.nameId};
			}

			name.stream(tables, false, nameIds); }

		if (cmap != null)
		{
			cmap.streamForSWF(tables, this.getNumGlyphs()); 
		}

		if (maxp != null)
		{
			if (glyf != null)
			{
				maxp.subsetAndStreamForCFF(this.getNumGlyphs(), tables);
			} else {
				maxp.stream(tables);
			}
		}

		if (glyf != null) {
			glyf.streamForCFF(this, tables, true);
		} 

		if (post != null) {
			if (glyf != null) {
				post.subsetAndStreamForCFF(tables);
			}
			else {
				post.stream(tables); 
			}
		}

		if (vorg != null) {
			vorg.stream(tables);		
		} else if (vmtx != null)
		{
			Subset subset = new SubsetDefaultImpl(getNumGlyphs(), false);
			Vorg.subsetAndStream(subset, tables, new NoVORGVertOrigFetcher(this));
		}

		if (kern != null) {
			kern.stream(tables);
		}

		if (vhea != null) {
			vhea.stream(tables);
		}

		if (vmtx != null) {
			vmtx.stream(tables);
		}

		OTByteArrayBuilder headData = null;
		if (head != null) {
			headData = head.stream(tables);
		}

		finalizeStreamData(headData, tables, out, false);
	}

	private void finalizeStreamData(OTByteArrayBuilder headData, Map tables, OutputStream out, boolean useTTVersion) 
	throws InvalidFontException, IOException
	{
		int headerSize = 12 + 16 * tables.size ();
		int searchRange = 1;
		int entrySelector = 0;
		while (searchRange <= tables.size ()) {
			searchRange *= 2;
			entrySelector++; }
		searchRange *= 8;
		entrySelector--;
		int rangeShift = tables.size () * 16 - searchRange;

		OTByteArrayBuilder headerBuilder = OTByteArray.getOTByteArrayBuilderInstance(headerSize);

		if (useTTVersion)
			headerBuilder.setFixed (0, 1, 0);
		else
			headerBuilder.setFixed(0, 0x4f54, 0x544f); /*OTTO*/
		headerBuilder.setuint16 (4, tables.size ());
		headerBuilder.setuint16 (6, searchRange);
		headerBuilder.setuint16 (8, entrySelector);
		headerBuilder.setuint16 (10, rangeShift);

		long offset = headerSize;
		int headerOffset = 12;
		long fileChecksum = 0;

		if (head != null)
			Head.clearChecksumAdjust (headData);

		for (Iterator it = tables.keySet ().iterator (); it.hasNext (); ) {
			Integer tag = (Integer) it.next ();
			OTByteArrayBuilder data = (OTByteArrayBuilder) tables.get (tag);
			long tableChecksum = data.checksum (0, data.getSize(), 0);

			fileChecksum = (fileChecksum + tableChecksum) & 0xffffffffL;

			headerBuilder.setuint32 (headerOffset,      tag.intValue ());
			headerBuilder.setuint32 (headerOffset + 4,  (int)tableChecksum);
			headerBuilder.setuint32 (headerOffset + 8,  (int)offset);
			headerBuilder.setuint32 (headerOffset + 12, data.getSize());
			headerOffset += 16;

			offset += data.getSize();
			if (data.getSize() % 4 != 0) {
				offset += 4 - (data.getSize() % 4); }}

		fileChecksum =   (fileChecksum + 
				headerBuilder.checksum (0, headerBuilder.getSize(), 0)) 
				& 0xffffffffL;

		long checksumAdjust = (0xb1b0afbaL - fileChecksum) & 0xffffffffL;
		Head.setChecksumAdjust (headData, checksumAdjust);

		OTByteArray header = headerBuilder.toOTByteArray();
		header.write(out);

		for (Iterator it = tables.keySet ().iterator (); it.hasNext ();) {
			Integer tag = (Integer) it.next ();
			OTByteArrayBuilder builder = (OTByteArrayBuilder) tables.get (tag);
			OTByteArray data = builder.toOTByteArray();
			data.write(out);

			if (data.getSize() % 4 != 0) {
				int pad = ((data.getSize() / 4) + 1) * 4 - data.getSize(); 
				for (int k = 0; k < pad; k++) {
					out.write (0); }}}
	}

	public void subsetAndStreamForSWFEmbedding(Subset subset, OutputStream out, 
			TreeMap<Integer /*lookup index*/, List /*subtables*/> gsubLookups, TreeSet cmapData, boolean includeVariationCmap) 
	throws InvalidFontException, UnsupportedFontException, IOException
	{
		Map tables = new TreeMap ();
		TreeMap gposLookups = new TreeMap();

		if (hmtx != null) {
			hmtx.subsetAndStream (subset, tables); }

		if (hhea != null) {
			hhea.subsetAndStream (subset, tables); }

		if (os2 != null) {
			os2.subsetAndStream (subset, tables); }

		if (gsub != null) {
			gsub.subsetAndStream(subset, gsubLookups, tables, getNumGlyphs()); 
		}

		// We subset the kern table before we subset the gpos so that we only preserve
		// newly empty kern features if we preserve the kern table. 
		if (kern != null) {
			kern.subsetAndStreamForSWF(subset, tables);
		}

		if (gpos != null){
			GposHarvester harvester = new GposHarvester(gpos, getNumGlyphs());
			gposLookups = harvester.gatherPossibleLookups(subset);
			gpos.subsetAndStream(subset, gposLookups, tables, getNumGlyphs(), tables.containsKey(new Integer (Tag.table_kern)));
		}


		if (gdef != null) {
			gdef.subsetAndStream(subset, getNumGlyphs(), tables);
		}

		if (base != null) {
			base.subsetAndStream(subset, tables);
		}

		if (testing_subsetAndStreamForSWFEmbedding) {
			class SBST extends Table {
			  SBST (Subset gidMap, TreeMap gsubMap, TreeMap gposMap) 
			  throws UnsupportedFontException, InvalidFontException, java.io.IOException {
				super();
				int size = 6;
				int gsubSize = sizeNeeded(gsubMap);
				int gposSize = sizeNeeded(gposMap);
				// offset, count, array
				int gidSize = 2 + 2 + gidMap.getNumGlyphs() * 2;
				
				OTByteArrayBuilder builder = OTByteArray.getOTByteArrayBuilderInstance(size + gposSize + gsubSize +gidSize);
				builder.setuint16(0, 6);
				builder.setuint16(2, 6+gsubSize);
				builder.setuint16(4, 6+gsubSize+gposSize);

				writeMapping(builder, 6, gsubMap);
				writeMapping(builder, 6+gsubSize, gposMap);
				writeMapping(builder, 6+gsubSize+gposSize, gidMap);

				data = builder.toOTByteArray();
			  }

			  private void writeMapping(OTByteArrayBuilder builder, int offset, TreeMap mapping)
			  {
				if (mapping == null)
					return;
				Set keys = mapping.keySet();
				Iterator iter = keys.iterator();

				builder.setuint16(offset, mapping.size()); offset +=2;
				while (iter.hasNext())
				{
					Integer key = (Integer)iter.next();

					// index 
					builder.setuint16(offset, key); offset +=2;

					List list = (List)mapping.get(key);

					// subtable count
					builder.setuint16(offset, list.size()); offset +=2;
					for (int j=0; j<list.size() ; j++) {
						builder.setuint16(offset, (Integer)list.get(j)); offset +=2;
					}
				}
			  }
			  private void writeMapping(OTByteArrayBuilder builder, int offset, Subset mapping)
			  {
				  int count = mapping.getNumGlyphs();
				  builder.setuint16(offset, count);
				  for (int i=0; i<count ; i++) {
					  builder.setuint16(offset+2 + i*2, mapping.getFullGid(i));
				  }
			  }

			  private int sizeNeeded(TreeMap tbl)
			  {
				if (tbl == null)
					return 0;
				Set keys = tbl.keySet();
				Iterator iter = keys.iterator();
				int size = 1;

				while (iter.hasNext())
				{
					Integer key = (Integer)iter.next();
					List list = (List)tbl.get(key);

					size += 2; // index , count
					size += list.size();
				}
				return size*2;
			  }

			  public void stream(Map tables) {
				  OTByteArrayBuilder newData = this.getDataAsByteArray();
				  tables.put (new Integer (Tag.table_SBST), newData);
			  }
			}

			SBST sbst = new SBST(subset, gsubLookups, gposLookups);
			sbst.stream(tables);
		}

		if (cff != null) { 
			cff.subsetAndStream(subset, tables, false, null, true); 
		} else {
			if (glyf != null) {
				glyf.subsetAndStreamForCFF(subset, this, tables, true); //converting tt to cff
			} else {
				throw new InvalidFontException("Either cff or glyf must be present");
			}
		}

		if (vmtx != null) {
			Vorg.subsetAndStream(subset, tables, 
					vorg != null ? (VerticalOriginFetcher)vorg: new NoVORGVertOrigFetcher(this)); 
		}

		if (cmap != null)
		{
			cmap.subsetAndStreamForSWF(cmapData, subset, tables, includeVariationCmap);
		}


		if (name != null) {
			name.subsetAndStream(subset, false, new int[] {Name.PredefinedNames.FONT_FAMILY, Name.PredefinedNames.POSTSCRIPT_NAME}, tables); }


		if (maxp != null) {
			if (cff != null) {
				maxp.subsetAndStream(subset, tables); 
			} else {
				maxp.subsetAndStreamForCFF(subset.getNumGlyphs(), tables); 
			}
		}

		if (post != null) {
			if (cff != null) {
				post.stream(tables);
			} else
				post.subsetAndStreamForCFF(tables); 
		}

		if (vhea != null) {
			vhea.subsetAndStream(subset, tables); 
		}

		if (vmtx != null) {
			vmtx.subsetAndStream (subset, tables); } 

		OTByteArrayBuilder headData = null;
		if (head != null)
			headData = head.subsetAndStream (subset, tables);

		finalizeStreamData(headData, tables, out, false);
	}

	/** Subset and stream this font for PDF use. 
	 * The stream is either a TrueType stream or a CID-keyed CFF stream.
	 * @param out the OutputStream to which the bytes are streamed
	 */
	private void subsetAndStream(Subset subset, SubsetSimpleTrueType ttSubset, OutputStream out, boolean preserveROS)
	throws InvalidFontException, UnsupportedFontException, IOException
	{

		if (cff != null) {
			cff.subsetAndStream (subset, out, preserveROS, 
					os2 != null ? new Integer(os2.getRawFSType()): null);
			return; }

		if (glyf == null) {
			throw new InvalidFontException("cff or glyf table required"); }

		Map tables = new TreeMap ();

		if (hmtx != null) {
			hmtx.subsetAndStream (subset, tables); }

		if (hhea != null) {
			hhea.subsetAndStream (subset, tables); }

		if (maxp != null) {
			maxp.subsetAndStream (subset, tables); }

		if (os2 != null) {
			os2.subsetAndStream (subset, tables); }

		glyf.subsetAndStream (subset, tables);

		if (name != null) {
			name.subsetAndStream(subset, true, new int [] {Name.PredefinedNames.FONT_FAMILY, Name.PredefinedNames.POSTSCRIPT_NAME}, tables); }

		if (cmap != null) {
			cmap.subsetAndStream(subset, ttSubset, tables); }

		if (post != null) {
			post.subsetAndStream(subset, ttSubset, tables); }

		if (cvt != null) {
			cvt.subsetAndStream (subset, tables); }
		if (fpgm != null) {
			fpgm.subsetAndStream (subset, tables); }
		if (prep != null) {
			prep.subsetAndStream (subset, tables); }

		OTByteArrayBuilder headData = null;
		if (head != null)
			headData = head.subsetAndStream (subset, tables);

		finalizeStreamData(headData, tables, out, true);

		return;
	}

	public CacheSupportInfo getCacheSupportInfo()
		throws InvalidFontException, UnsupportedFontException
	{
		return new CacheSupportInfo(getClass().getSimpleName(), getNumGlyphs(), cff != null);
	}

	//--------------------------------------------------------- PS properties ---
	public PostscriptFontDescription[] getPostscriptFontDescription ()
	throws InvalidFontException, UnsupportedFontException {
		if (base14PSName != null)
		{
			PostscriptFontDescription[] retVal = new PostscriptFontDescription[1];
			retVal[0] = new PostscriptFontDescription(base14PSName);
			return retVal;
		}

		if (name == null) {
			if (cff != null)
				return this.cff.cffFont.getPostscriptFontDescription();
			return new PostscriptFontDescription[0];
		}

		Set s = name.getPostscriptNames();
		int numEntries = s.size();

		PostscriptFontDescription[] retVal = new PostscriptFontDescription[numEntries];
		Iterator iter = s.iterator();

		for (int i = 0; i < numEntries; i++) {
			retVal[i] = new PostscriptFontDescription((String)iter.next()); }
		return retVal;
	}

	//--------------------------------------------------------- CSS properties ---

	/** Return the set of CSS font-family names that this font matches. */
	public Set /*<String>*/ getCSSFamilyNames ()
	throws InvalidFontException, UnsupportedFontException {
		if (base14CSSName != null) {
			Set/*<String>*/ s = new HashSet/*<String>*/ ();
			s.add (base14CSSName); 
			return s; }
		else if (name != null) {      
			return name.getCSSFamilyNames (); }
		else {
			return new HashSet/*<String>*/ (); }
	}

	public String getPreferredCSSFamilyName()
	throws InvalidFontException, UnsupportedFontException {
		if (base14CSSName != null)
			return base14CSSName;

		if (name != null)
			return name.getPreferredCSSFamilyName();

		return null;
	}


	/** Tell if the font matches the CSS font-style normal. */
	public boolean isCSSStyleNormal ()
	throws InvalidFontException, UnsupportedFontException {
		// If the feature 'ital' is present, we assume that applying it
		// gives italic glyphs, and not applying it give normal glyphs
		return    os2 == null 
		|| (os2.getSelection () & Os2.SelectionBits.italic) == 0
		|| gsub != null && gsub.featureIsPresent (Tag.feature_ital);
	}

	/** Tell if the font matches the CSS font-style italic. */
	public boolean isCSSStyleItalic ()
	throws InvalidFontException, UnsupportedFontException {
		// If the feature 'ital' is present, we assume that applying it
		// gives italic glyphs, and not applying it give normal glyphs
		return    (   os2 != null  
				&& (os2.getSelection () & Os2.SelectionBits.italic) != 0)
				|| gsub != null && gsub.featureIsPresent (Tag.feature_ital);
	}

	/** Tell if the font matches the CSS font-style oblique. */
	public boolean isCSSStyleOblique ()
	throws InvalidFontException, UnsupportedFontException {
		// For OpenType, we do not distinguish Oblique and Italic
		return isCSSStyleItalic ();
	}

	/** Tell if the font matches the CSS font-variant normal. */
	public boolean isCSSVariantNormal () {
		// An OpenType font is always a candidate: we simply do
		// not turn the 'scmp' feature on
		return true;
	}

	/** Tell if the font matches the CSS font-variant small-caps. */
	public boolean isCSSVariantSmallCaps () {
		// An OpenType font is always a candidate: we simply turn 
		// the 'smcp' feature on
		return true;
	}

	/** Return the CSS weight of this font. */
	public int getCSSWeight ()  throws InvalidFontException {
		if (os2 != null) {
			return os2.getWeightClass (); }
		return 400;
	}

	/** {@inheritDoc} */
	protected CSSStretchValue getCSSStretchValue ()  throws InvalidFontException {
		if (os2 != null) {
			switch (os2.getWidthClass()) {
			case 1: return CSSStretchValue.ULTRACONDENSED;
			case 2: return CSSStretchValue.EXTRACONDENSED;
			case 3: return CSSStretchValue.CONDENSED;
			case 4: return CSSStretchValue.SEMICONDENSED;
			case 5: return CSSStretchValue.NORMAL;
			case 6: return CSSStretchValue.SEMIEXPANDED;
			case 7: return CSSStretchValue.EXPANDED;
			case 8: return CSSStretchValue.EXTRAEXPANDED;
			case 9: return CSSStretchValue.ULTRAEXPANDED; }}

		return CSSStretchValue.NORMAL; 
	}

	//---------------------------------------------------------- FXG Properties ---
	/* (non-Javadoc)
	 * @see com.adobe.fontengine.font.FontData#getFXGFontDescription(com.adobe.fontengine.fontmanagement.Platform, com.adobe.agl.util.ULocale)
	 */
	public FXGFontDescription[] getFXGFontDescription(Platform platform, ULocale locale)
		throws InvalidFontException, UnsupportedFontException
	{
		List<FXGFontDescription> fxgDescriptions = new ArrayList<FXGFontDescription>();
		if (name != null) {
			Set macFamilyNames = null;
			if (platform == null || platform == Platform.MAC_OSX) {
				macFamilyNames = name.getMacFXGFamilyNames(locale);
			}
			Set winFamilyNames = null;
			if (((platform == null || platform == Platform.WINDOWS)) && os2 != null) {
				winFamilyNames = name.getWindowsFXGFamilyNames(locale);
			}
			if (macFamilyNames != null) {
				Iterator namesIter = macFamilyNames.iterator();
				while (namesIter.hasNext()) {
					Name.NameEntry name = (Name.NameEntry) namesIter.next();
					ULocale uLocale = Name.getLocaleForLocaleID(name.getPlatformID(), name.getLanguage());
					String familyName = name.getName();
					if (uLocale != null && familyName != null) {
						fxgDescriptions.add(new FXGFontDescription(Platform.MAC_OSX, uLocale, familyName,
											   false, false));
					}
				}
			}
			if (winFamilyNames != null) {
				Iterator namesIter = winFamilyNames.iterator();
				int selectionBits = os2.getSelection();
				while (namesIter.hasNext()) {
					Name.NameEntry name = (Name.NameEntry) namesIter.next();
					ULocale uLocale = Name.getLocaleForLocaleID(name.getPlatformID(), name.getLanguage());
					String familyName = name.getName();
					if (uLocale != null && familyName != null) {
						fxgDescriptions.add(new FXGFontDescription(Platform.WINDOWS, uLocale, familyName,
											   (selectionBits & Os2.SelectionBits.bold) > 0,
											   (selectionBits & Os2.SelectionBits.italic) > 0));
					}
				}
			}
		}
		if (fond != null) {
			ULocale fondLocale = fond.getLocale();
			if ((platform == Platform.MAC_OSX || platform == null) && Name.NameEntrySet.isContainedWithin(fondLocale, locale)) {
				FXGFontDescription fxgDesc = new FXGFontDescription(Platform.MAC_OSX, fondLocale, fond.getName(), fond.isBold(), fond.isItalic());
				String fxgDescStr = fxgDesc.toString();
				// Filter out duplicates
				for (FXGFontDescription desc : fxgDescriptions) {
					if (desc.toString().equals(fxgDescStr)) {
						fxgDesc = null;
						break;
					}
				}
				if (fxgDesc != null) {
					fxgDescriptions.add(fxgDesc);
				}
			}
		}
		return fxgDescriptions.toArray(new FXGFontDescription[fxgDescriptions.size()]);
	}

	//---------------------------------------------------------- Platform Properties ---
	public PlatformFontDescription[] getPlatformFontDescription(Platform platform, ULocale locale)
	throws InvalidFontException, UnsupportedFontException
	{
		if (this.name == null)
		{
			return new PlatformFontDescription[0];
		}
		Set macFamilyNames = null;
		if (platform == null || platform == Platform.MAC_OSX)
		{
			macFamilyNames = this.name.getMacPlatformNames(locale);
		}
		Set winFamilyNames = null;
		if (((platform == null || platform == Platform.WINDOWS)))
		{
			winFamilyNames = this.name.getWindowsPlatformNames(locale);
		}

		List<PlatformFontDescription> platformDescriptions = new ArrayList<PlatformFontDescription>();
		if (macFamilyNames != null)
		{
			Iterator namesIter = macFamilyNames.iterator();
			while (namesIter.hasNext())
			{
				Name.NameEntry name = (Name.NameEntry) namesIter.next();
				ULocale uLocale = Name.getLocaleForLocaleID(name.getPlatformID(), name.getLanguage());
				String familyName = name.getName();
				if (uLocale != null && familyName != null) {
					platformDescriptions.add(new PlatformFontDescription(
											     Platform.MAC_OSX, 
											     uLocale,
											     familyName));
				}
			}
		}

		if (winFamilyNames != null)
		{
			Iterator namesIter = winFamilyNames.iterator();
			while (namesIter.hasNext())
			{
				Name.NameEntry name = (Name.NameEntry) namesIter.next();
				ULocale uLocale = Name.getLocaleForLocaleID(name.getPlatformID(), name.getLanguage());
				String familyName = name.getName();
				if (uLocale != null && familyName != null) {
					platformDescriptions.add(new PlatformFontDescription(
											     Platform.WINDOWS, 
											     uLocale,
											     familyName));
				}
			}
		}
		return platformDescriptions.toArray(new PlatformFontDescription[platformDescriptions.size()]);
	}

	//---------------------------------------------------------- Optical size ---
	public OpticalSizeData getOpticalSizeData () throws InvalidFontException {

		if (gpos == null) {
			return null; }

		// Some fonts have been incorrectly built. We first try to interpret
		// them according to the spec, and validate the resulting values. If 
		// they are not plausible, we then interpret them taking into account
		// the bug.

		OpticalSizeData osd = gpos.getOpticalSizeData (/*compensateForFontBug*/ false);

		if (osd == null) {
			return null; }

		boolean plausible;
		if (osd.designSize == 0) {
			plausible = false; }
		else if (   osd.subfamilyId == 0
				&& osd.nameId == 0
				&& osd.minSize == 0
				&& osd.maxSize == 0) {
			plausible = true; }
		else if (   osd.designSize < osd.minSize
				|| osd.maxSize < osd.designSize
				|| osd.nameId < 256
				|| osd.nameId > 32767
				|| (name != null && name.getName (-1, osd.nameId) == null)) {
			plausible = false; }
		else {
			plausible = true; }

		if (! plausible) {
			osd = gpos.getOpticalSizeData (/*compensateForFontBug*/ true); }

		return osd;
	}

	/** Return the range of point sizes for which this font has been designed.
	 * 
	 * @return an array with exactly two elements. The first is the smallest
	 * intended point size (inclusive), the second is the largest 
	 * intended point size (exclusive). Both numbers are in decipoints.
	 */
	public double[] getPointSizeRange () throws InvalidFontException {
		OpticalSizeData osd = getOpticalSizeData ();
		if (osd == null || osd.subfamilyId == 0) {
			return new double[] {0d, Double.POSITIVE_INFINITY}; }
		else {
			return new double[] {osd.minSize, osd.maxSize}; }
	}

	//----------------------------------------------------------------------------

	private String getVersion () 
	throws InvalidFontException, UnsupportedFontException {
		String version = null;

		try { 
			if (name == null)
				return "";

			version = name.getName (3, 1, 0x409, 5);

			if (version == null) { // try the symbol one
				version = name.getName (3, 0, 0x409, 5); }}
		catch (UnsupportedEncodingException e) {
		/* ignore this name entry */ }

		if (version == null) {
			return ""; }

		int start = 0;
		while (   start < version.length ()
				&& (version.charAt (start) < '0' || '9' < version.charAt (start))) {
			start++; }

		int end = start;
		while (end < version.length ()
				&& '0' <= version.charAt (end)
				&& version.charAt (end) <= '9') {
			end++; }

		if (start == end) {
			// malformed, take the full version
			return version; }

		if (end < version.length () && '.' == version.charAt (end)) {
			end++;
			while (   end < version.length ()
					&& '0' <= version.charAt (end)
					&& version.charAt (end) <= '9') {
				end++; }}

		return version.substring (start, end);
	}

	static class NameConsumer implements Name.NameSelector {
		public String bestName;
		public int bestNameLanguage;

		public boolean nameFound (String name, int nameID, int platformID, 
				int encodingID, int languageID)
		throws InvalidFontException, UnsupportedFontException {
			return false;
		}
	}

	public CatalogDescription getSelectionDescription () 
	throws InvalidFontException, UnsupportedFontException {

		final Set/*<String>*/ names = new HashSet/*<String>*/ ();

		if (name != null) {
			NameConsumer ncWindows = new NameConsumer () {
				public boolean nameFound (String n, int nameID, int platformID, 
						int encodingId, int languageId) 
				throws InvalidFontException, UnsupportedFontException {
					String style = null;
					try {
						style = name.getName (platformID, encodingId, languageId, 
								Name.PredefinedNames.FONT_SUBFAMILY); }
					catch (UnsupportedEncodingException e) {
					/* ignore this style */ }

					if (style == null) {
						style = "Regular"; }
					n = n + ", " + style;

					if (bestName == null || languageId == Name.MicrosoftLCID.ENGLISH_UNITED_STATES.getLanguageCode()) 
					{
						bestName = n; 
						bestNameLanguage = languageId; }

					names.add (n); 
					return false;}};
					name.enumerateNames (ncWindows, Name.PredefinedNames.FONT_FAMILY);

					NameConsumer ncPreferred = new NameConsumer () {
						public boolean nameFound (String n, int nameID, int platformId, 
								int encodingId, int languageId) 
						throws InvalidFontException, UnsupportedFontException {
							String style = null;
							try {
								style = name.getName (platformId, encodingId, languageId,
										Name.PredefinedNames.PREFERRED_SUBFAMILY);
								if (style == null) {
									style = name.getName (platformId, encodingId, languageId, 
											Name.PredefinedNames.FONT_SUBFAMILY); }}
							catch (UnsupportedEncodingException e) {
							/* ignore this style */ }

							if (style == null) {
								style = "Regular"; }
							names.add (n + ", " + style);
							return false;}};
							name.enumerateNames (ncPreferred, Name.PredefinedNames.PREFERRED_FAMILY);

							String bestName = ncPreferred.bestName != null 
							? ncPreferred.bestName : ncWindows.bestName;

							return new CatalogDescription ("OT", names, bestName, getVersion ()); }

		else {
			names.add ("<no name table>");
			return new CatalogDescription ("OT", names, "<no best name>", 
			"<no version>"); }
	}

	public Rect getCoolTypeGlyphBBox(int glyphID) throws UnsupportedFontException, InvalidFontException {
		if (cff != null)
			return this.getGlyphBBox(glyphID);
		else {
			// CoolType returns calculated, hinted bboxes, where the hints are run at 1000ppem.			
			TrueTypeGlyphBBoxCalculator calculator = new TrueTypeGlyphBBoxCalculator(true, 1000);
			Rect bbox = calculator.calculateBBox(this, glyphID);
			Rect retVal = new Rect(bbox.xmin * this.getUnitsPerEmX()/1000, bbox.ymin * this.getUnitsPerEmX()/1000, 
					bbox.xmax * this.getUnitsPerEmX()/1000, bbox.ymax * this.getUnitsPerEmX()/1000);
			return retVal;
		}
	}

	public CFFFont getCFFFont()
	{
		if (cff == null)
			return null;
		return cff.getCFFFont();
	}
}
