/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons.jcr;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.nodetype.NodeDefinition;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/** Recursive Node removal utility */
public class JcrRecursiveRemove {
    private final Logger log = LoggerFactory.getLogger(getClass());
    public static final int DEFAULT_SAVE_EVERY_HOW_MANY = 1000;

    /** Remove Node n and its children recursively, saving every N nodes as specified
     * 	(to avoid growing the transient space too much)
     * 
     * 	@param n node to delete
     * 	@param saveEveryHowManyNodes if &lt; 1, DEFAULT_SAVE_EVERY_HOW_MANY is used
     * 	@return number of nodes deleted
     * 	@throws RepositoryException if some error occurs
     */
    public int removeRecursive(Node n, int saveEveryHowManyNodes) throws RepositoryException {
    	if(saveEveryHowManyNodes < 1) {
    		saveEveryHowManyNodes = DEFAULT_SAVE_EVERY_HOW_MANY;
    	}
    	final String path = n.getPath();
    	final int result = remove(n, n.getPath(), saveEveryHowManyNodes, 0);
		log.info("removeRecursive({}) done: {} nodes deleted, saving...", path, result);
		n.getSession().save();
		onSave();
		return result;
    }

	private int remove(Node n, String startPath, int saveEvery, int deletedCount) throws RepositoryException {
		if(!canContinue()) {
			log.info("canContinue() returns false, aborting recursive delete");
			return deletedCount;
		}
		
		// Remove child nodes
    	final NodeIterator ni = n.getNodes();
    	while(ni.hasNext()) {
    		deletedCount = remove(ni.nextNode(), 
    				startPath, saveEvery, deletedCount);
    	}
    	
    	// Remove this node, unless it is protected or mandatory - 
    	// that might create inconsistencies, such nodes will be deleted
    	// later when their parent is
    	final NodeDefinition def = n.getDefinition();
    	if(!def.isProtected() && !def.isMandatory()) {
    		n.remove();
    		deletedCount++;
    	}
    	
    	// Save if we reached saveEvery nodes deleted
    	if(deletedCount % saveEvery == 0) {
    		log.info("removeRecursive({}) in progress: {} nodes deleted, saving...", startPath, deletedCount);
    		n.getSession().save();
    		onSave();
    	}
    	
    	return deletedCount;
	}

	/** Called after each save in the remove loop, can be used for example to wait for observation events */
	protected void onSave() {
	}
	
	/**
	 * Can be overridden to stop an ongoing deletion
	 * @return true if ongoing deletion can continue, false otherwise.
	 */
	protected boolean canContinue() {
		return true;
	}
}
