/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.mcm.api;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.commons.json.JSONArray;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Filters maps by checking if each of a filter map's values is contained
 * in the keys of the filtered map. 
 * 
 */
public class MapFilter {
	
	private final static Logger log = LoggerFactory.getLogger(MapFilter.class);
	
	private Map<String, Object> filterMap;

	public MapFilter() {
		super();
	}
	
	public void setFilterMap(Map<String, Object> filterMap) {
		this.filterMap = filterMap;
	}
	
	public boolean filter(Map<String, Object> filteredMap) {
		boolean pass = true;
		if (this.filterMap != null) {
			for (String key: filterMap.keySet()) {
				
				Object oFilterVal = filterMap.get(key);
				Object oCandVal = filteredMap.get(key);
				try {
					if (oFilterVal instanceof String) {
						String filterVal = (String) oFilterVal;
						String candVal = (String) oCandVal;
						if (StringUtils.isEmpty(candVal) 
								|| !candVal.toUpperCase().contains(filterVal.toUpperCase())) {
							pass = false;
							break;
						}
					
					} else if (oFilterVal instanceof JSONObject) {
						
						JSONObject jsonFilterDef = (JSONObject) oFilterVal;
						try {
							String type = jsonFilterDef.getString("type");
							if ("listmember".equals(type)) {
								String searchedKey = jsonFilterDef.optString("key");
								String searchedVal = jsonFilterDef.getString("value");
								pass = false;
								if (oCandVal != null) {
									JSONArray array = (JSONArray) oCandVal;
									String searchedValUpper = searchedVal.toUpperCase();
									for (int i= 0; i < array.length(); i++) {
										Object listMember = array.get(i);
										// allow searching of String-only lists, too. key ignored then
										if (listMember instanceof String) {
											String listMemberString = (String) listMember;
											if (listMemberString.toUpperCase().contains(searchedValUpper)) {
												pass = true;
												break;
											}
										} else if (listMember instanceof JSONObject) {
											String memberVal = ((JSONObject) listMember).optString(searchedKey);
											if (memberVal != null && memberVal.toUpperCase().contains(searchedValUpper)) {
												pass = true;
												break;
											}
										}
									}
								}
							} else {
								throw new RuntimeException("Unknown filtering type: " + type);
							}
						} catch (JSONException je) {
							throw new RuntimeException("Problem with json filter definition '" 
									+ jsonFilterDef + "'.", je);
						}
					} else {
						if (oFilterVal != null) {
							log.error("Unknown filter value type: " + oFilterVal.getClass() 
									+ " val: '" + oFilterVal.toString() + "'. Ignored.");
						}
					}
				} catch (ClassCastException cce) {
					log.error("Cannot filter {} with filter type {}, values: {}, {}", new String[] {
						oCandVal == null ? "null" : oCandVal.getClass().toString(),
						oFilterVal.getClass().toString(),
						oCandVal == null? "null" : oCandVal.toString(),
						oFilterVal == null? "null" : oFilterVal.toString()
					});
				}
			}
		}
		return pass;
	}

	/**
	 * 
	 * @param filterParam
	 * @return number of added filtered keys
	 * @throws JSONException 
	 */
	public int ingestJson(String filterParam) throws JSONException {
		int retval = 0;
		if ( !StringUtils.isEmpty(filterParam)) {
			JSONObject filterJsonData = new JSONObject(filterParam);
			Iterator<String> keys = filterJsonData.keys();
			if (keys.hasNext() && this.filterMap == null) {
				this.filterMap = new HashMap<String, Object>();
			}
			while (keys.hasNext()) {
				String key = keys.next();
				Object value = filterJsonData.get(key);
				// just add everything, also possible JSONObjects:
				filterMap.put(key, value);
				retval++;
			}
    	}
		return retval;
	}
	
	@Override
	public String toString() {
		return "MapFilter with filter: {" + this.filterMap + "}";
	}
}
