/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.adobe.granite.workflow.exec;

import java.util.Date;
import java.util.Set;

import org.osgi.annotation.versioning.ProviderType;

/**
 * Defines an item which is available in the workflow inbox
 */
@ProviderType
public interface InboxItem {

    /**
     * The possible priority values for InboxItem.
     */
    enum Priority {
        /**
         * Low Priority.
         */
        LOW(-200),

        /**
         * Medium Priority
         */
        MEDIUM(0),

        /**
         * High Priority
         */
        HIGH(200);

        private final long value;

        private Priority(long value) {
            this.value = value;
        }

        /**
         * Get the long value for this priority.
         * @return the long equivalent for the priority.
         */
        public long getPriorityValue() {
            return this.value;
        }

        /**
         * Converts a long to its equivalent priority value.
         * The ranges are as follows:
         * LOW: {@code value <= -100}
         * MEDIUM: {@code -100 < value < 100}
         * HIGH: {@code value >= 100}
         * @param value the long value to convert to a Priority
         * @return Priority
         */
        public static final Priority getPriority(long value) {
            if (value <= -100) {
                return LOW;
            } else if (value >= 100) {
                return HIGH;
            }
            return MEDIUM;
        }
    }

    /**
     * Returns the unique identifier for this InboxItem
     * @return the ID
     */
    String getId();

    /**
     * Returns the current assignee, respectively the info in which
     * inbox the InboxItem "resides".
     *
     * @return current assignee
     */
    String getCurrentAssignee();

    /**
     * Returns the start time of the <code>InboxItem</code>.
     *
     * @return The start time of the <code>InboxItem</code>.
     */
    Date getTimeStarted();

    /**
     * Returns the end time of the <code>InboxItem</code>.
     *
     * @return The end time of the <code>InboxItem</code>.
     */
    Date getTimeEnded();

    /**
     * Returns the due time of the <code>InboxItem</code>.
     * 
     * @return The due time of the <code>InboxItem</code>
     */
    Date getDueTime();

    /**
     * Returns the progress begin time of the <code>InboxItem</code>
     *
     * @return The progress begin time of the <code>InboxItem</code>
     */
    Date getProgressBeginTime();

    /**
     * Returns the status of the <code>InboxItem</code>.
     *
     * @return the status of the InboxItem
     */
    Status getStatus();

    /**
     * Returns the type of this InboxItem.  Different implementations of InboxItem should return unique values
     * @return the item type
     */
    String getItemType();

    /**
     * Returns the subtype of this InboxItem.
     * @return the subtype of this item, or null if there is no subtype.
     */
    String getItemSubType();

    /**
     * Returns the path of the payload's content for this InboxItem if possible.  If the payload's content is not
     * stored as a path then null is returned.
     * @return the payload's content path or null.
     */
    String getContentPath();

    /**
     * Returns the priority of this InboxItem.
     * @return the priority of this InboxItem
     */
    Priority getPriority();
}
