/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.template.automation;

import com.adobe.xfa.Element;
import com.adobe.xfa.EnumAttr;
import com.adobe.xfa.EnumType;
import com.adobe.xfa.EnumValue;
import com.adobe.xfa.Model;
import com.adobe.xfa.Node;
import com.adobe.xfa.ProtoableNode;
import com.adobe.xfa.StringAttr;
import com.adobe.xfa.TextNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.ut.StringUtils;

/**
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */
public final class EventTag extends ProtoableNode {
	static final String gsFormCalc = "FormCalc";

	static final String gsJavaScript = "JavaScript";

	static final String gsMimeFormCalc = "application/x-formcalc";

	static final String gsMimeJavaScript = "application/x-javascript";

	/**
	 * Factory method. Creates either a new <code>&lt;calculate&gt;</code>,
	 * <code>&lt;validate&gt;</code>, or <code>&lt;event&gt;</code> tag,
	 * depending on the activity.
	 * 
	 */
	public static EventTag createEvent(Model model, String sActivity,
			String sRef) {
		String aActivity = sActivity.intern();

		int eTag = XFA.getTag(aActivity);
		if (eTag != -1 && eTag == XFA.CALCULATETAG
				|| eTag == XFA.VALIDATETAG) {
			return (EventTag) model.createElement(null, null, null, aActivity,
					aActivity, null, 0, null);

		} else {
			EventTag event = (EventTag) model.createElement(null, null, null,
					aActivity, aActivity, null, 0, null);

			event.setActivity(sActivity);
			event.setRef(sRef);
			return event;
		}
	}

	public EventTag(Element pParent, Node prevSibling) {
		super(pParent, prevSibling, null, XFA.EVENT, XFA.EVENT, null,
				XFA.EVENTTAG, XFA.EVENT);
	}

	public int defaultElement() {
		return XFA.SCRIPTTAG;
	}

	/**
	 * Get the activity (ie: OnEnter, OnClick, etc.) for this event.
	 */

	public String getActivity() {
		String className = getClassName();
		if (className == XFA.EVENT)
			return getAttribute(XFA.ACTIVITYTAG).toString();
		else
			return className;
	}

	/**
	 * Get the contentType for this event's script (eg. "javascript" or
	 * "formcalc").
	 */

	public String getContentType() {
		// ContentType is stored on <script> child.
		Script oScript = peekScript();
		if (oScript != null)
			return oScript.getAttribute(XFA.CONTENTTYPETAG).toString();
		return gsMimeFormCalc;
	}

	/**
	 * Get the language for this event's script.
	 * 
	 * While the "language" isn't as well-specified as the contentType, it is a
	 * useful shortcut for those clients supporting the primary languages (that
	 * is, JavaScript and FormCalc). "application/x-formcalc" -. "FormCalc"
	 * "application/x-javascript" -. "JavaScript" Others are copied straight
	 * across.
	 */
	public String getLanguage() {
		String sContentType = getContentType();

		if (sContentType.compareToIgnoreCase(gsMimeJavaScript) == 0)
			return gsJavaScript;
		else if (sContentType.compareToIgnoreCase(gsMimeFormCalc) == 0)
			return gsFormCalc;
		else
			return sContentType;
	}

	/**
	 * Get the reference of the activity (ie: $form, $layout, etc.) for this
	 * event.
	 */
	public String getRef() {
		if (getClassName() == XFA.EVENT)
			return getAttribute(XFA.REFTAG).toString();
		else
			return "";
	}

	/**
	 * Get the run location (client or server) for this event.
	 */
	public String getRunAt() {
		Script script = peekScript();
		if (script != null)
			return script.getAttribute(XFA.RUNATTAG).toString();
		else
			return "";
	}

	public Script getScript() {
		if (isSameClass(XFA.EVENTTAG)) {
			Node oNode = getOneOfChild();
			if (oNode != null && oNode.isSameClass(XFA.SCRIPTTAG))
				return (Script) oNode;
		} else
			return (Script) getElement(XFA.SCRIPTTAG, false, 0, true, false);

		return null;
	}

	/**
	 * Get the content of the script for this event.
	 */
	public String getScriptText() {
		Script oScript = peekScript();
		if (oScript == null)
			return "";

		TextNode oTextNode = oScript.getText(true, false, false);
		if (oTextNode != null)
			return oTextNode.getValue();

		return "";
	}

	/**
	 * Get the script for this event.
	 */
	public Script peekScript() {
		if (isSameClass(XFA.EVENTTAG)) {
			Node oNode = getOneOfChild(true, false);
			if (oNode != null && oNode.isSameClass(XFA.SCRIPTTAG))
				return (Script) oNode;
		} else
			return (Script) peekElement(XFA.SCRIPTTAG, false, 0);

		return null;
	}

	/**
	 * Set the activity (ie: OnEnter, OnClick, etc.) for this event.
	 * 
	 * @param sActivity -
	 *            the name of the event to set,
	 */
	public void setActivity(String sActivity) {
		// Can't change the activity on a <calculate> or <validate>!
		assert (getClassName().compareToIgnoreCase(XFA.EVENT) == 0);

		setAttribute(new StringAttr(XFA.ACTIVITY, sActivity), XFA.ACTIVITYTAG);
	}

	/**
	 * Set the contentType for this event's script (eg. "javascript" or
	 * "formcalc").
	 * 
	 * @param sContentType -
	 *            the content type to set for this element.
	 */
	public void setContentType(String sContentType) {
		// ContentType is stored on <script> child.
		Script oScript = getScript();
		if (oScript != null)
			oScript.setAttribute(new StringAttr(XFA.CONTENTTYPE, sContentType),
					XFA.CONTENTTYPETAG);
	}

	/**
	 * Set the language for this event's script.
	 * <p>
	 * While the "language" isn't as well-specified as the contentType, it is a
	 * useful shortcut for those clients supporting the primary languages (that
	 * is, JavaScript and FormCalc). "FormCalc" -. "application/x-formcalc"
	 * "JavaScript" -. "application/x-javascript" Others are copied straight
	 * across.
	 */
	public void setLanguage(String sLanguage) {
		if (sLanguage.compareToIgnoreCase(gsJavaScript) == 0)
			setContentType(gsMimeJavaScript);
		else if (sLanguage.compareToIgnoreCase(gsFormCalc) == 0)
			setContentType(gsMimeFormCalc);
		else
			setContentType(sLanguage);
	}

	/**
	 * Set the reference of the activity (ie: $form, $layout, etc.) for this
	 * event.
	 * 
	 * @param sRef -
	 *            The object this event refers to.
	 */
	public void setRef(String sRef) {
		// Can't change the ref on a <calculate> or <validate>!
		assert (getClassName().compareToIgnoreCase(XFA.EVENT) == 0);

		setAttribute(new StringAttr(XFA.REF, sRef), XFA.REFTAG);
	}

	/**
	 * Set the run location (client or server) for this event.
	 * 
	 * @param sTarget -
	 *            The target: client, server etc.
	 */
	public void setRunAt(String sTarget) {
		Script script = getScript();
		if (script != null)
			script.setAttribute(EnumValue.getEnum(XFA.RUNATTAG, EnumAttr.getEnum(EnumType.RUNAT_TYPE, sTarget)), XFA.RUNATTAG);
	}

	/**
	 * Set the script for this event.
	 */
	public void setScript(Script script) {
		if (isSameClass(XFA.EVENTTAG))
			setOneOfChild(script);
		else
			setElement(script, XFA.SCRIPTTAG, 0);
	}

	/**
	 * Set the content of the script for this event.
	 */
	public void setScriptText(String sScript) {
		// #ifdef UNDOENABLED
		// {
		// // Must expand UNDO macro because this node is a wrapper, not an
		// impl.
		// Node poNodeImpl = getImplPtr();
		// XFAUndoStackImpl* poSI = XFAUndo.getUndoStack(poNodeImpl);
		// if (poSI) poSI.addUndo(new XFAXMLUndo (poNodeImpl));
		// }
		// #endif // UNDOENABLED

		Script oScript = getScript();

		if (oScript == null && StringUtils.isEmpty(sScript)) {
			// nothing to do....
			return;
		}

		if (StringUtils.isEmpty(sScript)) {
			removeChild(oScript);
			return;
		}

		if (oScript == null) {
			oScript = (Script) getModel().createElement(null, null, null,
					XFA.SCRIPT, XFA.SCRIPT, null, 0, null);
			setScript(oScript);
		}
		TextNode oTextNode = oScript.getText(true, false, false);
		if (oTextNode == null) {
			oTextNode = getModel().createTextNode(oScript, null, null, 0, 0);
		}
		oTextNode.setValue(sScript,true,false);
	}
}
