package com.adobe.xfa.text;

import com.adobe.xfa.ut.UnitSpan;


/**
 * This class represents a single tab stop.  As such, it contains two
 * pieces of information: an offset and an alignment (referred to as
 * type in this interface).  The offset is measured in form units and
 * the type is one of left, centre, right or decimal (radix).
 * </p>
 * <p>
 * In left-to-right (LTR) text, tab offsets measure from the left-hand
 * side.  In right-to-left (RTL) text, tab offsets measure from the
 * right-hand side and the meanings of the left and right types are
 * inverted.
 * </p>
 * <p>
 * For more information, please see the extenral documentation.
 * </p>
 *
 * @exclude from published api.
 */

// TODO: Check against current implementation

public class TextTab {
	public final static int TYPE_LEFT = 0;
	public final static int TYPE_CENTRE = 1;
	public final static int TYPE_RIGHT = 2;
	public final static int TYPE_DECIMAL = 3;
	public final static int TYPE_ALIGN_AFTER = 4;
	public final static int TYPE_ALIGN_BEFORE = 5;

	public final static TextTab DEFAULT_TAB = new TextTab (0.5f, TextTab.TYPE_ALIGN_AFTER);
	public final static TextTab ZERO_TAB = new TextTab();

	private int meType = TYPE_LEFT;
	private UnitSpan moStop = null;

/**
 * Default constructor.
 * <p>
 * The offset is initialized to zero and the tab has an initial type of
 * left.
 */
	public TextTab () {
	}

/**
 * Copy constructor.
 * <p>
 * Copy both the offset and the type from the source tab stop object.
 * @param oSource - Source tab stop to copy.
 */
	public TextTab (TextTab oSource) {
		meType = oSource.meType;
		moStop = oSource.moStop;
	}

/**
 * Constructor with offset and optional type.
 * @param oNewStop - Offset value for the tab stop object.
 * @param eNewType - (optional) Initial tab type.  Default is left.
 */
	public TextTab (UnitSpan oNewStop, int eNewType) {
		meType = eNewType;
		moStop = oNewStop;
	}

	public TextTab (double fNewStop, int eNewType) {
		meType = eNewType;
		moStop = new UnitSpan (fNewStop, UnitSpan.INCHES_72K);
	}

/**
 * Optain the tab stop's current type.
 * @return Current tab stop type (left, centre, right or decimal).
 */
	public int tabType () {
		return meType;
	}

/**
 * Change the tab stop's type.
 * @param eNewType - New tab type.
 */
	public void tabType (int eNewType) {
		meType = eNewType;
	}

/**
 * Obtain the tab stop offset.
 * @return Current offset value for the tab stop.
 */
	public UnitSpan tabStop () {
		return (moStop == null) ? UnitSpan.ZERO : moStop;
	}

/**
 * Change the tab stop's offset.
 * @param oNewStop - New offset value for the tab stop.
 */
	public void tabStop (UnitSpan oNewStop) {
		moStop = oNewStop;
	}

/**
 * Obtain the numeric offset value.
 * <p>
 * This method returns the value portion of the tab offset measurement.
 * It is meant as a short-cut for sero comparisons only.  For example,
 * you can say oTab.Value() rather than oTab.TabStop().Value().  Big
 * deal.
 * @return Tab stop numeric value, unadorned by unit type.
 */
	public int value () {
		return (moStop == null) ? 0 : moStop.value();
	}

/**
 * Resolve the tab type, based on text direction.
 * <p>
 * Given a tab type and text direction, this method returns a resolved
 * type.  The behaviour of the resolution also depends on whether it is
 * occuring for layout.
 * <p>
 * General (non-layout) resolution: For LTR text, TYPE_ALIGN_AFTER
 * resolves to TYPE_LEFT and TYPE_ALIGN_BEFORE resolves to TYPE_RIGHT. 
 * For RTL text, TYPE_ALIGN_AFTER resolves to TYPE_RIGHT and
 * TYPE_ALIGN_BEFORE resolves to TYPE_LEFT.  All other types do not
 * change.
 * </p>
 * <p>
 * Layout resoultion: For LTR text, TYPE_LEFT resolves to
 * TYPE_ALIGN_AFTER and TYPE_RIGHT resolves to TYPE_ALIGN_BEFORE.  For
 * RTL text, TYPE_LEFT resolves to TYPE_ALIGN_BEFORE and TYPE_RIGHT
 * resolves to TYPE_ALIGN_AFTER.
 * </p>
 * @param eSource - Source type code to resolve.
 * @param bRTL - True for RTL text; false for LTR text;
 * @param bLayout - (optional) True if this resolution is for layout;
 * false (default) for general resolution.
 * @return Resolved type.
 */
	public static int resolveType (int eSource, boolean bRTL, boolean bLayout) {
		if (bLayout) {
			switch (eSource) {
				case TYPE_LEFT:
					return bRTL ? TYPE_ALIGN_BEFORE : TYPE_ALIGN_AFTER;
				case TYPE_RIGHT:
					return bRTL ? TYPE_ALIGN_AFTER : TYPE_ALIGN_BEFORE;
			}
		}

		else {
			switch (eSource) {
				case TYPE_ALIGN_BEFORE:
					return bRTL ? TYPE_LEFT : TYPE_RIGHT;
				case TYPE_ALIGN_AFTER:
					return bRTL ? TYPE_RIGHT : TYPE_LEFT;
			}
		}

		return eSource;
	}

/**
 * Assignment operator.
 * <p>
 * Copies both the offset and type from the source tab stop object.
 * @param oSource - Source tab stop object to copy.
 * @return A reference to this object.
 */
	public TextTab copyFrom (TextTab oSource) {
		meType = oSource.meType;
		moStop = oSource.moStop;

		return this;
	}

/**
 * Equality comparison.
 * <p>
 * Two tab stops are considered equal if their offsets compare as equal
 * and they have the same type.
 * @param object - Tab stop object to compare against.
 * @return TRUE if the tab stops are equal; FALSE otherwise.
 */
	public boolean equals (Object object) {
		if (this == object)
			return true;
		
		// This overrides Object.equals(boolean) directly, so...
		if (object == null)
			return false;
		
		if (object.getClass() != getClass())
			return false;
		
		TextTab test = (TextTab) object;
		
		if ((moStop != null) && (test.moStop != null)) {
			if (! moStop.equals(test.moStop)) {
				return false;
			}
		} 
		else if (moStop != test.moStop) {	// at least one null: both null OK
			return false;
		}
		
		return (meType == test.meType);
	}

	public int hashCode() {
		int hash = 67;
		if (moStop != null)
			hash = (hash * 31) ^ moStop.hashCode();
		hash = (hash * 31) ^ meType;
		return hash;
	}

/**
 * Inequality comparison.
 * <p>
 * Two tab stops are considered unequal if either their offsets are not
 * equal or they have different types.
 * @param oCompare - Tab stop object to compare against.
 * @return TRUE if the tab stops are not equal; FALSE if equal.
 */
	public boolean notEqual (TextTab oCompare) {
		return ! equals (oCompare);
	}

/**
 * Less than comparison.
 * <p>
 * Compare the offset values only; ignores the tab stops' types.
 * @param oCompare - Tab stop object to compare against.
 * @return TRUE if this tab stop is less than the given tab stop; FALSE
 * otherwise.
 */
	public boolean lessThan (TextTab oCompare) {
		return lt (oCompare.tabStop());
	}

/**
 * Less than or equal to comparison.
 * <p>
 * Compare the offset values only; ignores the tab stops' types.
 * @param oCompare - Tab stop object to compare against.
 * @return TRUE if this tab stop is less than or equal to the given tab
 * stop; FALSE otherwise.
 */
	public boolean lessThanOrEqual (TextTab oCompare) {
		return lte (oCompare.tabStop());
	}

/**
 * Greater than comparison.
 * <p>
 * Compare the offset values only; ignores the tab stops' types.
 * @param oCompare - Tab stop object to compare against.
 * @return TRUE if this tab stop is greater than the given tab stop;
 * FALSE otherwise.
 */
	public boolean greaterThan (TextTab oCompare) {
		return gt (oCompare.tabStop());
	}

/**
 * Greater than or equal to comparison.
 * <p>
 * Compare the offset values only; ignores the tab stops' types.
 * @param oCompare - Tab stop object to compare against.
 * @return TRUE if this tab stop is greater than or equal to the given
 * tab stop; FALSE otherwise.
 */
	public boolean greaterThanOrEqual (TextTab oCompare) {
		return gte (oCompare.tabStop());
	}

/**
 * Equality comparison with measurement.
 * <p>
 * Compare the tab stop's offset against a measurement (jfUnitSpan) for
 * equality.  The tab stop type is ignored.
 * @param oCompare - Measurement to compare against.
 * @return TRUE if this tab stop's offset is equal to the given
 * measurement; FALSE otherwise.
 */
	public boolean equals (UnitSpan oCompare) {
		return tabStop().equals (oCompare);
	}

/**
 * Inequality comparison with measurement.
 * <p>
 * Compare the tab stop's offset against a measurement (jfUnitSpan) for
 * inequality.	The tab stop type is ignored.
 * @param oCompare - Measurement to compare against.
 * @return TRUE if this tab stop's offset is not equal to the given
 * measurement; FALSE otherwise.
 */
	public boolean notEqual (UnitSpan oCompare) {
		return ! equals (oCompare);
	}

/**
 * Less than comparison with measurement.
 * <p>
 * Compare the tab stop's offset against a measurement (jfUnitSpan).
 * The tab stop type is ignored.
 * @param oCompare - Measurement to compare against.
 * @return TRUE if this tab stop's offset is less than the given
 * measurement; FALSE otherwise.
 */
	public boolean lt (UnitSpan oCompare) {
		return tabStop().lt (oCompare);
	}

/**
 * Less than or equal to comparison with measurement.
 * <p>
 * Compare the tab stop's offset against a measurement (jfUnitSpan).
 * The tab stop type is ignored.
 * @param oCompare - Measurement to compare against.
 * @return TRUE if this tab stop's offset is less than or equal to the
 * given measurement; FALSE otherwise.
 */
	public boolean lte (UnitSpan oCompare) {
		return tabStop().lte (oCompare);
	}

/**
 * Greater than comparison with measurement.
 * <p>
 * Compare the tab stop's offset against a measurement (jfUnitSpan).
 * The tab stop type is ignored.
 * @param oCompare - Measurement to compare against.
 * @return TRUE if this tab stop's offset is greater than the given
 * measurement; FALSE otherwise.
 */
	public boolean gt (UnitSpan oCompare) {
		return tabStop().gt (oCompare);
	}

/**
 * Greater than or equal to comparison with measurement.
 * <p>
 * Compare the tab stop's offset against a measurement (jfUnitSpan).
 * The tab stop type is ignored.
 * @param oCompare - Measurement to compare against.
 * @return TRUE if this tab stop's offset is greater than or equal to
 * the given measurement; FALSE otherwise.
 */
	public boolean gte (UnitSpan oCompare) {
		return tabStop().gte (oCompare);
	}

	public void debug () {
		debug (0);
	}

	void debug (int indent) {
		System.out.print (Pkg.doIndent (indent+1) + "Tab: ");
		String type = "(unknown)";
		switch (meType) {
		case TYPE_LEFT:
			type = "left";
			break;
		case TYPE_CENTRE:
			type = "centre";
			break;
		case TYPE_RIGHT:
			type = "right";
			break;
		case TYPE_DECIMAL:
			type = "decimal";
			break;
		}
		System.out.println (type + ' ' + tabStop().toString());
	}
}
