/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.wcm.api.designer;

import java.io.IOException;
import java.io.Writer;
import java.util.Calendar;
import java.util.Map;

import javax.jcr.RepositoryException;
import javax.servlet.jsp.PageContext;

import org.apache.sling.api.resource.Resource;
import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.commons.Doctype;

/**
 * Provides methods to access design information.
 */
@ProviderType
public interface Design {

    /**
     * Name of the property that defines the doctype of a design
     *
     * @since 5.4
     */
    static final String PN_DOCTYPE = "cq:doctype";

    /**
     * Name of the node that holds special component styles
     *
     * @since 5.4
     */
    static final String NN_STYLES= "cq:styles";
    
    
    /**
     * Name of the property that defines components that are included when the
     * style is generated via the {@link #getStyle(Resource, boolean)} call
     */
    static final String PN_STYLE_EXCLUDED_COMPONENTS = "cq:styleExcludedComponents";    

    /**
     * Returns the path of the design
     * @return the path of the design
     */
    String getPath();

    /**
     * Returns the id of the design.
     * @return the id of the design.
     */
    String getId();

    /**
     * Returns the resource of the design content or <code>null</code> if
     * this design has no content.
     * @return resource or <code>null</code>
     *
     * @see #hasContent()
     */
    Resource getContentResource();

    /**
     * Checks if this design has content attached.
     * @return <code>true</code> if this design has content;
     *         <code>false</code> otherwise.
     */
    boolean hasContent();

    /**
     * Returns the style for the given cell path. If the style is not defined
     * and empty style is returned.
     *
     * Please note that this form of addressing does not respect style
     * inheritance via the cell stack.
     *
     * @param path the path of the cell
     * @return the style
     */
    Style getStyle(String path);

    /**
     * Returns the style for the given cell. The returned style will
     * respect cell stack inheritance.
     *
     * @param cell the cell
     * @return the style
     */
    Style getStyle(Cell cell);

    /**
     * Returns the style for the given resource. Note that only the content
     * structure of the resource is respected and not the component include
     * hierarchy.
     * @param res resource
     * @return the style
     *
     * @since 5.4
     */
    Style getStyle(Resource res);
    
    /**
     * Returns the style for the given resource
     * 
     * <p>Note that only the content structure of the resource is respected and not the component include
     * hierarchy.</p>
     * 
     * <p>The hierarchy may be modified by removing certain components based on the information from the request
     * if the ignoreExcludedComponents parameter is set to true. Excluded components are defined in the 
     * design with the {@value #PN_STYLE_EXCLUDED_COMPONENTS} property.</p>
     * 
     * @param res the resource
     * @param ignoreExcludedComponents whether to ignore excluded components
     * @return the style
     * 
     * @see #PN_STYLE_EXCLUDED_COMPONENTS
     * @since 6.2
     */
    Style getStyle(Resource res, boolean ignoreExcludedComponents);    

    /**
     * Returns the path of the css file for this design
     * @return the path of the css file.
     */
    String getCssPath();

    /**
     * Returns the path of the static css file for this design or <code>null</code>
     * if no such file exists.
     * @return the path of the static css file or <code>null</code>
     */
    String getStaticCssPath();

    /**
     * Convenience method that writes the CSS include strings to the given
     * writer. This assumes a non XHTML doctype.
     *
     * @param out the writer
     * @throws IOException if an I/O error occurs
     */
    void writeCssIncludes(Writer out) throws IOException;

    /**
     * Convenience method that writes the CSS include strings to the given
     * writer. The specified doctype ensures proper empty elements for
     * XHTML types. If <code>doctype</code> is <code>null</code>,
     * {@link Doctype#HTML_401_STRICT} is used.
     *
     * @param out the writer
     * @param doctype doctype of the document the include is written to.
     * @throws IOException if an I/O error occurs
     */
    void writeCssIncludes(Writer out, Doctype doctype) throws IOException;

    /**
     * Convenience method that writes the CSS include strings to the response.
     * It uses the {@link Doctype} defined in the request or defaults to
     * {@link Doctype#HTML_401_STRICT} if non defined.
     *
     * @param page the jsp page context
     * @throws IOException if an I/O error occurs
     * @since 5.3
     */
    void writeCssIncludes(PageContext page) throws IOException;

    /**
     * Writes the dynamic CSS of this design to the given writer.
     *
     * @param out writer to write to
     * @param includeCustom if <code>true</code> non-CSS rules are included as comments
     *
     * @throws RepositoryException if a repository error occurs
     * @throws IOException if an I/O error occurs
     */
    void writeCSS(Writer out, boolean includeCustom) throws IOException, RepositoryException;

    /**
     * Returns the doctype of the current style, as defined in the
     * {@value #PN_DOCTYPE} property. If the property does not exist or if
     * <code>currentStyle</code> is <code>null</code>, the property is searched
     * on the toplevel node. the value of the property needs to be a name of
     * a {@link Doctype} enum. if any of the above fails, {@link Doctype#XHTML_10_STRICT}
     * is returned.
     *
     * @param currentStyle current style or <code>null</code>
     * @return the doctype
     *
     * @since 5.4
     */
    Doctype getDoctype(Style currentStyle);

    /**
     * Returns the component styles for the given cell.
     *
     * @param cell the cell
     * @return the map of component styles
     *
     * @since 5.4
     */
    Map<String, ComponentStyle> getComponentStyles(Cell cell);

    /**
     * Returns a json representation of this design
     * @return a json string
     *
     * @since 5.4
     */
    String getJSON();

    /**
     * Returns the time when the design was last modified or <code>null</code> if
     * this information is not available or if the design has no content.
     * @return last modification date or <code>null</code>
     *
     * @since 5.4
     */
    Calendar getLastModified();

}