/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.haf.annotations;

import java.lang.annotation.ElementType;
import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.lang.annotation.Target;

/**
 * Registers a property getter with the API framework.  When a model is serialized the property is included in
 * the properties section of the serialized JSON.
 */
@Target({ ElementType.METHOD, ElementType.FIELD })
@Retention(RetentionPolicy.RUNTIME)
public @interface ApiProperty {

    /**
     * The name of the property to be serialized.  If the name isn't explicity specified, then the name of the
     * method or field is used in lowercase, with prefixes "get" or "is" removed.
     * @return The name of the property
     */
    String name() default "";

    /**
     * The scope in which the property should be serialized.  If SCOPE.RESOURCE the property will be serialized for a
     * resource that is the main entity.  If SCOPE.INLINE the property will be serialized for a resource that is a
     * sub-entity of another resource.  By default the property will be serialized in both cases.
     * @return The scope in which this property should be serialized.
     */
    SCOPE scope() default SCOPE.BOTH;

    /**
     * Boolean used when a method or field is a Map of name/value pairs.  If true the properties contained in the
     * Map will be serialized each on their own at the top level of the properties section of the entity.  If false,
     * the Map will be serialzied as a single property in the property section.
     * @return Whether or not to flatten a map in the properties section
     */
    boolean flattenMap() default false;

    /**
     * Defines the scope in which the property is being serialized in.
     */
    enum SCOPE {
        /**
         *  The property will only be serialized as part of a main entity.
         */
        RESOURCE,
        /**
         * The property will only be serialized as part of a sub-enity.
         */
        INLINE,
        /**
         * The property will be serialized as part of both a main and a sub-entity.
         */
        BOTH;

        /**
         * Determines if the current scope contains the passed in scope value.  If the current scope is SCOPE.BOTH
         * true is always returned.  Otherwise a direct comparison of the current scope and the passed one is done.
         * @param scope The scope value to test the current scope against.
         * @return True is the current scope contains the passed in scope, false otherwise.
         */
        public boolean contains(SCOPE scope) {
            if (this == SCOPE.BOTH) {
                return true;
            } else {
                return this == scope;
            }
        }
    }
}
