/*
 * File: TIFFInputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.IOException;
import java.io.InputStream;

/**
 * TIFFInputStream
 *  Performs TIFF image filtering of an input stream
 */
public class TIFFInputStream extends DecodeInputStream
{
	// Variables
	private TIFFEngine	tiff;
	

	// Constructors
	/**
	 * Creates an input stream filter.
	 * @param in	the input stream
	 */
	public TIFFInputStream(InputStream in, int inSize, int outSize, FilterParams diparams)
	{
		super(in, inSize, outSize, 1, diparams);
		tiff = new TIFFEngine(in, diparams);
	}

	public TIFFInputStream(InputStream  in, FilterParams p)
	{
		super(in, 1, p);
		tiff = new TIFFEngine(this, p);
	}

	public TIFFInputStream(InputStream  in)
	{
		this(in, null);
	}

	// Methods
	/**
	 * Set the TIFF predictor. This must be done before reading any bytes.
	 * --------- TIFF pre-compression/post-decompression predictors --------
	 * LZW and Flate use these predictors.
	 * NOTE: The predictors are all very slow.  Hor. differencing, for example,
	 * is 100% slower compressing and 600% slower expanding than no predictor.
	 * Predictors:   0      (no prediction)
	 *	       1      DEFAULT (no prediction)
	 *	     Component sizes 1, 2, 4, and 8:
	 *	       2      hor. differencing (see TIFF 6.0 spec)
	 *	     Component size 8 only:
	 *	       3..9   unused
	 *	       10     PNG code 0: None
	 *	       11     PNG code 1: Sub
	 *	       12     PNG code 2: Up
	 *	       13     PNG code 3: Average
	 *	       14     PNG code 4: Paeth
	 *	       15     PNG optimum (best efforts)
	 */
	@Override
	public void fill(){
		int	newChar;
		int bitsPerComponents = tiff.getBitsPerComponent();
		if(bitsPerComponents==16){
			while (outCount <=  (outBuf.length - 1) && !pendingEOF && pendingException == null) {
				if(isByteAvailable()){
					break;
				}
				int channel =0, result =0;
                newChar = (short) (inBuf[inPos++] & 0x00FF);
                channel = (short) (channel | (newChar<<8));
				if(isByteAvailable()){
					break;
				}
                newChar = (short) (inBuf[inPos++] & 0x00FF);
                channel = (short) (channel | newChar);
                try {
					result = tiff.applyTIFFPredictor(channel);
	                outBuf[outCount++] = (byte) ((result & 0xFF00) >> 8);
	                outBuf[outCount++] = (byte) (result & 0x00FF);
				} catch (IOException e) {
					/* IOException is only thrown when doing TIFF OUTPUT */
					pendingException =
						new TIFFFilterException("TIFF error: "+e.getMessage());
					break;
				}
			}				
		}else{
			fillLoop:
			while (outCount <= (outBuf.length - 1) && !pendingEOF && pendingException == null) {
				if ( inCount <= inPos ) {
					if (!fillInputBuffer()) {
						pendingEOF = true;
						break fillLoop;
					}
				}
				newChar = (char) (inBuf[inPos++] & 0xFF);
				try {
					outBuf[outCount++] = (byte)tiff.applyTIFFPredictor(newChar);
				}
				catch (IOException e) {
					/* IOException is only thrown when doing TIFF OUTPUT */
					pendingException =
						new TIFFFilterException("TIFF error: "+e.getMessage());
					break;
				}
			} /* fillLoop */
		}
	}

	/* Callback from engine for another character */
	int read1()
	{
		if ( inCount <= inPos ) {
			if (!fillInputBuffer()) {
				pendingEOF = true;
				return -1;
			}
		}
		return (inBuf[inPos++] & 0xFF);
	}
	
	/**
	 * Returns true, if byte is available to read.
	 * @return boolean
	 */
	private boolean isByteAvailable(){
		if ( inCount <= inPos ) {
			if (!fillInputBuffer()) {
				pendingEOF = true;
				return true;
			}
		}
		return false;
	}

}
