/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.protocol;


import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.util.List;
import java.util.Map;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormat;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.Resolver;


/**
 * 
 * @exclude from published api.
 */

public class SysProtocol implements Protocol {
	
	protected final AuthenticationHandler mAuthHandler;	// ALWAYS check for null before using!
	protected final URLStreamHandler mURLStreamHandler;
	private final static int BUFFSIZE = 1024;

	/**
	 * The default c'tor -- instantiate a Protocol object.
	 */
	public SysProtocol() {
		this(null, null);
	}
	
	public SysProtocol(AuthenticationHandler authenticationHandler,
					   URLStreamHandler urlStreamHandler) {
		
		mAuthHandler = authenticationHandler;
		mURLStreamHandler = urlStreamHandler;
	    Resolver.setProtocol(this);
	}

	/**
	 * Method to get (download) a file designated by the given URL.
	 * @param sUrl the URL to download.
	 * @return an InputStream that contains the downloaded data.
	 * @throws ExFull "Couldn't read/get data from '%s'." exception.
	 */
    public InputStream get(String sUrl) {
		URL url = null;
		URLConnection connection = null;
		InputStream	iFile = null;
		try {
			url = new URL(sUrl);
			connection = url.openConnection();
			connection.setUseCaches(true);
			iFile = connection.getInputStream();
		} 
		catch (MalformedURLException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_READ, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			throw oEx;
		}
		catch (IOException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_READ, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			throw oEx;
		}

		return iFile;
    }

	/**
	 * Method to put (upload) the given file to a file designated
	 * by the given URL.
	 * @param sFileName the file to upload.
	 * @param sUrl the name of the URL to upload to.
	 * @throws ExFull "stream implementation unavailable" exception.
	 */
    public void put(String sFileName, String sUrl) {
		InputStream iFile = null;
		try {
			//
			// Open the input file.
			//				
			iFile = new BufferedInputStream(new FileInputStream(sFileName));
			this.put(iFile, sUrl);
		}
		catch (IOException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_WRITE, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			throw oEx;
		}
    	finally {
			try {
				if (iFile != null)
					iFile.close();
			} 
			catch (IOException e) {
			}
    	}
    }

	/**
	 * Method to put (upload) memory to a designated URL.
	 * The Internet server targeted by the URL must obviously
	 * be configured to allow uploads.
	 * @param iFile the InputStream to be uploaded to the URL.
	 * @param sUrl the name of the URL to upload memory to.
	 */
	public void put(InputStream iFile, String sUrl) {
    	URL url = null;
    	URLConnection connection = null;
    	OutputStream oFile = null;

    	try {
			url = new URL(sUrl);
			connection = url.openConnection();
			connection.setUseCaches(true);
			connection.setDoOutput(true);
			if (connection instanceof HttpURLConnection)
				((HttpURLConnection) connection).setRequestMethod("PUT");
			oFile = connection.getOutputStream();

			byte[] buf = new byte[BUFFSIZE];
			int iCnt = 0;
			while ((iCnt = iFile.read(buf)) != -1)
				oFile.write(buf, 0, iCnt);
			if (connection instanceof HttpURLConnection) {
				HttpURLConnection httpConnection = (HttpURLConnection)connection;
				if (httpConnection.getResponseCode() >= 300) {
					ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_WRITE, sUrl));
					ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, httpConnection.getResponseMessage()));
					oEx.insert(oSysEx, true);
					throw oEx;
				}
			}
		} 
		catch (MalformedURLException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_WRITE, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			throw oEx;
		} 
		catch (IOException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_WRITE, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			throw oEx;
		}
    	finally {
			try {
				if (oFile != null)
					oFile.close();
			} 
			catch (IOException e) {
			}
    	}
	}

	/**
	 * Method to post form data to a designated URL.
	 * The targeted URL must obviously be designed to read
	 * the data posted.
	 * @param oData a description of the data being posted to the URL.
	 * @param sUrl the name of the URL to post data to.
	 * @return The status and response from the Post as a PostRsvp object.
	 */
	public PostRsvp post(SimplePostData oData, String sUrl) {
		URL url = null;
		HttpURLConnection connection = null;
		
		String sResponseType = null;
		byte[] response = null;
		ExFull exception = null;
		
		try {
			url = new URL(sUrl);
			connection = (HttpURLConnection)url.openConnection();
			connection.setDoInput(true);
			connection.setDoOutput(true);
			connection.setRequestMethod("POST");
			if ((oData.headerMap != null) && (oData.headerMap.size() > 0)) {
				for (Map.Entry<String, String> entry : oData.headerMap.entrySet()) {
					connection.setRequestProperty(entry.getKey(), entry.getValue());					
				}
			}
			else
				connection.setRequestProperty ("Content-Type", "application/x-www-form-urlencoded");
		    
			OutputStream out = connection.getOutputStream();
			try {
				if (oData.data != null) {
					// send the encoded message
					out.write(oData.data);
					out.flush();
				}
			}
			finally {
				if (out != null)
					try { out.close(); }
					catch (IOException ignored) {}
			}
			
					

			InputStream responseStream = (connection.getResponseCode() >= 300)
											? connection.getErrorStream()
											: connection.getInputStream();
			
			try {
				response = readResponseStream(responseStream, connection.getContentLength());
			}
			finally {
				if (responseStream != null)
					try { responseStream.close(); }
					catch (IOException ignored) {}
			}
			
			sResponseType = connection.getContentType();
			
			if (connection.getResponseCode() >= 300) {
				ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_POST, sUrl));
				String detail;
				if (response == null)
					detail = connection.getResponseMessage();
				else {
					// ASSUME: the response content is a text type
					detail = new String(response, getResponseCharset(sResponseType));
				}
				ExFull oSysEx = new ExFull (detail, ResId.PROTOCOL_ERR_SYS);
				oEx.insert(oSysEx, true);
				exception = oEx;
			}
		} 
		catch (MalformedURLException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_POST, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			exception = oEx;
		}
		catch (IOException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_POST, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			exception = oEx;
		}
		
		return new PostRsvp(0, sResponseType, response, exception);
	}
    	
	/**
	 * Virtual method to post form data to a designated URL.
	 * The targeted URL must obviously be designed to read
	 * the data posted.
	 * @param oData a description of the data being posted to the URL.
	 * @param sUrl the name of the URL to post data to.
	 * @return The status and response from the Post as a PostRsvp object
	 */
	public PostRsvp post(List<? extends MultiPartDesc> oData, String sUrl) {
		URL url = null;
		HttpURLConnection connection = null;		

		ExFull exception = null;
		String sResponseType = null;
		byte[] response = null;
		
		try {
			url = new URL(sUrl);
			connection = (HttpURLConnection)url.openConnection();
			connection.setDoInput(true);
			connection.setDoOutput(true);
			connection.setUseCaches(false);
            connection.setRequestProperty("MIME-version", "1.0");
            byte[] boundary = ProtocolUtils.mimeBoundary(null);
			connection.setRequestProperty("Content-Type", "multipart/form-data; boundary=" + new String(boundary, "US-ASCII"));
			connection.setRequestMethod("POST");

			OutputStream out = connection.getOutputStream();
			
			// send the encoded message
			
			boolean bReset = true;
			byte[] name = null;
			byte[] file = null;
			byte[] type = null;
			byte[] value = null;
			
			for (MultiPartDesc desc : oData) {
				
				if (bReset) {
					name = file = type = value = null;
					bReset = false;
				}
				
				Protocol.SectionDataOption eOption = desc.eSectionDataOption;
				
				if (eOption != Protocol.SectionDataOption.SECTION_END) {
				
					if (eOption == Protocol.SectionDataOption.SECTION_CONTENT_NAME)			// section name.
						name = desc.value;
					else if (eOption == Protocol.SectionDataOption.SECTION_CONTENT_FILE)	// section file name.
						file = desc.value;
					else if (eOption ==	Protocol.SectionDataOption.SECTION_CONTENT_TYPE)	// section content type.
						type = desc.value;
					else if (eOption == Protocol.SectionDataOption.SECTION_CONTENT_VALUE)	// section content.
						value = desc.value;
					
					continue;
				}
				else {
					out.write(ProtocolUtils.mimeSection(boundary, name, file, type, value));
					out.write(ProtocolUtils.mimeTrailer(boundary));
					out.flush();
					
					bReset = true;
				}
			}
	
			out.close();

			// Get the status code
			if (connection.getResponseCode() >= 300) {
				ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_POST, sUrl));
				ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, connection.getResponseMessage()));
				oEx.insert(oSysEx, true);
				exception = oEx;
			}
			
			// get the response
			sResponseType = connection.getContentType();			
			InputStream responseStream = connection.getInputStream();
			try {
				response = readResponseStream(responseStream, connection.getContentLength());
			}
			finally {
				if (responseStream != null)
					try { responseStream.close(); }
					catch (IOException ignored) {}
			}
		} 
		catch (MalformedURLException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_POST, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			exception = oEx;
		}
		catch (IOException e) {
			ExFull oEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_POST, sUrl));
			ExFull oSysEx = new ExFull(new MsgFormat(ResId.PROTOCOL_ERR_SYS, e.getMessage()));
			oEx.insert(oSysEx, true);
			exception = oEx;
		}

		return new PostRsvp(0, sResponseType, response, exception);	
	}
	
	private byte[] readResponseStream(InputStream responseStream, int nContentLength) throws IOException {
		
		if (responseStream == null)
			return null;
		
		// get the response
		byte[] response = null;
		
		if (nContentLength != -1) {
			response = new byte[nContentLength];
			int nBytesRead = responseStream.read(response);
			assert nBytesRead == nContentLength;
		}
		else {
			ByteArrayOutputStream responseBuffer = new ByteArrayOutputStream();
			byte[] buffer = new byte[BUFFSIZE];
			int nBytesRead;
			while ((nBytesRead = responseStream.read(buffer)) != -1)
				responseBuffer.write(buffer, 0, nBytesRead);
			
			response = responseBuffer.toByteArray();
		}
		
		return response;
	}
	
	private String getResponseCharset(String sResponseType) {
		String sCharset = "ISO-8859-1";	// HTTP 1.1 default
		if (sResponseType != null) {
			int i = sResponseType.indexOf("charset=");
			if (i != -1) {
				int beginIndex = i + "charset=".length();
				char endDelimiter = ';';
				if (beginIndex < sResponseType.length() && sResponseType.charAt(beginIndex) == '"') {
					beginIndex++;
					endDelimiter = '"';
				}
				
				int endIndex = sResponseType.indexOf(endDelimiter, beginIndex);
				if (endIndex == -1)
					endIndex = sResponseType.length();
				
				sCharset = endIndex == -1 ? sResponseType.substring(beginIndex) : 
					                        sResponseType.substring(beginIndex, endIndex);
			}
		}
		
		return sCharset;
	}


	/**
	 * Gets this protocol's authentication handler.
	 */
	public AuthenticationHandler getAuthenticationHandler() {
		return mAuthHandler;
	}

	public URLStreamHandler getURLStreamHandler() {
		return mURLStreamHandler;
	}

	public boolean isTrusted(String sURL, TrustType eTrustType, boolean bThrow) {
		return true;
	}

	public String scheme() {
		return "";
	}
}
