/*
 *
 *	File: FontLoader.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2004-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */
package com.adobe.fontengine.fontmanagement;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.lang.ref.SoftReference;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import com.adobe.fontengine.FontEngineException;
import com.adobe.fontengine.font.Font;
import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.FontData;
import com.adobe.fontengine.font.FontException;
import com.adobe.fontengine.font.FontImpl;
import com.adobe.fontengine.font.FontInputStream;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.pdffont.PDFSimpleFont;
import com.adobe.fontengine.font.type1.MetricFile;
import com.adobe.fontengine.font.type1.Type1Font;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontResolver;
import com.adobe.fontengine.fontmanagement.fxg.FXGFontResolverImpl;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontResolver;
import com.adobe.fontengine.fontmanagement.platform.PlatformFontResolverImpl;
import com.adobe.fontengine.fontmanagement.postscript.PSNameFontDatabase;
import com.adobe.fontengine.fontmanagement.postscript.PSNameResolver;
import com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer;

/**
 * Looks at the input and creates Font objects from it.
 *
 */
final public class FontLoader {

	// the recognized font types in AFE
	private static final int kType1Font = 1; // legacy type1
	private static final int kOpenTypeFont = 2; // ttc, otf, or truetype
	private static final int kCFFFont = 3; // naked cff
	private static final int kPFM = 4;
	private static final int kAFM = 5;
	private static final int kMacDFont = 6;
	private static final int kMacRsrcFont = 7;

	private static final int maxBytesNeeded;
	private static final int t1BN;
	private static final int otBN;
	private static final int cffBN;
	private static final int afmBN;
	private static final int pfmBN;
	private static final int macBN;

	private HashMap<String, HashMap<String, FontImpl[]>> mGlobalCacheMap = new HashMap<String, HashMap<String, FontImpl[]>>();
	private HashMap<String, FontImpl[]> mGlobalFontMap = new HashMap<String, FontImpl[]>();

	static {
		// calculate maxBytesNeeded...
		int mbn;

		t1BN = com.adobe.fontengine.font.type1.FontFactory.getNumBytesNeededToIdentify();
		otBN = com.adobe.fontengine.font.opentype.FontFactory.getNumBytesNeededToIdentify();
		cffBN = com.adobe.fontengine.font.cff.FontFactory.getNumBytesNeededToIdentify();
		afmBN = com.adobe.fontengine.font.type1.FontFactory.getNumBytesNeededToIdentifyAFM();
		pfmBN = com.adobe.fontengine.font.type1.FontFactory.getNumBytesNeededToIdentifyPFM();
		macBN = com.adobe.fontengine.font.mac.FontFactory.getNumBytesNeededToIdentify();

		//	  figure out the max number of bytes needed to identify the font
		mbn = pfmBN;
		if (mbn < otBN)
			mbn = otBN;

		if (mbn < cffBN)
			mbn = cffBN;

		if (mbn < t1BN)
			mbn = t1BN;

		if (mbn < afmBN)
			mbn = afmBN;

		if (mbn < macBN)
			mbn = macBN;

		maxBytesNeeded = mbn;
	}

	/**
	 * An interface that allows metric files and outline files to be
	 * aligned via different policies (eg, font name vs file name).
	 */
	static interface MetricFileAlignmentData {
		boolean dataAligns(MetricFileAlignmentData data);
	}

	private static class FontNameAlignmentData implements MetricFileAlignmentData {

		private final String fontName;

		FontNameAlignmentData(String fontName)
		{
			this.fontName = fontName;
		}

		public boolean dataAligns(MetricFileAlignmentData data)
		{
			// if we are aligning on fontnames and the fontnames are the same, the data aligns.
			if (data instanceof FontNameAlignmentData 
					&& ((FontNameAlignmentData)data).fontName.equals(this.fontName))
				return true;

			return false;
		}
	}

	private static class T1AlignmentHolder
	{
		final URLFont font;
		final MetricFileAlignmentData data;

		T1AlignmentHolder(URLFont font, MetricFileAlignmentData data)
		{
			this.font = font;
			this.data = data;
		}
	}

	private static class MetricFileAlignmentHolder
	{
		final URL metricFileURL;
		final MetricFileAlignmentData data;
		final int metricFileType;
		final SoftReference metricFile;

		MetricFileAlignmentHolder(URL metricFileURL, MetricFileAlignmentData data, int metricFileType, MetricFile metricFile)
		{
			this.metricFileURL = metricFileURL;
			this.data = data;
			this.metricFileType = metricFileType;
			this.metricFile = new SoftReference(metricFile);
		}
	}

	/**
	 * Create a new empty <code>PSNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @return A new empty PSNameResolver
	 */
	static public PSNameResolver getPSNameResolverInstance ()
	{
		return new PSNameFontDatabase ();
	}

	/**
	 * Create a new empty <code>PSNameResolver</code>.
	 * 
	 * This method allows a client to specify a 
	 * {@link com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer FamilyNameNormalizer} object
	 * that is used to modify the font family name of every {@link com.adobe.fontengine.font.Font Font} object
	 * added to the PSNameResolver.  It is also used to modify the font names inside of any search request.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @param normalizer A {@link com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer FamilyNameNormalizer} 
	 * used for adjusting font family names.
	 * @return A new empty PSNameResolver
	 */
	static public PSNameResolver getPSNameResolverInstance(FamilyNameNormalizer normalizer)
	{
		return new PSNameFontDatabase(normalizer);	    
	}	

	/**
	 * Create a copy of a <code>PSNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @param original the PSNameResolver to copy
	 * @return A new empty PSNameResolver
	 */
	static public PSNameResolver getPSNameResolverInstance(PSNameResolver original)
	{
		if (original instanceof PSNameFontDatabase)
		{
			return new PSNameFontDatabase((PSNameFontDatabase) original);	    
		} else {
			return null;
		}
	}

	/**
	 * Create a new empty <code>PlatformNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @return A new empty PlatformFontResolver
	 */
	static public PlatformFontResolver getPlatformFontResolverInstance()
	{
		return new PlatformFontResolverImpl();
	}

	/**
	 * Create a copy of a <code>PlatformNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @param original the PlatformFontResolver to copy
	 * @return A new empty PlatformFontResolver
	 */
	static public PlatformFontResolver getPlatformFontResolverInstance(PlatformFontResolver original)
	{
		if (original instanceof PlatformFontResolverImpl)
		{
			return new PlatformFontResolverImpl((PlatformFontResolverImpl) original);	    
		} else {
			return null;
		}
	}

	/**
	 * Create a new empty <code>PlatformNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * 
	 * @param normalizer A {@link com.adobe.fontengine.inlineformatting.css20.FamilyNameNormalizer FamilyNameNormalizer} 
	 * used for adjusting platform font family names.
	 * @return A new empty PlatformFontResolver
	 */
	static public PlatformFontResolver getPlatformFontResolverInstance(FamilyNameNormalizer normalizer)
	{
		return new PlatformFontResolverImpl(normalizer);
	}

	/**
	 * Create a new empty <code>FXGNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @return A new empty PSNameResolver
	 */
	static public FXGFontResolver getFXGFontResolverInstance()
	{
		return new FXGFontResolverImpl();
	}

	/**
	 * Create a copy of a <code>FXGNameResolver</code>.
	 * 
	 * <h4>Concurrency</h4>
	 *  
	 * The instance implementing the <code>PSNameResolver</code> interface that is returned from this method offers some tighter
	 * thread safety guarantees than that guaranteed by the <code>PSNameResolver</code> interface.  It is safe 
	 * to use the <code>PSNameResolver</code>
	 * object for font resolution from multiple threads at one time as long as the methods which
	 * change the state of the <code>PSNameResolver</code> object are not called at any time that 
	 * it may be being used for font resolution.  There is no mechanism 
	 * within <code>PSNameResolver</code> or
	 * <code>FontLoader</code> to guarantee that this doesn't occur.  A client program can guarantee this by modifying the
	 * <code>PSNameResolver</code> object fully before giving it to one or more
	 * threads.  A client program could also use some synchronization constructs of its own to guarantee this.
	 * This means that the <code>addFont()</code> and <code>setResolutionPriority()</code> methods can <b>not</b> 
	 * be called while there is any chance that the <code>PSNameResolver</code>
	 * object may be being used for font resolution.
	 * 
	 * It is also safe to copy this <code>PSNameResolver</code> object except when it is possible that it may be being modified.
	 * It is again the responsibility of the client to ensure that this doesn't happen.
	 * @param original the PSNameResolver to copy
	 * @return A new empty PSNameResolver
	 */
	static public FXGFontResolver getFXGFontResolverInstance(FXGFontResolver original)
	{
		if (original instanceof FXGFontResolverImpl)
		{
			return new FXGFontResolverImpl ((FXGFontResolverImpl) original);	    
		} else {
			return null;
		}
	}

	static private int determineFileType(FontInputStream stream, URL url)
	throws IOException
	{
		int numRead;
		byte[] bytes;


		int fonttype = -1;



		bytes = new byte[maxBytesNeeded];

		// read in the bytes
		numRead = stream.read(bytes);

		// see what kind of font it is
		if (numRead >= otBN 
				&& com.adobe.fontengine.font.opentype.FontFactory.isOpenType(bytes))	    
		{
			fonttype = kOpenTypeFont;
		}
		else if (numRead >= t1BN
				&& com.adobe.fontengine.font.type1.FontFactory.isType1(bytes))
		{
			fonttype = kType1Font;
		}
		// look for metric files...
		else if (numRead >= pfmBN
				&& com.adobe.fontengine.font.type1.FontFactory.isPFM(bytes))
		{
			fonttype = kPFM;
		}
		else if (numRead >= afmBN
				&& com.adobe.fontengine.font.type1.FontFactory.isAFM(bytes))
		{
			fonttype = kAFM;
		}
		// as a last check, see if it looks like a cff font.
		else if (numRead >= cffBN
				&& com.adobe.fontengine.font.cff.FontFactory.isCFF(bytes))
		{
			fonttype = kCFFFont;
		}
		else if (url != null)
		{
			switch (com.adobe.fontengine.font.mac.FontFactory.isResourceFont(bytes, url))
			{
			case com.adobe.fontengine.font.mac.FontFactory.DATA_FORK_FONT:
				fonttype = kMacDFont;
				break;
			case com.adobe.fontengine.font.mac.FontFactory.RESOURCE_FORK_FONT:
				fonttype = kMacRsrcFont;
				break;
			}
		}

		// unread the bytes
		if (numRead > 0)
			stream.unread(bytes, 0, numRead);

		return fonttype;
	}

	/**
	 * Looks to see what find of input url represents. If it is a
	 * metric file, a MetricFileAlignmentHolder is added to metricFiles. If it is a type1 font,
	 * a T1AlignmentHolder is added to t1Fonts. If lastVectorContainsFonts is true, a FontData 
	 * representing it is added to fontOrFontData. If it is false, a Font is added to fontOrFontData.
	 * CFF fonts are ignored by this function. 
	 * @throws FontLoadingException 
	 */
	private static void loadFont(java.net.URL url, 
			List /*<T1AlignmentHolder>*/ t1Fonts, List /*<MetricFileAlignmentHolder>*/ metricFiles,
			List /*<Font>|<FontData>*/ fontOrFontData, boolean lastVectorContainsFontData)
	throws IOException, InvalidFontException, UnsupportedFontException, FontLoadingException
	{
		FontData[] fontarray;
		FontInputStream stream = null;

		try {
			stream = new FontInputStream(url);
			switch (FontLoader.determineFileType(stream, url))
			{
			case kType1Font:
			{
				fontarray = com.adobe.fontengine.font.type1.FontFactory.load(stream, url);
				if (fontarray.length != 1)
					break;

				URLFont urlFont = new URLFont(url, 0, fontarray[0]);
				t1Fonts.add(new T1AlignmentHolder(urlFont, new FontNameAlignmentData(((Type1Font)fontarray[0]).getPostscriptName())));
				if (lastVectorContainsFontData)
					fontOrFontData.add(fontarray[0]);
				else
				{
					fontOrFontData.add(urlFont);
				}
				break;
			}
			case kOpenTypeFont:
			{

				fontarray = com.adobe.fontengine.font.opentype.FontFactory.load(stream);
				for (int i = 0; i < fontarray.length; i++)
				{
					if (lastVectorContainsFontData)
						fontOrFontData.add(fontarray[i]);
					else
					{
						fontOrFontData.add(new URLFont(url, i, fontarray[i]));
					}
				}
				break;
			}

			case kCFFFont:
				// naked cff not handled in a directory
				break;

			case kPFM:
			{
				MetricFile pfm = com.adobe.fontengine.font.type1.FontFactory.loadPFM(stream, url);
				if (pfm != null)
					metricFiles.add(new MetricFileAlignmentHolder(url, new FontNameAlignmentData(pfm.getFontName()), kPFM, pfm));
				break;
			}
			case kAFM:
			{
				MetricFile afm = com.adobe.fontengine.font.type1.FontFactory.loadAFM(stream, url);
				if (afm != null)
					metricFiles.add(new MetricFileAlignmentHolder(url, new FontNameAlignmentData(afm.getFontName()), kAFM, afm));
				break;
			}

			case kMacDFont:
			{
				Font[] fonts = com.adobe.fontengine.font.mac.FontFactory.load(
						url, com.adobe.fontengine.font.mac.FontFactory.DATA_FORK_FONT);
				for (int i = 0; i < fonts.length; i++)
				{
					fontOrFontData.add(fonts[i]);
				}
				break;
			}
			case kMacRsrcFont:
			{
				Font[] fonts = com.adobe.fontengine.font.mac.FontFactory.load(
						url, com.adobe.fontengine.font.mac.FontFactory.RESOURCE_FORK_FONT);
				for (int i = 0; i < fonts.length; i++)
				{
					fontOrFontData.add(fonts[i]);
				}
				break;
			}
			}
		}
		finally
		{
			if (stream != null)
			{
				stream.close();
			}
		}

	}

	/**
	 * Creates Font objects from a URL.
	 * 
	 * Note that fonts may be parsed at this point.
	 * 
	 * @param url The url of the font data. OpenType, TrueType, and Type1 fonts 
	 * are the only acceptable input. In order to have predictible behavior,
	 * url must be available and must point to the same data for the lifetime of the
	 * returned Fonts. If the Font is added to a FontSet that is to be
	 * serialized, the urls must remain unchanged for as long as the serialized data is to be used.
	 * @return An array of Fonts representing the font data. An empty array if 
	 * the font data does not represent a font that can be handled
	 * @throws FontLoadingException thrown if the font data cannot be accessed
	 * @throws InvalidFontException thrown if the font data appears to represent a font, but 
	 * there is something about the data that cannot be understood and cannot be ignored
	 * @throws UnsupportedFontException thrown if the font contains data that indicates that AFE cannot
	 * support that category of fonts.
	 */
	public Font[] load (java.net.URL url) 
	throws FontLoadingException, InvalidFontException, UnsupportedFontException
	{

		try 
		{
			FontInputStream stream = new FontInputStream(url);

			switch (FontLoader.determineFileType(stream, url))
			{
			case kOpenTypeFont:
			{
				FontData[] fonts = com.adobe.fontengine.font.opentype.FontFactory.load(stream);
				Font[] ret = new Font[fonts.length];
				for (int i = 0; i < fonts.length; i++)
					ret[i] = new URLFont(url, i, fonts[i]);
				return ret;
			}

			case kType1Font:
			{
				Font[] ret = new Font[1];
				ret[0] = new URLFont(url, 0);
				return ret;
			}

			case kMacDFont:
			{
				Font[] fonts = com.adobe.fontengine.font.mac.FontFactory.load(
						url, com.adobe.fontengine.font.mac.FontFactory.DATA_FORK_FONT);
				return fonts;
			}
			case kMacRsrcFont:
			{
				Font[] fonts = com.adobe.fontengine.font.mac.FontFactory.load(
						url, com.adobe.fontengine.font.mac.FontFactory.RESOURCE_FORK_FONT);
				return fonts;
			}

			default:
				// we don't handle naked cff or metric files from a url.
				break;
			}

			return new Font[0];
		} catch (IOException e)
		{
			throw new FontLoadingException(e);
		}

	}

	/**
	 * Creates Font objects from an array of URL.
	 * 
	 * Note that fonts are parsed at this point.
	 * 
	 * @param url The url of the font data. OpenType, TrueType and Type1 fonts and
	 * metric files associated with the type1 fonts are the only acceptable input.
	 * In order to have predictible behavior,
	 * url must be available and must point to the same data for the lifetime of the
	 * returned Fonts.  If the Font is added to a FontSet that is to be
	 * serialized, the urls must remain unchanged for as long as the serialized data is to be used.
	 * @param exceptions Any exceptions that are encountered during the loading process will added
	 * to the provided <code>List</code>.  This can be a <code>null</code> if the caller doesn't want to
	 * know about the exceptions that occured. 
	 * @return An array of Fonts representing the font data. An empty array if 
	 * the font data does not represent a font that can be handled
	 */
	public Font[] load(java.net.URL[] url, List exceptions)
	{
		List /* Font */ fonts = new ArrayList();
		List /* T1AlignmentHolder */ t1Fonts = new ArrayList();
		List /* MetricFileAlignmentHolder */ metricFiles = new ArrayList();

		for (int i = 0; i < url.length; ++i)
		{
			try
			{
				loadFont(url[i], t1Fonts, metricFiles, fonts, false);
			} catch (InvalidFontException e)
			{
				if (exceptions != null)
				{
					exceptions.add(e);
				}
			} catch (UnsupportedFontException e)
			{
				if (exceptions != null)
				{
					exceptions.add(e);
				}
			} catch (IOException e)
			{
				if (exceptions != null)
				{
					exceptions.add(new FontLoadingException(e));
				}
			} catch (FontLoadingException e) {
				if (exceptions != null)
				{
					exceptions.add(new FontLoadingException(e));
				}
			}
		}

		try
		{
			alignMetricFilesWithOutlines(metricFiles, t1Fonts);
		} catch (FontEngineException e)
		{
			if (exceptions != null)
			{
				exceptions.add(e);
			}
		} 

		if (fonts.isEmpty())
			return new Font[0];

		Font[] res = new Font [fonts.size ()];
		fonts.toArray (res);
		return res;
	}

	/**
	 * Creates Font objects from an array of URL.
	 * 
	 * Note that fonts are parsed at this point.
	 * 
	 * @param url The url of the font data. OpenType, TrueType and Type1 fonts and
	 * metric files associated with the type1 fonts are the only acceptable input.
	 * In order to have predictible behavior,
	 * url must be available and must point to the same data for the lifetime of the
	 * returned Fonts.  If the Font is added to a FontSet that is to be
	 * serialized, the urls must remain unchanged for as long as the serialized data is to be used.
	 * @param exceptions Any exceptions that are encountered during the loading process will added
	 * to the provided <code>List</code>.  This can be a <code>null</code> if the caller doesn't want to
	 * know about the exceptions that occured. 
	 * @param badUrls The URLs causing any exception are recorded here.  These are in
	 * 1 to 1 alignment with the exceptions returned.  This can be a <code>null</code> if the 
	 * caller doesn't want to know about the bad URLs that occured. 
	 * @return An array of Fonts representing the font data. An empty array if 
	 * the font data does not represent a font that can be handled
	 */
	public Font[] load(java.net.URL[] url, List exceptions, List badUrls)
	{
		List /* Font */ fonts = new ArrayList();
		List /* T1AlignmentHolder */ t1Fonts = new ArrayList();
		List /* MetricFileAlignmentHolder */ metricFiles = new ArrayList();

		for (int i = 0; i < url.length; ++i)
		{
			try
			{
				loadFont(url[i], t1Fonts, metricFiles, fonts, false);
			} catch (FontException e)
			{
				if (exceptions != null)
				{
					exceptions.add(e);
				}
				if (badUrls != null)
				{
					badUrls.add(url[i]);
				}
			} catch (IOException e)
			{
				if (exceptions != null)
				{
					exceptions.add(new FontLoadingException(e));
				}
				if (badUrls != null)
				{
					badUrls.add(url[i]);
				}
			}
		}

		alignMetricFilesWithOutlines(metricFiles, t1Fonts, exceptions, badUrls); 

		if (fonts.isEmpty())
			return new Font[0];

		Font[] res = new Font [fonts.size ()];
		fonts.toArray (res);
		return res;
	}

	/**
	 * Creates a Font object from an InputStream and data describing
	 * the font.
	 * 
	 * The returned Font cannot be serialized. Thus, it must not be added
	 * to a CSS or PS fontset that is intended to be serialized.
	 * @param inputStream The input stream containing the font data. This stream will be read from
	 * to create the <code>Font</code>.  After returning from this method the stream will not
	 * be accessed again and it is the responsibilty of the caller to close this stream.
	 * @param length The number of bytes in the InputStream. 
	 * @param wasEmbedded true iff this font is embedded in a document. This affects the way
	 * this font can be used. See Technote 5147 "Font Embedding Guidelines for Adobe Third-party
	 * Developers" for more information.
	 * 
	 * @return A Font representing the font.
	 */
	public Font load(InputStream inputStream, int length, boolean wasEmbedded) 
	throws FontLoadingException
	{
		try {
			FontInputStream stream = new FontInputStream(inputStream);
			switch (FontLoader.determineFileType(stream, null /*no url*/))
			{
			case kType1Font:
			case kOpenTypeFont:
			case kCFFFont:
				return new StreamFont(stream, length, wasEmbedded);
			default:
				return null;
			}
		} catch (IOException e)	{
			throw new FontLoadingException(e);
		} catch (InvalidFontException e) {
			throw new FontLoadingException(e);
		} catch (UnsupportedFontException e) {
			throw new FontLoadingException(e);
		}
	}

	/**
	 * Creates Font objects for all fonts in a file or directory represented by f.
	 * 
	 * Fonts are parsed at this point. AFE does not track changes in the file system dynamically.
	 * If fonts are added or deleted, the directory must be reloaded to see those changes. Note that
	 * this function does NOT resolve windows shortcuts.
	 * @param f The root directory or file to be read in. 
	 * Only host font formats (OpenType, TrueType, or Type1) will
	 * be processed by this api. In order to have predictible behavior,
	 * the files pointed to by f must be available and must point to the 
	 * same data for the lifetime of the returned Font.If these fonts become part
	 * of serialized font sets, f (and its subdirectories, if they are recursively searched) must be 
	 * accessible from all servers sharing the serialized data. Likewise, the contents of f
	 * must not change for as long as the serialized data is to be used.
	 * @param recurseDirectories if f is a directory, recurseDirectories specifies whether
	 * load should recursively walk all subdirectories. If true, the recursion happens. It
	 * does not if recurseDirectories is false. This parameter is ignored if f is not a directory.
	 * @param exceptions Any exceptions that are encountered during the loading process will added
	 * to the provided <code>List</code>.  This can be a <code>null</code> if the caller doesn't want to
	 * know about the exceptions that occured. 
	 * @return An array of Fonts representing the fonts found. 
	 * An empty array if no fonts are found.
	 */
	public Font[] load(File f, boolean recurseDirectories, List exceptions)
	{
		DirectoryWalker w = new DirectoryWalker(f, recurseDirectories);
		File next; 
		List /*<T1AlignmentHolder>*/ t1Fonts = new ArrayList();
		List /*<MetricFileAlignmentHolder>*/ metricFiles = new ArrayList();
		List /*<Font>*/fonts = new ArrayList();

		while ((next = w.getNextFile()) != null)
		{
			URL url; 
			try
			{
				url = next.toURI().toURL();
				loadFont(url, t1Fonts, metricFiles, fonts, false);
			} catch (FontEngineException e)
			{
				if (exceptions != null)
				{
					exceptions.add(e);
				}
			} catch (IOException e)
			{
				if (exceptions != null)
				{
					exceptions.add(new FontLoadingException(e));
				}
			}
		}

		try
		{
			alignMetricFilesWithOutlines(metricFiles, t1Fonts);
		} catch (FontEngineException e)
		{
			if (exceptions != null)
			{
				exceptions.add(e);
			}
		} 

		if (fonts.size() == 0)
			return new URLFont[0];

		Font[] handles = new Font[fonts.size()];
		fonts.toArray(handles);
		return handles;
	}   

	/**
	 * Creates Font objects for all fonts in a file or directory represented by f.
	 * 
	 * Fonts are parsed at this point. AFE does not track changes in the file system dynamically.
	 * If fonts are added or deleted, the directory must be reloaded to see those changes. Note that
	 * this function does NOT resolve windows shortcuts.
	 * @param f The root directory or file to be read in. 
	 * Only host font formats (OpenType, TrueType, or Type1) will
	 * be processed by this api. In order to have predictible behavior,
	 * the files pointed to by f must be available and must point to the 
	 * same data for the lifetime of the returned Font.If these fonts become part
	 * of serialized font sets, f (and its subdirectories, if they are recursively searched) must be 
	 * accessible from all servers sharing the serialized data. Likewise, the contents of f
	 * must not change for as long as the serialized data is to be used.
	 * @param recurseDirectories if f is a directory, recurseDirectories specifies whether
	 * load should recursively walk all subdirectories. If true, the recursion happens. It
	 * does not if recurseDirectories is false. This parameter is ignored if f is not a directory.
	 * @param exceptions Any exceptions that are encountered during the loading process will added
	 * to the provided <code>List</code>.  This can be a <code>null</code> if the caller doesn't want to
	 * know about the exceptions that occured. 
	 * @param badUrls If any exceptions are encountered during the loading process then the URL which
	 * casued the exception is recorded in this List.  It is in 1 to 1 correlation with the exceptions
	 * returned in the exceptions List.  This can be a <code>null</code> if the caller doesn't want to
	 * know about the bad URLs that occured. 
	 * @return An array of Fonts representing the fonts found. 
	 * An empty array if no fonts are found.
	 */
	public Font[] load(File f, boolean recurseDirectories, List exceptions, List badUrls)
	{
		DirectoryWalker w = new DirectoryWalker(f, recurseDirectories);
		File next; 
		List /*<T1AlignmentHolder>*/ t1Fonts = new ArrayList();
		List /*<MetricFileAlignmentHolder>*/ metricFiles = new ArrayList();
		List /*<Font>*/fonts = new ArrayList();

		while ((next = w.getNextFile()) != null)
		{
			URL url = null; 
			try
			{
				url = next.toURI().toURL();
				loadFont(url, t1Fonts, metricFiles, fonts, false);
			} catch (FontEngineException e)
			{
				if (exceptions != null)
				{
					exceptions.add(e);
				}
				if (badUrls != null)
				{
					badUrls.add(url);
				}
			} catch (IOException e)
			{
				if (exceptions != null)
				{
					exceptions.add(new FontLoadingException(e));
				}
				if (badUrls != null)
				{
					badUrls.add(url);
				}
			}
		}

		alignMetricFilesWithOutlines(metricFiles, t1Fonts, exceptions, badUrls); 

		if (fonts.size() == 0)
			return new URLFont[0];

		URLFont[] handles = new URLFont[fonts.size()];
		fonts.toArray(handles);
		return handles;
	}   

	/**
	 * Creates FontData objects from an InputStream.
	 * 
	 * Note that fonts are parsed at this point.
	 * 
	 * @param stream An input stream containing any of an OpenType font, a CFF font or a Type1 font.
	 * @param size The length of the input data
	 * @return An array of fonts representing the data in the input stream. An empty array if 
	 * the font data does not represent a font that can be handled
	 * @throws IOException thrown if the font data cannot be accessed
	 * @throws InvalidFontException thrown if the font data appears to represent a font, but 
	 * there is something about the data that can be understood or ignored
	 * @throws UnsupportedFontException thrown if the font data contains information the indicates that
	 * AFE does not support the font.
	 */
	static FontData[] fromStream (FontInputStream stream, int size) 
	throws IOException, InvalidFontException, UnsupportedFontException
	{

		switch (FontLoader.determineFileType(stream, null /*no url*/))
		{

		case kOpenTypeFont:
			return com.adobe.fontengine.font.opentype.FontFactory.load(stream);

		case kCFFFont:
			FontByteArray array = new FontByteArray(stream, size);
			return com.adobe.fontengine.font.cff.FontFactory.load(array);

		case kType1Font:
			return com.adobe.fontengine.font.type1.FontFactory.load(stream, null);

		default:
			// we don't handle metric files from an input stream
			break;
		}

		return new FontData[0];
	}


	static private void alignMetricFilesWithOutlines(List /*<MetricFileAlignmentHolder>*/ metricFiles, 
			List /*<T1AlignmentHolder>*/ fonts)
	throws InvalidFontException, UnsupportedFontException, FontLoadingException
	{
		int mfCount;
		int fCount;

		for (mfCount = 0; mfCount < metricFiles.size(); ++mfCount)
		{
			MetricFileAlignmentHolder mf = (MetricFileAlignmentHolder)metricFiles.get(mfCount);

			for (fCount = 0; fCount < fonts.size(); ++fCount)
			{
				T1AlignmentHolder f = (T1AlignmentHolder)fonts.get(fCount);

				// if it looks like the metric file and the outline file align, load them in
				// and set the metric file.
				if (mf.data.dataAligns(f.data))
				{
					MetricFile metricFile;
					try 
					{
						FontInputStream stream = new FontInputStream(mf.metricFileURL);
						if (mf.metricFileType == kPFM)
						{
							metricFile = (MetricFile)mf.metricFile.get();
							if (metricFile == null)
								metricFile = com.adobe.fontengine.font.type1.FontFactory.loadPFM(stream, mf.metricFileURL);
						} else if (mf.metricFileType == kAFM)
						{
							metricFile = (MetricFile)mf.metricFile.get();
							if (metricFile == null)
								metricFile = com.adobe.fontengine.font.type1.FontFactory.loadAFM(stream, mf.metricFileURL);
						} else
						{
							throw new FontLoadingException("unexpected metric file type");
						}
					} catch (IOException e)
					{
						throw new FontLoadingException(e);
					}

					f.font.setMetricURL(mf.metricFileURL, metricFile);
				}
			}
		}
	}

	static private void alignMetricFilesWithOutlines(List /*<MetricFileAlignmentHolder>*/ metricFiles, 
			List /*<T1AlignmentHolder>*/ fonts, List /*<FontException>*/ exceptions, 
			List /*<URL>*/ badUrls)
	{
		int mfCount;
		int fCount;

		for (mfCount = 0; mfCount < metricFiles.size(); ++mfCount)
		{
			MetricFileAlignmentHolder mf = (MetricFileAlignmentHolder)metricFiles.get(mfCount);

			for (fCount = 0; fCount < fonts.size(); ++fCount)
			{
				T1AlignmentHolder f = (T1AlignmentHolder)fonts.get(fCount);

				// if it looks like the metric file and the outline file align, load them in
				// and set the metric file.
				if (mf.data.dataAligns(f.data))
				{
					MetricFile metricFile;
					try 
					{
						FontInputStream stream = new FontInputStream(mf.metricFileURL);
						if (mf.metricFileType == kPFM)
						{
							metricFile = (MetricFile)mf.metricFile.get();
							if (metricFile == null)
								metricFile = com.adobe.fontengine.font.type1.FontFactory.loadPFM(stream, mf.metricFileURL);
						} else if (mf.metricFileType == kAFM)
						{
							metricFile = (MetricFile)mf.metricFile.get();
							if (metricFile == null)
								metricFile = com.adobe.fontengine.font.type1.FontFactory.loadAFM(stream, mf.metricFileURL);
						} else
						{
							throw new FontLoadingException("unexpected metric file type");
						}

						f.font.setMetricURL(mf.metricFileURL, metricFile);
					} catch (IOException e)
					{
						if (exceptions != null)
						{
							exceptions.add(new FontLoadingException(e));
						}
						if (badUrls != null)
						{
							badUrls.add(f.font.outlineFileURL);
						}
					}
					catch (FontException e)
					{
						if (exceptions != null)
						{
							exceptions.add(e);
						}
						if (badUrls != null)
						{
							badUrls.add(f.font.outlineFileURL);
						}
					}
				}
			}
		}
	}

	/**
	 * Creates FontData objects from an array of URLs.
	 * 
	 * Note that fonts are parsed at this point.
	 * 
	 * @param outlineFileURL The URL of the font data. OpenType and Type1 fonts 
	 * are the only acceptable input
	 * @param metricFileURL The URL of a metric file (for Type1 fonts)
	 * @return An array of fonts representing the font data. An empty array if 
	 * the font data does not represent a font that can be handled
	 * @throws FontLoadingException
	 * @throws UnsupportedFontException
	 * @throws InvalidFontException
	 */
	static FontData[] fromURL(java.net.URL outlineFileURL, java.net.URL metricFileURL) 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException, IOException
	{
		List /*<FontData>*/ fonts = new ArrayList();
		List /*<T1AlignmentHolder>*/ t1Fonts = new ArrayList();
		List /*<MetricFile*/ metricFiles = new ArrayList();

		loadFont(outlineFileURL, t1Fonts, metricFiles, fonts, true);

		if (metricFileURL != null)
			loadFont(metricFileURL, t1Fonts, metricFiles, fonts, true);

		alignMetricFilesWithOutlines(metricFiles, t1Fonts);

		if (fonts.isEmpty())
			return new FontData[0];

		FontData[] res = new FontData [fonts.size ()];
		fonts.toArray (res);
		return res;
	}

	/**
	 * Creates a Font object from a PDF /Font object with subtype /Type1.
	 * 
	 * The returned Font cannot be serialized. Thus, it must not be added
	 * to a CSS or PS fontset that is intended to be serialized.
	 * 
	 * @param accessor an object through which the PDF keys will be accessed.
	 * 
	 * @return A Font representing the font.
	 */

	public Font load (PDFSimpleFontValuesAccessor accessor) 
	throws InvalidFontException, UnsupportedFontException
	{
		return new MemoryFont (new PDFSimpleFont (accessor));
	}

	/*package protected*/ static FontData fromResourceURL(URL resourceURL, int type, int fontID, int fondID) 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException 
	{
		return com.adobe.fontengine.font.mac.FontFactory.load(resourceURL, type, fontID, fondID);
	}

	/**
	 * The new font cache support code starts here, although some supporting changes have also been
	 * made in code above here. The first is the specialized "load" API for use with the font cache.
	 * It is similar to the other load APIs except that it takes an additional argument to pass in
	 * the serialized cache file against which this load operation is to take place. It is
	 * synchronized so that two load operations can not occur in parallel against the same
	 * FontLoader object.
	 */
	public synchronized Font[] load(File f, boolean recurseDirectories, List exceptions, List badUrls, File cacheFile)
	{
		DirectoryWalker w = new DirectoryWalker(f, recurseDirectories);
		File next;
		List /*<T1AlignmentHolder>*/ t1Fonts = new ArrayList();
		List /*<MetricFileAlignmentHolder>*/ metricFiles = new ArrayList();
		List /*<Font>*/fonts = new ArrayList();
		HashMap<String, FontImpl[]> ourCache = null;
		try {
			ourCache = initializeCacheFile(cacheFile);
		} catch (IOException e) {
			if (exceptions != null) {
				exceptions.add(e);
			}
		}
		while ((next = w.getNextFile()) != null)
		{
			URL url = null;
			try {
				if (ourCache != null) {
					String canonicalPath = next.getCanonicalPath();
					FontImpl[] localFileNode = ourCache.get(canonicalPath);
					FontImpl[] globalFileNode = mGlobalFontMap.get(canonicalPath);
					globalFileNode = mergeFileNodes(next, localFileNode, globalFileNode);
					if (globalFileNode == null) {
						url = next.toURI().toURL();
						int oldSize = fonts.size();
						try {
							loadFont(url, t1Fonts, metricFiles, fonts, false);
						} catch (FontEngineException e) {
							if (exceptions != null) {
								exceptions.add(e);
							}
							if (badUrls != null) {
								badUrls.add(url);
							}
						}
						globalFileNode = new FontImpl[fonts.size() - oldSize];
						for (int i = 0; i < fonts.size() - oldSize; i++) {
							FontImpl fontImpl = (FontImpl)fonts.get(oldSize + i);
							globalFileNode[i] = fontImpl;
						}
					} else {
						for (int i = 0; i < globalFileNode.length; i++) {
							FontImpl fontImpl = globalFileNode[i];
							fonts.add(fontImpl);
						}
					}
					ourCache.put(canonicalPath, globalFileNode);
					mGlobalFontMap.put(canonicalPath, globalFileNode);
					for (HashMap<String, FontImpl[]> oneCache : mGlobalCacheMap.values()) {
						if (oneCache != ourCache && oneCache.containsKey(canonicalPath)) {
							oneCache.put(canonicalPath, globalFileNode);
						}
					}
				} else {
					url = next.toURI().toURL();
					loadFont(url, t1Fonts, metricFiles, fonts, false);
				}
			} catch (FontEngineException e) {
				if (exceptions != null) {
					exceptions.add(e);
				}
				if (badUrls != null) {
					badUrls.add(url);
				}
			} catch (IOException e) {
				if (exceptions != null) {
					exceptions.add(new FontLoadingException(e));
				}
				if (badUrls != null) {
					badUrls.add(url);
				}
			}
		}
		alignMetricFilesWithOutlines(metricFiles, t1Fonts, exceptions, badUrls);
		if (fonts.size() == 0)
			return new URLFont[0];
		Font[] handles = new Font[fonts.size()];
		fonts.toArray(handles);
		return handles;
	}

	/**
	 * This encapsulates the readin of a serialized cache file. The call can give four
	 * possilbe results. If the cache file's path points to a cache file that is already
	 * in the map of caches we simply return that cache. If the cache file is null or
	 * is malformed in some way we return a new, empty cache. If the cache file has
	 * access problems we throw an IOException because there will never be any way
	 * to save the cache file even if we create an empty one. Otherwise we deserialize
	 * the cache file passed and return the resulting cache object.
	 */
	private HashMap<String, FontImpl[]> initializeCacheFile(File cacheFile)
		throws IOException
	{
		HashMap<String, FontImpl[]> fontCache = null;
		if (cacheFile == null)
			return fontCache;
		if (!(cacheFile.exists() || cacheFile.createNewFile()))
			throw new IOException("Cannot create font cache");
		if (!cacheFile.canRead())
			throw new IOException("Cannot read font cache");
		if (!cacheFile.canWrite())
			throw new IOException("Cannot write font cache");
		String cachePath = cacheFile.getCanonicalPath();
		if (mGlobalCacheMap.containsKey(cachePath)) {
			fontCache = mGlobalCacheMap.get(cachePath);
		} else {
			if (cacheFile.length() != 0) {
				FileInputStream fileInStm = null;
				ObjectInputStream objInStm = null;
				try {
					fileInStm = new FileInputStream(cacheFile);
					objInStm = new ObjectInputStream(fileInStm);
					fontCache = (HashMap<String, FontImpl[]>)objInStm.readObject();
				} catch (ClassNotFoundException e) {
				} catch (IOException e) {
				} finally {
					if (objInStm != null) {
						objInStm.close();
					} else if (fileInStm != null) {
						fileInStm.close();
					}
				}
			}
			if (fontCache == null) {
				// Empty, invalid or obsolete
				cacheFile.delete();
				cacheFile.createNewFile();
				fontCache = new HashMap<String, FontImpl[]>();
			}
			mGlobalCacheMap.put(cachePath, fontCache);
		}
		return fontCache;
	}

	/**
	 * This encapsulates the merging of one "file node" from a newly imput cache file to the global cache.
	 * It checks the new cache node against the file's information to make sure it is current and if it is
	 * it merges any possibly missing FontDescriptions from the new node to the global one. A "file node"
	 * is an array of FontImpl objects, as many as are manufactured from a single font file path.
	 */
	private FontImpl[] mergeFileNodes(File fontFile, FontImpl[] localFileNode, FontImpl[] globalFileNode)
		throws IOException
	{
		if (localFileNode != null && localFileNode.length > 0) {
			FontImpl implZero = localFileNode[0];
			if (implZero.getLength() != fontFile.length() || implZero.getLastModified() != fontFile.lastModified()) {
				localFileNode = null;
			}
		}
		if (globalFileNode == null)
			return localFileNode;
		if (localFileNode != null) {
			if (localFileNode.length != globalFileNode.length)
				return null;
			for (int i = 0 ; i < localFileNode.length; i++) {
				FontImpl localImpl = localFileNode[i];
				FontImpl globalImpl = globalFileNode[i];
				for (String key : localImpl.getCachedFontDescriptionMap().keySet()) {
					if (globalImpl.getCachedFontDescription(key) == null) {
						globalImpl.setCachedFontDescription(key, localImpl.getCachedFontDescription(key));
					}
				}
			}
		}
		return globalFileNode;
	}

	/**
	 * The cache files MUST be explictly save by the client. They are never saved automatically.
	 * It might be possible to use "finalize" for this in some environments but on servers where
	 * nothing ever terminates until the server crashes this won't work. It could also be hooked
	 * up to some daemon, but that is the client's job, not AFEs. The "clean" boolean throws away
	 * entries that were never loaded against before saving. This gets rid of "stale" entries for
	 * fonts that may have been deleted since the previous cycle of operation.
	 */
	public synchronized void saveAllCaches(boolean clean)
		throws IOException
	{
		for (String cachePath : mGlobalCacheMap.keySet()) {
			File cacheFile = new File(cachePath);
			saveCache(cacheFile, clean);
		}
	}

	/**
	 * The inner loop, publically available, that saves just one cache file
	 */
	public synchronized boolean saveCache(File cacheFile, boolean clean)
		throws IOException
	{
		String cachePath = cacheFile.getCanonicalPath();
		HashMap<String, FontImpl[]> oneCache = mGlobalCacheMap.get(cachePath);
		if (oneCache != null) {
			if (clean) {
				Iterator<String> iter = oneCache.keySet().iterator();
				while (iter.hasNext()) {
					String key = iter.next();
					if (!mGlobalFontMap.containsKey(key)) {
						iter.remove();
					}
				}
			}
			cacheFile.delete();
			cacheFile.createNewFile();
			FileOutputStream fileOutStm = new FileOutputStream(cacheFile);
			ObjectOutputStream objOutStm = new ObjectOutputStream(fileOutStm);
			objOutStm.writeObject(oneCache);
			objOutStm.close();
			return true;
		}
		return false;
	}

	/**
	 * This removes a single font file from all caches. This is the API clients would
	 * use if they have some list of specific fonts that they want to delete or replace.
	 */
	public synchronized boolean decommissionCachedFont(File fontFile)
		throws IOException
	{
		String fontPath = fontFile.getCanonicalPath();
		boolean removed = mGlobalFontMap.remove(fontPath) != null;
		if (removed) {
			for (HashMap<String, FontImpl[]> oneCache : mGlobalCacheMap.values()) {
				oneCache.remove(fontPath);
			}
		}
		return removed;
	}

	/**
	 * This provides clients a way to remove caches from service. It is orthogonal to
	 * saving but if you also save you need to do the saving first. It simply clears
	 * all the caches and then reloads the builtin fonts to get a clean, initialized
	 * state.
	 */
	public synchronized void decommissionAllCaches()
	{
		mGlobalCacheMap.clear();
		mGlobalFontMap.clear();
	}

	/**
	 * This allows a single cache to be separately decomissioned. It throws away the
	 * local cache referred to and removes all corresponding entries in the global cache.
	 * It may be quite slow in order to maintain consistency in other local caches. Use
	 * decommissionAllCaches() when possible.
	 */
	public synchronized boolean decommissionCache(File cacheFile)
	{
		try {
			String cachePath = cacheFile.getCanonicalPath();
			HashMap<String, FontImpl[]> oneCache = mGlobalCacheMap.remove(cachePath);
			if (oneCache != null) {
				for (String fontPath : oneCache.keySet()) {
					boolean foundOtherReference = false;
					for (HashMap<String, FontImpl[]> otherCache : mGlobalCacheMap.values()) {
						if (otherCache.containsKey(fontPath)) {
							foundOtherReference = true;
							break;
						}
					}
					if (!foundOtherReference) {
						mGlobalFontMap.remove(fontPath);
					}
				}
				return true;
			}
		} catch (IOException e) {
		}
		return false;
	}
}
