/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2009 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


import java.util.ArrayList;
import java.util.List;

import com.adobe.xfa.ut.FindBugsSuppress;


/**
 * @exclude from published api.
 */
final class SaveNameSpaceChecker {
	
	private final static class NameSpaceOwner {
		
		private final Element mOwner;
		private final String maUri;
		
		
		public NameSpaceOwner(Element owner, String uri) {
			this.mOwner = owner;
			this.maUri = uri;
		}

		public Element getOwner() {
			return mOwner;
		}

		public String getUri() {
			return maUri;
		}
	}

	private final static class NameSpaceHolder {
		
		private final List<NameSpaceOwner> mNameSpaceOwners = new ArrayList<NameSpaceOwner>();
		private final String maPrefix;
		

		public NameSpaceHolder(String aPrefix) {
			maPrefix = aPrefix;
		}		

		public List<NameSpaceOwner> getNamespaceOwners() {
			return mNameSpaceOwners;
		}
		
		public String getPrefix() {
			return maPrefix;
		}
	}
	

	private final Node mStopNode;
	private final List<NameSpaceHolder> mPrefixes = new ArrayList<NameSpaceHolder>();

	/**
	 * stop searching for namespace prefixes
	 * one node beyond the node being saved
	 */
	public SaveNameSpaceChecker(Node oSaveNode) {
		if (oSaveNode != null)
			oSaveNode = Element.getXMLParent(oSaveNode);
		
		mStopNode = oSaveNode;
	}

	public Node stopNode() {
		return mStopNode;
	}
	
    @FindBugsSuppress(code="ES")
    // This method is named exactly backwards, but it kept as is for consistency with the C++.
	public boolean missingPrefix(Element element, String aPrefix, String aNameSpaceURI, boolean bSuppressEmptyNSCheck) {
		boolean bEmpty = aPrefix == "" && aNameSpaceURI == "";
		// don't bother emitting an extraneous xmlns="" with attributes
		// or when "xml" namespace is used
		if (bEmpty && (aNameSpaceURI == STRS.XMLNSURI || aPrefix == STRS.LOWERXMLSTR)) {
			return true;
		}
		else {
			NameSpaceHolder nameSpaceHolder = locatePrefix(aPrefix);
			if (nameSpaceHolder == null || nameSpaceHolder.getNamespaceOwners().size() == 0) {
				//
				// this suppresses putting out an extraneous xmlns="" on all nodes
				//
				// bSuppressEmptyNSCheck is only set to true when checking for redundant namespaces
				// that is, when this method is called from AttrImpl::displayAttr
				//
				if (bEmpty && bSuppressEmptyNSCheck) {
					return true;
				}
			}
			else {
				List<NameSpaceOwner> nameSpaceOwners = nameSpaceHolder.getNamespaceOwners();
				for (int i = nameSpaceOwners.size() - 1; i >= 0; i--) {
					Element checkElement = element;
					NameSpaceOwner nameSpaceNode = nameSpaceOwners.get(i);

					// check to see if nameSpaceOwner is an ancestor node
					// and if it is, check to see if the namespace URIs
					// are the same
					while (checkElement != mStopNode) {
						if (checkElement == nameSpaceNode.getOwner()) {
							return nameSpaceNode.getUri() == aNameSpaceURI;
						}
						
						checkElement = Element.getXMLParent(checkElement);
					}
				}
			}
		}
		return false;
	}

	public void addPrefix(Element element, String aPrefix, String aNamespace) {
		NameSpaceHolder nameSpaceHolder = locatePrefix(aPrefix);
		if (nameSpaceHolder == null) {
			nameSpaceHolder = new NameSpaceHolder(aPrefix);
			mPrefixes.add(nameSpaceHolder);
		}
		
		nameSpaceHolder.getNamespaceOwners().add(new NameSpaceOwner(element, aNamespace));
	}

	public void removePrefix(Element element, String aPrefix) {
		NameSpaceHolder nameSpaceHolder = locatePrefix(aPrefix);
		if (nameSpaceHolder != null) {
			List<NameSpaceOwner> nameSpaceOwner = nameSpaceHolder.getNamespaceOwners();
			for (int i = nameSpaceOwner.size() - 1; i >= 0; i--) {
				if (element == nameSpaceOwner.get(i).getOwner()) {
					nameSpaceOwner.remove(i);
					break;
				}
			}
		}
	}

	@FindBugsSuppress(code="ES")
	private NameSpaceHolder locatePrefix(String aPrefix) {
		final int nSize = mPrefixes.size();
		for (int i = 0; i < nSize; i++) {
			NameSpaceHolder nameSpaceHolder = mPrefixes.get(i);
			if (aPrefix == nameSpaceHolder.getPrefix()) {
				return nameSpaceHolder;
			}
		}
		return null;
	}
}