/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import java.util.List;
import java.util.ArrayList;

import org.apache.jackrabbit.util.Text;

/**
 * Implements a version.
 */
public class Version implements Comparable<Version> {

    /**
     * The empty version
     */
    public static final Version EMPTY = new Version("", new String[0]);

    /**
     * internal string representation
     */
    private final String str;

    /**
     * All segments of this version
     */
    private final String[] segments;

    /**
     * Creates a new version from the given string.
     * @param str the version string.
     * @return the new version or {@link Version#EMPTY} if <code>str</code> is an empty string.
     */
    public static Version create(String str) {
        if (str == null || str.length() == 0) {
            return Version.EMPTY;
        }else {
            return new Version(str, Text.explode(str, '.'));
        }
    }

    /**
     * Creates a new version from version segments
     * @param segments version segments
     * @return the new version or {@link Version#EMPTY} if <code>segments</code> is empty.
     */
    public static Version create(String[] segments) {
        if (segments == null || segments.length == 0) {
            return Version.EMPTY;
        } else {
            return new Version(Text.implode(segments, "."), segments);
        }
    }

    /**
     * Internal constructor
     * @param str string
     * @param segments segments
     */
    private Version(String str, String[] segments) {
        if (str == null) {
            throw new NullPointerException("Version String must not be null.");
        }
        this.str = str;
        List<String> list = new ArrayList<String>();
        for (String seg: segments) {
            for (String sub: Text.explode(seg, '-')) {
                list.add(sub);
            }
        }
        this.segments = list.toArray(new String[list.size()]);
    }

    @Override
    public int hashCode() {
        return str.hashCode();
    }

    @Override
    public boolean equals(Object o) {
        return this == o ||
                o instanceof Version && str.equals(((Version) o).str);

    }

    @Override
    public String toString() {
        return str;
    }

    /**
     * Returns all segments.
     * @return all segments.
     */
    public String[] getNormalizedSegments() {
        return segments;
    }

    /**
     * Compares this version to the given one, segment by segment with a special
     * &quot;SNAPSHOT&quot; handling.
     *
     * Examples:<br>
     * &quot;1&quot; &lt; &quot;2&quot;<br>
     * &quot;1.0&quot; &lt; &quot;2&quot;<br>
     * &quot;2.0.1&quot; &lt; &quot;2.1&quot;<br>
     * &quot;2.1&quot; &lt; &quot;2.1.1&quot;<br>
     * &quot;2.1&quot; &lt; &quot;2.1-SNAPSHOT&quot;<br>
     * &quot;2.1&quot; &lt; &quot;2.1-R1234556&quot;<br>
     * &quot;2.1-R12345&quot; &lt; &quot;2.1-SNAPSHOT&quot;
     *
     * @param o the other version
     * @return  a negative integer, zero, or a positive integer as this version
     *		is less than, equal to, or greater than the specified version.
     */
    public int compareTo(Version o) {
        String[] oSegs = o.getNormalizedSegments();
        for (int i=0; i< Math.min(segments.length, oSegs.length); i++) {
            int comp = segments[i].compareTo(oSegs[i]);
            if (comp != 0) {
                return comp;
            }
        }
        return segments.length - oSegs.length;
    }
}