/*
 * Copyright 1997-2009 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.dam.api;

import org.apache.sling.api.resource.ValueMap;

import javax.jcr.version.Version;
import java.util.Calendar;

/**
 * Defines the interface of a CQ asset revision. The <code>Revision</code> wraps a {@link javax.jcr.version.Version},
 * providing easy access to the contained content (properties) and convenience methods for accessing often used asset
 * properties, such as e.g. the title.
 * <p>
 * Revisions of an asset are created via the <code>Asset</code> interface. Access to existing versions and restoring
 * versions are handled via the <code>AssetManager</code>.
 *
 * @see com.day.cq.dam.api.Asset#createRevision(String, String)
 */
public interface Revision {

    /**
     * Returns the underlying version.
     * @return the JCR version
     * @since 5.5
     */
    Version getVersion();

    /**
     * Returns a {@link org.apache.sling.api.resource.ValueMap} containing the asset's main content properties as frozen
     * in this revision (formerly the <i>&lt;asset-path&gt;/jcr:content</i> properties).
     *
     * @return a value map
     */
    ValueMap getProperties();

    /**
     * Returns a {@link org.apache.sling.api.resource.ValueMap} containing the asset's metadata properties as frozen in
     * this revision (formerly the <i>&lt;asset-path&gt;/jcr:content/metadata</i> properties).
     *
     * @return a value map
     * @see com.day.cq.dam.api.Asset#getMetadata()
     * @see com.day.cq.dam.api.Asset#getMetadata(String)
     * @see com.day.cq.dam.api.Asset#getMetadataValue(String)
     */
    ValueMap getMetadataProperties();

    /**
     * Returns a {@link org.apache.sling.api.resource.ValueMap} containing the content properties of the node addressed
     * via the <code>relPath</code> parameter. The <code>relPath</code> parameter must be a relative path (must not
     * start with a "/").
     * <p>
     * For example the content properties of one of the renditions contained in the asset's revision would be addressed
     * as follows:
     * <blockquote><pre>
     *  ...
     *  ValueMap properties = revision.getProperties("renditions/original/jcr:content");
     *  ...
     * </pre></blockquote>
     *
     * @param relPath relative path to the content
     * @return a value map
     * @throws IllegalArgumentException Thrown if the given path is not relative.
     */
    ValueMap getProperties(String relPath);

    /**
     * Return the concatenated labels of the history for this version, as entered by the user during version creation,
     * or the value of {@link javax.jcr.version.Version#getName()} if no history labels are present.
     *
     * @return version label
     */
    String getLabel();

    /**
     * Return the unique version id (as provided by {@link javax.jcr.version.Version#getIdentifier()}).
     *
     * @return version id
     */
    String getId();

    /**
     * Return a flag indicating whether this revision belongs to a deleted asset.
     *
     * @return <code>true</code> if the associated asset is deleted; <code>false</code> otherwise
     */
    boolean isDeleted();

    /**
     * Returns the name of the asset. This is the last path segment of the asset path (file name).
     *
     * @return asset name
     */
    String getName();

    /**
     * Return the parent path of the asset.
     *
     * @return parent path
     */
    String getParentPath();

    /**
     * Returns the <code>dc:title</code> of the asset or <code>null</code> if none defined.
     *
     * @return title of the asset
     */
    String getTitle();

    /**
     * Returns the description of this resource.
     *
     * @return the description of this resource or <code>null</code> if not defined.
     */
    String getDescription();

    /**
     * Return the creation date.
     *
     * @return creation date
     */
    Calendar getCreated();

    /**
     * Return the optional comment for this revision.
     *
     * @return comment or <code>null</code>
     */
    String getComment();

    /**
     * Return the path to the rendition
     *
     * @param renditionName name of rendition
     * @return path of rendition or <code>null</code> if path does not exist
     */
    String getRenditionPath(String renditionName);
}
