/*************************************************************************
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2014 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/

package com.adobe.granite.translation.api;

import java.io.InputStream;
import java.util.Date;
import java.util.Map;

import com.adobe.granite.comments.Comment;
import com.adobe.granite.comments.CommentCollection;
import com.adobe.granite.translation.api.TranslationConstants.TranslationMethod;
import com.adobe.granite.translation.api.TranslationConstants.TranslationStatus;
import org.osgi.annotation.versioning.ProviderType;

//
/**
 * The TranslationService interface provides a single source for all translation functionality. It is made up of two
 * parts. The (1.0) API provides synchronous machine translation based functionality. The (2.0) API now also provides
 * asynchronous translation functionality to serve a wider range of project based needs.
 * <p>
 * The (1.0) API legacy translation interface provides functionality for synchronous translation, language detection,
 * and Translation Memory (TM) Updates. This interface is implemented by Translation Connectors to provide
 * connectivity to external MT engines. The Connectors contain the engine specific logic to authenticate against and
 * call their respective MT engines.
 * </p>
 * <p>
 * The 1.0 synchronous service is implemented in three parts.
 * </p>
 * <p>
 * <b>Helper methods</b> that make Machine Translation Possible
 * </p>
 * <ul>
 * <li><b>isServiceAvailable:</b> check if a Machine Translation Service is connected and available for use.
 * <li><b>isDirectionSupported:</b> check if a Machine Translation Service can translate between a particular source
 * and target pair in that direction.
 * <li><b>detectLanguage:</b> determine the language identity (French, German, etc.) of a source string.
 * </ul>
 * <p>
 * <b>Translation</b> methods that do the actual translation
 * </p>
 * <ul>
 * <li><b>translateString:</b> translate a single string from source language into target language
 * <li><b>translateArray:</b> translate an array of strings from one source language into one target language
 * </ul>
 * <p>
 * <b>Storage and Retrieval</b> methods that store translation with an MT provider and retrieve those translations.
 * </p>
 * <ul>
 * <li><b>getStoredTranslations:</b> send a source string for translation and retrieve existing ranked translations
 * for that source string.
 * <li><b>addStoredTranslation:</b> push an target translation for a particular source string into a services storage
 * repository. (string and array versions)
 * </ul>
 * <p>
 * The 2.0 asynchronous service is implemented in 3 parts.
 * </p>
 * <p>
 * <b>Translation Content</b> that provides objects for translation and supporting that translation
 * </p>
 * <ul>
 * <li><b>uploadTranslationObject:</b> Upload a unique Translation Object (TO) for translation.
 * <li><b>getTranslatedObject:</b> Download (Returns input stream of the now translated) TO.
 * <li><b>getTranslationObjectStatus:</b> Used to inquire on the current STATUS of a TO.
 * <li><b>getTranslationObjectsStatus:</b> Used to inquire on the current STATUS of a set of TOs.
 * <li><b>updateTranslationObjectState:</b> Change the STATE of a TO to direct it through the translation process.
 * <li><b>updateTranslationObjectsState:</b> Change the STATE of multiple TOs to direct them through the translation
 * process.
 * </ul>
 * <p>
 * <b>Translation Packaging</b> that provides containers for organizing and delivering objects for translation.
 * </p>
 * <ul>
 * <li><b>createTranslationJob:</b> Create a new Translation Job (TJ) for a set of Translation Objects.
 * <li><b>getFinalScope:</b> Returns the SCOPE (word count, size, etc.) of a SUBMITTED TJ
 * <li><b>updateTranslationJobMetadata</b> Modify Metadata attached to a Translation Job (TJ)
 * <li><b>updateTranslationJobState:</b> Modify the STATE of a Translation Job (TJ)
 * </ul>
 * <p>
 * <b>Translation Communication</b> that provides tools for using comments and metadata with the objects and job.
 * </p>
 * <ul>
 * <li><b>addTranslationJobComment:</b> Add a new comment to a translation job.
 * <li><b>addTranslationObjectComment:</b> Add a new comment to a translation object.
 * <li><b>getTranslationJobCommentCollection:</b> Used to gather all the comments attached to a particular TJ.
 * <li><b> :</b> Used to return the current STATUS of a particular TJ.
 * <li><b>getTranslationObjectCommentCollection:</b> Used to gather the comments attached to a particular TO.
 * </ul>
 */
@ProviderType
public interface TranslationService {

    /**
     * Returns the full list of supported language codes for this service. This is a comprehensive list of all
     * supported languages. Use supportedLanguages to determine which languages are supported by a TranslationService.
     * It returns a human readable name for the languages as well a language code for those languages. The service has
     * the capability to recognize these codes and the language named. These are the languages which detectLanguage
     * can recognize. It doesn't not necessarily mean the service can translate between all these languages. Use
     * isDirectionSupported (see below) to retrieve the subset of direction pairs.
     * @return {@code Map<String, String>} containing the full list of supported languages and their English names.
     */
    Map<String, String> supportedLanguages();

    /**
     * Returns true/false whether the current service supports the specified translation direction. At it's simplest a
     * translation service consists of a language pair and a direction. A language pair is any two languages. For
     * example: French and English. A direction is which language is the Source language and which language is the
     * Target language. For example: I want to translate from English to French, so English would be the Source and
     * French would be the Target. If the service supports translation from English to French that would be one
     * direction that was supported. Similarly if the service supports translation from French to English that would
     * be another direction. Use isDirectionSupported to learn if the required language pair and direction is
     * available.
     * @param sourceLanguage The language code to be translated from
     * @param targetLanguage The language code to be translated to
     * @return Returns true if engine supports direction, false otherwise
     * @throws TranslationException the translation exception
     */
    boolean isDirectionSupported(String sourceLanguage, String targetLanguage) throws TranslationException;

    /**
     * Detects the language of the passed in String (accepts both PLAIN and HTML text). Use to determine the language
     * of piece of content. For example: A forum user comments on a discussion thread in their native script. The
     * forum administrator would like to provide a translation of this comment for all the users, but doesn't
     * recognize the language it was posted in. The method detectLanguage makes known the language ID of the comment
     * in question so it may be translated.
     * @param detectSource The content to determine the language of
     * @param contentType The content type of the source (@see TranslationConstants.ContentTypes)
     * @return Returns the language code of the passed in string (language or locale (if known)) Returns "UNKNOWN" if
     *         language cannot be determined (@see TranslationException .ErrorCodes#UNKNOWN_LANGUAGE)
     * @throws TranslationException the translation exception
     */
    String detectLanguage(String detectSource, TranslationConstants.ContentType contentType)
        throws TranslationException;

    /**
     * Translates a given String and returns the best fit (highest rated) translation. The connector may determine a
     * specific engine to use based on the input parameters. This is the core of the translation API. A user calls the
     * translateString method to send a Source string for translation. The translation (the Target String) is then
     * returned to the user.
     * @param sourceString The string to be translated
     * @param sourceLanguage (optional) The string representing the FROM language code. This allows a NULL value to be
     *            passed. If NULL, the source language is determined by leveraging the "detectLanguage" function. A
     *            TranslationException with ErrorCode UNKNOWN_LANGUAGE will be thrown if it cannot be determined.
     * @param targetLanguage The string representing the TO language code.
     * @param contentType The content type of the source (@see TranslationConstants.ContentTypes)
     * @param contentCategory (optional) The content category of the source (@see TranslationConstants.Categories)
     *            This will use the default value if NULL or blank is passed.
     * @return A translation of the "sourceString" String in the "targetLanguage" (best match)
     * @throws TranslationException the translation exception
     */
    TranslationResult translateString(String sourceString, String sourceLanguage, String targetLanguage,
        TranslationConstants.ContentType contentType, String contentCategory) throws TranslationException;

    /**
     * Translates all strings in a given String Array and returns the best fit (highest rated) translations. The
     * connector may determine a specific engine to use based on the input parameters. This function only allows one
     * direction (TO/FROM pair). This method is like translateString with one difference. The translateArray method
     * sends an Array of strings out for translation at the same time. Translations are returned for each string in
     * the array. This allows the user to batch process strings for translation.
     * @param sourceStringArr An Array of Strings to be translated
     * @param sourceLanguage (optional) The string representing the FROM language code. This allows a NULL value to be
     *            passed. If NULL, the source language is determined by leveraging the "detectLanguage" function. A
     *            TranslationException with ErrorCode UNKNOWN_LANGUAGE will be thrown if it cannot be determined.
     *            NOTE: Only the first array position will be used to determine a NULL or blank FROM value.
     * @param targetLanguage The string representing the TO language code.
     * @param contentType The content type of the source (@see TranslationConstants.ContentTypes)
     * @param contentCategory (optional) The content category of the source (@see TranslationConstants.Categories)
     *            This will use the default value if NULL or blank is passed.
     * @return An array of TranslationReturnObjects in the same order as the sourceStringArr
     * @throws TranslationException the translation exception
     */
    TranslationResult[] translateArray(String[] sourceStringArr, String sourceLanguage, String targetLanguage,
        TranslationConstants.ContentType contentType, String contentCategory) throws TranslationException;

    /**
     * Returns all available translation responses for a given string, array will be sorted by rating (best fit first,
     * in "0"). This function is only available if this connector leverages a Translation Memory system. Use
     * getAllStoredTranslations when the user would like to retrieve not only the highest ranked Machine Translated
     * options but also any other alternate translations as they are ranked. These stored Translation Memory options
     * are rated by the admin/moderator. This permits service admin/moderator to better control the translations
     * returned by the Machine Translation Service. It also may be used to support the moderation of crowd sourced and
     * edited translations.
     * @param sourceString The string to be translated
     * @param sourceLanguage (optional) The string representing the FROM language code. This allows a NULL value to be
     *            passed. If NULL, the source language is determined by leveraging the "detectLanguage" function. A
     *            TranslationException with ErrorCode UNKNOWN_LANGUAGE will be thrown if it cannot be determined.
     * @param targetLanguage The string representing the TO language code.
     * @param contentType The content type of the source (@see TranslationConstants.ContentTypes)
     * @param contentCategory (optional) The content category of the source (@see TranslationConstants.Categories)
     *            This will use the default value if NULL or blank is passed.
     * @param userId (optional) filters the results specific to a passed in User
     * @param maxTranslations (optional) Specifies the maximum number of translation response Strings in array. If not
     *            specified ALL available translations will be returned.
     * @return an array of TranslationResult objects which contain all translations stored in the Machine Translation
     *         store
     * @throws TranslationException the translation exception
     */
    TranslationResult[] getAllStoredTranslations(String sourceString, String sourceLanguage, String targetLanguage,
        TranslationConstants.ContentType contentType, String contentCategory, String userId, int maxTranslations)
        throws TranslationException;

    /**
     * Updates a translation in the Translation Memory engine. Use storeTranslation when an admin/moderator would like
     * to push a reviewed translation into Translation Memory for future use. This API is used for overwriting to the
     * MT service. Stored Translations are added to Translation Memory with a confidence rating. If there is an exact
     * match with a stored translation the translation with the highest confidence level rating will be returned. The
     * default score for translation quality in the TIF (Translation Integration Framework) is fixed at "0" on a scale
     * from "-n" to "+n". To work with the different scales of external services "-n" and "+n" can represent any value
     * range (i.e. -100 .. +100). Individual connectors for each translation service (synchronous or asynchronous)
     * handle conversions between this scale and the scale used by the service being called.
     * @param originalText The string that was originally translated. This can be only one sentence.
     * @param sourceLanguage The string representing the locale that was translated from (source) This accepts all
     *            acceptable locale strings (2 and 5 char versions) Example: en, en_us, en-us, en_US
     * @param targetLanguage The string representing the locale that was translated to (destination) This accepts all
     *            acceptable locale strings (2 and 5 char versions) Example: en, en_us, en-us, en_US
     * @param updatedTranslation The string representing the updated text after an editor reviewed and modified
     * @param contentType The content type of the source (@see TranslationConstants.ContentTypes)
     * @param contentCategory (optional) The Content Category @see TranslationConstants.Categories
     * @param userId The user who modified the translation
     * @param rating (optional) A number representing the "quality" rating of the new translation. The scale is from
     *            -n .. +n with 0 being the default setting for translation quality.
     * @param path (optional) Path to the node containing the translated content
     * @throws TranslationException the translation exception
     */
    void storeTranslation(String originalText, String sourceLanguage, String targetLanguage,
        String updatedTranslation, TranslationConstants.ContentType contentType, String contentCategory,
        String userId, int rating, String path) throws TranslationException;

    /**
     * Updates an array of translations in the Translation Memory engine. Use storeTranslation when an admin/moderator
     * would like to push a reviewed translation into Translation Memory for future use. This API is used for
     * overwriting to the MT service. Stored Translations are added to Translation Memory with a confidence rating. If
     * there is an exact match with a stored translation the translation with the highest confidence level rating will
     * be returned. The default score for translation quality in the TIF (Translation Integration Framework) is fixed
     * at "0" on a scale from "-n" to "+n". To work with the different scales of external services "-n" and "+n" can
     * represent any value range (i.e. -100 .. +100). Individual connectors for each translation service (synchronous
     * or asynchronous) handle conversions between this scale and the scale used by the service being called.
     * @param originalText An array of the original text that was translated
     * @param sourceLanguage The string representing the locale that was translated from (source) This accepts all
     *            acceptable locale strings (2 and 5 char versions) Example: en, en_us, en-us, en_US
     * @param targetLanguage The string representing the locale that was translated to (destination) This accepts all
     *            acceptable locale strings (2 and 5 char versions) Example: en, en_us, en-us, en_US
     * @param updatedTranslation An array of strings containing the updated translations, this must be in the same
     *            order as the originalText array (originalText[1] is the original text of translated string
     *            updatedTranslation[1]). Thus originalText .length must == updatedTranslation.length or an exception
     *            will be thrown.
     * @param contentType The content type of the source (@see TranslationConstants.ContentTypes)
     * @param contentCategory (optional) The Content Category @see TranslationConstants.Categories
     * @param userId The user who modified the translation
     * @param rating (optional) A number representing the "quality" rating of the new translation. The scale is from
     *            -n .. +n with 0 being the default setting for translation quality.
     * @param path (optional) Path to the node containing the translated content
     * @throws TranslationException the translation exception
     */
    void storeTranslation(String[] originalText, String sourceLanguage, String targetLanguage,
        String[] updatedTranslation, TranslationConstants.ContentType contentType, String contentCategory,
        String userId, int rating, String path) throws TranslationException;

    /**
     * Gets the default category.
     * @return Returns a the configured Default Category for this TranslationService instance
     */
    String getDefaultCategory();

    /**
     * Sets the default category.
     * @param defaultCategory The string representing the category which will be set as the default. This is the value
     *            the TranslationService will use if no category is specified elsewhere.
     */
    void setDefaultCategory(String defaultCategory);

    /**
     * Gets information on a translation service.
     * @return Returns a TranslationInfo object which contains the Name, Label, and Attribution Strings for this
     *         specific TranslationService implementation
     */
    TranslationServiceInfo getTranslationServiceInfo();

    /**
     * Is the core method of TranslationService Class for Asynchronous translation. Implementation creates the
     * Translation Job at the translation requester's end of the connector. The Translation Job is the top level
     * container to hold unique Translation Objects, their metadata, and to track the status of the Objects during
     * translation. Each language in a translation process is given a unique translation job per language. The
     * Translation Service monitors the STATUS of Translation Objects between the Translation Job Container with the
     * translation requester and a paired copy of it with the external translation service.
     * @param name of translation job
     * @param description description
     * @param strTargetLanguage the target language into which a project will be translated
     * @param strSourceLanguage source language of the Translation Job.
     * @param dueDate due date when this Translation Job need to be translated
     * @param state current state of translation job
     * @param jobMetadata metadata for translation job
     * @return id, This should return unique translation job id, which can identify the job at connector end.
     * @throws TranslationException the translation exception
     */
    String createTranslationJob(String name, String description, String strSourceLanguage, String strTargetLanguage,
        Date dueDate, TranslationState state, TranslationMetadata jobMetadata) throws TranslationException;

    /**
     * Update translation job metadata.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param jobMetadata metadata for translation job
     * @param translationMethod translation Method
     * @throws TranslationException the translation exception
     */
    void updateTranslationJobMetadata(String strTranslationJobID, TranslationMetadata jobMetadata,
        TranslationMethod translationMethod) throws TranslationException;

    /**
     * Uploads the Translation Object to the Translation Service Provider. This method is used to uploads the unique
     * Object which needs to be translated.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObject the unique thing to be translated
     * @return translation Object id, this should be unique Object id for a given Translation Job. this way it can be
     *         tracked per Translation Job.
     * @throws TranslationException the translation exception
     */
    String uploadTranslationObject(String strTranslationJobID, TranslationObject translationObject)
        throws TranslationException;

    /**
     * Gets the final scope for a Translation Job.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @return TranslationScope, if all calculation is done return the final scope. Returns null in case final scope
     *         calculation is not done.
     */
    TranslationScope getFinalScope(String strTranslationJobID) throws TranslationException;

    /**
     * Update the Translation Job state. Is used to modify the state of the Translation Job. This allows the user to
     * both GET and SET the Status and Comments attached to a Translation Job
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param state the state
     * @return the translation status which is an update of the project status
     * @throws TranslationException the translation exception
     */
    TranslationStatus updateTranslationJobState(String strTranslationJobID, TranslationState state)
        throws TranslationException;

    /**
     * Gets the Translation Job status.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @return the translation job status
     * @throws TranslationException the translation exception
     */
    TranslationStatus getTranslationJobStatus(String strTranslationJobID) throws TranslationException;

    /**
     * Gets the Translation Job comment collection. Return all the comments added at Translation Job level. AEM code
     * will check for comment date, author, text to find out if the comment is already part of Collection or not. if
     * it is not present then the comment will be added in collection. this is used to sync comments from connector
     * back to AEM
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @return the Translation Job comment collection
     */
    CommentCollection<Comment> getTranslationJobCommentCollection(String strTranslationJobID)
        throws TranslationException;

    /**
     * Adds a comment at the translation job level. Translation Job level Comments are metadata about a particular TJ.
     * They are used to communicate between the Translation Requester and the Translation Service Provider about the
     * Translation Job while translation is is process.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param comment, Comments attached to the translation Object, a form of Object metadata
     * @throws TranslationException the translation exception
     */
    void addTranslationJobComment(String strTranslationJobID, Comment comment) throws TranslationException;

    /**
     * Gets the translated Object content input stream. This is the step by which Translated Objects are downloaded
     * from the External Translation Service's container back to the Translation Requesters container. This step is
     * carried out after the STATUS' of all Translation Objects is found to be TRANSLATED.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObject The translation Object
     * @return the translated Object as inputstream
     * @throws TranslationException the translation exception
     */
    InputStream getTranslatedObject(String strTranslationJobID, TranslationObject translationObject)
        throws TranslationException;

    /**
     * Allows the user to update the state for a unique Translation Object. Each Translation Object has a state DRAFT,
     * SUBMITTED, TRANSLATED, etc. The state of each object is used to indicate where it is in the translation process
     * and to move it through the steps of the process. For example: when the Translation Service can query all the
     * Translation Objects in a job and determines that all their status's are now: TRANSLATED then the Translation
     * Service can determine that the Translation Job is now finished. See translation Constants for a list of status
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObject The translation Object
     * @param state the state
     * @return the translation status
     * @throws TranslationException the translation exception
     */
    TranslationStatus updateTranslationObjectState(String strTranslationJobID, TranslationObject translationObject,
        TranslationState state) throws TranslationException;

    /**
     * Gets the translation Object status.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObject The translation Object
     * @return the translation Object status
     * @throws TranslationException the translation exception
     */
    TranslationStatus getTranslationObjectStatus(String strTranslationJobID, TranslationObject translationObject)
        throws TranslationException;

    /**
     * Update the state for an array of translation Objects. This is a batch operation to update the states for
     * multiple translation Objects with a single call. The call returns a list of status states for each translation
     * Object in the order they are stored. See translationConstants for list of states
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObjects array of translation objects
     * @param states the states
     * @return the translation status[]
     * @throws TranslationException the translation exception
     */
    TranslationStatus[] updateTranslationObjectsState(String strTranslationJobID,
        TranslationObject[] translationObjects, TranslationState[] states) throws TranslationException;

    /**
     * Gets the status of the translation Object. This can be used for batch operations where you can pass multiple
     * translation Objects as parts of a single Translation Job. In this batch operation the status for each
     * translation Object will be returned in the order they were submitted.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObjects array of translation objects
     * @return the list of status
     * @throws TranslationException the translation exception
     */
    TranslationStatus[] getTranslationObjectsStatus(String strTranslationJobID, TranslationObject[] translationObjects)
        throws TranslationException;

    /**
     * Gets the Translation Object comment collection. The method will return all the comments added at the
     * translation Object level. The Adobe Experience Manager core code then will check a Translation Object for
     * comments, date information, author attribution or text to find out if this data is already associated with the
     * translation Object. If it is not present then the comment will be added to the translation Object. This method
     * is used to sync comments via the connector back to the AEM Translation Job.
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObject The translation Object
     * @return the Translation Job comment collection
     */
    CommentCollection<Comment> getTranslationObjectCommentCollection(String strTranslationJobID,
        TranslationObject translationObject) throws TranslationException;

    /**
     * Adds a Comment to the Translation Object. This is used to for communication between the Translation Requester
     * and the Translation Service Provider
     * @param strTranslationJobID job id, which was returned on translation job creation
     * @param translationObject The translation Object
     * @param comment, Comments attached to the translation Object, a form of Object metadata
     * @throws TranslationException the translation exception
     */
    void addTranslationObjectComment(String strTranslationJobID, TranslationObject translationObject, Comment comment)
        throws TranslationException;

    /**
     * Update translation job due-date
     * @param strTranslationJobID Translation Job ID
     * @param date Due date
     * @throws TranslationException
     */
    void updateDueDate(String strTranslationJobID, Date date) throws TranslationException;

    /** The Interface TranslationServiceInfo. */
    public interface TranslationServiceInfo {

        /**
         * Gets the translation service attribution.
         * @return Optional property used to attribute this translation provider (such as
         *         "Translations by Adobe Translate")
         */
        String getTranslationServiceAttribution();

        /**
         * Gets the translation service label.
         * @return Optional property used to describe this translation service in a user-interface (such as to allow
         *         its selection from a drop-down menu).
         */
        String getTranslationServiceLabel();

        /**
         * Gets the translation service name.
         * @return Name of the service property indicating the particular implementation this factory provides, e.g.
         *         Adobe.
         */
        String getTranslationServiceName();

        /**
         * Gets the supported translation method.
         * @return the supported translation method
         */
        TranslationMethod getSupportedTranslationMethod();

        /**
         * Gets the service Cloud Config Root path Location.
         * @return Root path of the service Cloud Config. This will be used to determine the cloud config for the
         *         Translation service.
         */
        String getServiceCloudConfigRootPath();

    }

}