/*
 * File: Base64Engine.java
 *
 * ****************************************************************************
 *
 *  ADOBE CONFIDENTIAL
 *  ___________________
 *
 *  Copyright 1998-2006 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 *  NOTICE: All information contained herein is, and remains the property of
 *  Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *  and technical concepts contained herein are proprietary to Adobe Systems
 *  Incorporated and its suppliers and may be covered by U.S. and Foreign
 *  Patents, patents in process, and are protected by trade secret or
 *  copyright law. Dissemination of this information or reproduction of this
 *  material is strictly forbidden unless prior written permission is obtained
 *  from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Base64Engine
 *
 * This class provides Base64 Content-Transfer-Encoding encoding and decoding.
 * See RFC 3548 for details (http://www.ietf.org/rfc/rfc3548.txt?number=3548).
 */
public class Base64Engine
{
	/**
	 * @author sweet
	 *
	 * Create an OutputStream such that when written to with a Base64 encoded stream
	 * of bytes, will write the decoded bytes to the supplied output stream.
	 */
	public static class OutputDecodeStream extends OutputStream
	{
		private byte[] mInbuf;
		private byte[] mOutbuf;
		private int mInNdx;
		private OutputStream mOut;

		public OutputDecodeStream(OutputStream dest)
		{
			mInbuf = new byte[4];
			mOutbuf = new byte[3];
			mInNdx = 0;
			mOut = dest;
		}

		/* (non-Javadoc)
		 * @see java.io.OutputStream#write(int)
		 */
		@Override
		public void write(int b)
			throws IOException
		{
			if (b < 0)
				throw new Base64FilterException("invalid Base64 character");
			if (base64DecodeTable[b & 0x7f] == IL)
				return; // ignored character
			mInbuf[mInNdx++] = (byte)(b & 0x7f);
			if (mInNdx >= 4) {
				int outLen = decode(mInbuf, 0, 4, mOutbuf, 0);
				mOut.write(mOutbuf, 0, outLen);
				mInNdx = 0;
			}
		}

		@Override
		public void close()
			throws IOException
		{
			/*
			 * A valid Base64 encoded source has a multiple of 4 non-ignored characters.
			 * If you are closing the file, mInNdx had better be 0
			 */
			if (mInNdx != 0)
				throw new Base64FilterException("Incomplete Base64 data stream");
		}

	}
	
	/**
	 * Create an output stream that writes base64-encoded bytes to the supplied stream
	 */
	public static class EncodedOutputStream extends OutputStream
	{
		private OutputStream mOut;
		byte[] mInbuf;
		byte[] mOutbuf;
		//int mOutNdx;
		int mInNdx;
		int mOnLine;

		public EncodedOutputStream(OutputStream destination)
		{
			mOut = destination;
			mInbuf = new byte[3];
			mOutbuf = new byte[4];
			mOnLine = 0;
		//	mOutNdx = 0;
			mInNdx = 0;
		}

		@Override
		public void close() throws IOException
		{
			flush();
			mOut.close();
		}

		@Override
		public void flush() throws IOException
		{
			if (mInNdx > 0)
			{
				int outLen = encode(mInbuf, 0, mInNdx, mOutbuf, 0);
				mOut.write(mOutbuf, 0, outLen);
//				mOut.write('\r');
				mOut.write('\n');
				mOnLine = 0;
				mInNdx = 0;
			}
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#read()
		 */
		@Override
		public void write(int b) throws IOException
		{
			mInbuf[mInNdx++] = (byte) b;
			if (mInNdx >=3)
			{
				int outLen = encode(mInbuf, 0, mInNdx, mOutbuf, 0);
				mOut.write(mOutbuf, 0, outLen);
				mOnLine += outLen;
				mInNdx = 0;
				if (mOnLine >= 76)
				{
//					mOut.write('\r');
					mOut.write('\n');
					mOnLine = 0;
				}
			}
		}
	}

	/**
	 * @author sweet
	 *
	 * Create an input stream such that it reads bytes from the supplied stream
	 * and returns Base64 encoded bytes.
	 */
	public static class InputEncodeStream extends InputStream
	{
		private InputStream mIn;
		byte[] mInbuf;
		byte[] mOutbuf;
		int mOutNdx;
		int mOnLine;
		boolean mSendLF;

		public InputEncodeStream(InputStream source)
		{
			mIn = source;
			mInbuf = new byte[3];
			mOutbuf = new byte[4];
			mOnLine = 0;
			mOutNdx = 4; // force a read
			mSendLF = false;
		}

		/* (non-Javadoc)
		 * @see java.io.InputStream#read()
		 */
		@Override
		public int read() throws IOException {
			if (mSendLF) {
				mSendLF = false;
				return '\n';
			}
			else if (mOnLine == 76) {
				mSendLF = true;
				mOnLine = 0;
				return '\r';
			} else {
				if (mOutNdx >= 4) {
					/* get some more bytes */
					int mInMax = mIn.read(mInbuf, 0, 3);
					if (mInMax == -1)
						return -1;
					encode(mInbuf, 0, mInMax, mOutbuf, 0);
					mOutNdx = 0;
				}
				mOnLine++;
				return(mOutbuf[mOutNdx++]);
			}
		}
	}

	private static final byte ED = (byte)64;	/* EOD */
	private static final byte IL = (byte)65;	/* ignored character */

	private static byte base64DecodeTable[];
	private static byte	base64EncodeTable[];
    

	// Static initializer
	static
	{
		base64DecodeTable = new byte[128];
		base64EncodeTable = new byte[64];

		for (int i = 0; i < 128; i++ )
			base64DecodeTable[i] = IL;

		for (int i = 0; i < 26; i++) {
			base64DecodeTable['A' + i] = (byte)(i);
			base64EncodeTable[i] = (byte)('A' + i);
			base64DecodeTable['a' + i] = (byte)(i + 26);
			base64EncodeTable[i + 26] = (byte)('a' + i);
		}

		for (int i = 0; i < 10; i++) {
			base64DecodeTable['0' + i] = (byte)(i + 52);
			base64EncodeTable[i + 52] = (byte)('0' + i);
		}

		base64DecodeTable['+'] = (byte)(62);
		base64EncodeTable[62] = (byte)('+');
		base64DecodeTable['/'] = (byte)(63);
		base64EncodeTable[63] = (byte)('/');
		base64DecodeTable['='] = ED;
	}

	/**
	 * Decode an array of bytes into another array of bytes
	 * return the number of output bytes
	 *
	 * If output is null, just return the output byte count
	 */
	static public int decode(byte[] input, int inOffset, int length, byte[] output, int outOffset)
	{
		int maxout = 0;
		int count = 0;

		if (output != null)
			maxout = output.length - outOffset;

		if (length > input.length - inOffset)
			length = input.length - inOffset;

		while (length > 0) {
			int k = 0;
			int val = 0;

			while (k < 4 && length-- > 0) {
				int cval = base64DecodeTable[input[inOffset++] & 0x7f];
				if (cval < 64) {
					val = (val << 6) + cval;
					++k;
				} else if (cval == ED && k != 0) {
					length = 0;
					break;
				}
			}

			switch(k) {
			case 1:
				val <<= 6;
			case 2:
				val <<= 6;
			case 3:
				val <<= 6;
				break;
			}

			if (k >= 1) {
				++count;
				if (maxout-- > 0)
					output[outOffset++] = (byte)(val >> 16);
			}
			if (k >= 3) {
				++count;
				if (maxout-- > 0)
					output[outOffset++] = (byte)(val >> 8);
			}
			if (k == 4) {
				++count;
				if (maxout-- > 0)
					output[outOffset++] = (byte)(val);
			}
		}

		return count;
	}

	/**
	 * Encode an array of bytes into another array of bytes
	 * return the number of output bytes
	 *
	 * If output is null, just return the output byte count
	 */
	static public int encode(byte[] input, int inOffset, int length, byte[] output, int outOffset)
	{
		int maxout = 0;
		int count = 0;
		int val;

		if (output != null)
			maxout = output.length - outOffset;

		if (length > input.length - inOffset)
			length = input.length - inOffset;

		int inline = 0;
		while (length >= 3) {
			if (inline == 76) {
				if (maxout-- > 0)
					output[outOffset++] = (byte)'\r';
				if (maxout-- > 0)
					output[outOffset++] = (byte)'\n';
				count += 2;
				inline = 0;
			}

			val  = (input[inOffset++] & 0xff) << 16;
			val += (input[inOffset++] & 0xff) << 8;
			val += (input[inOffset++] & 0xff);

			if (maxout-- > 0)
				output[outOffset++] = base64EncodeTable[val >>> 18];

			if (maxout-- > 0)
				output[outOffset++] = base64EncodeTable[(val >>> 12) & 0x3f];

			if (maxout-- > 0)
				output[outOffset++] = base64EncodeTable[(val >>> 6) & 0x3f];

			if (maxout-- > 0)
				output[outOffset++] = base64EncodeTable[val & 0x3f];

			length -= 3;
			count  += 4;
			inline += 4;
		}

		if (length > 0) {   // 1 and 2 are the only possible values here
			if (inline == 76) {
				if (maxout-- > 0)
					output[outOffset++] = (byte)'\r';
				if (maxout-- > 0)
					output[outOffset++] = (byte)'\n';
				count += 2;
			}

			val  = (input[inOffset++] & 0xff) << 16;
			if (length > 1)
				val += (input[inOffset] & 0xff) << 8;

			if (maxout-- > 0)
				output[outOffset++] = base64EncodeTable[val >>> 18];

			if (maxout-- > 0)
				output[outOffset++] = base64EncodeTable[(val >>> 12) & 0x3f];

			if (maxout-- > 0) {
				if (length > 1)
					output[outOffset++] = base64EncodeTable[(val >>> 6) & 0x3f];
				else
					output[outOffset++] = (byte)'=';
			}

			if (maxout-- > 0)
				output[outOffset++] = (byte)'=';

			count  += 4;
		}

		return count;
	}

    
    
    
    
	/* Maps a 6 bit value to the character that represents it. */
    private static char[] bitsToChar = 
    { 'A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I', 'J', 'K', 
      'L', 'M', 'N', 'O', 'P', 'Q', 'R', 'S', 'T', 'U', 'V',
      'W', 'X', 'Y', 'Z', 
      'a', 'b', 'c', 'd', 'e', 'f', 'g', 'h', 'i', 'j', 'k',
      'l', 'm', 'n', 'o', 'p', 'q', 'r', 's', 't', 'u', 'v',
      'w', 'x', 'y', 'z', 
      '0', '1', '2', '3', '4', '5', '6', '7', '8', '9',
      '+', '/'};
    
    /* Map a character in the range U+0000..U+007F, to the 6 bit value
     * it represents, or -1 if that character is not allowed.
     */
    private static final byte[] charToBits =
    { -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  // 0x
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,  // 1x
      -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62, -1, -1, -1, 63,  // 2x
      52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1,  // 3x
      -1,  0,  1,  2,  3,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14,  // 4x
      15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, -1, -1, -1, -1, -1,  // 5x
      -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40,  // 6x
      41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1,  // 7x
    };    
    
    /* Return the Base64 representation of a sequence of bytes.
     */
    static public char[] encode (byte[] input, int inOffset, int length)
    {
        char[] output = new char [(((length * 8) + 23) / 24) * 4];
        int outOffset = 0;

        while (length >= 3) {
            int val  = (input [inOffset++] & 0xff) << 16;
            val |= (input [inOffset++] & 0xff) << 8;
            val |= (input [inOffset++] & 0xff);

            output [outOffset++] = bitsToChar [(val >>> 18) & 0x3f];
            output [outOffset++] = bitsToChar [(val >>> 12) & 0x3f];
            output [outOffset++] = bitsToChar [(val >>>  6) & 0x3f];
            output [outOffset++] = bitsToChar [(val       )  & 0x3f];

            length -= 3; }

        if (length > 0) {   // 1 and 2 are the only possible values here
            int val  = (input[inOffset++] & 0xff) << 16;
            if (length > 1)
                val |= (input[inOffset] & 0xff) << 8;

            output [outOffset++] = bitsToChar [(val >>> 18) & 0x3f];
            output [outOffset++] = bitsToChar [(val >>> 12) & 0x3f];
            output [outOffset++] = length > 1 ? bitsToChar [(val >>> 6) & 0x3f] : '=';
            output [outOffset++] = (byte)'='; }

        return output;
    }
    
    /* Return the sequence of bytes represented by Base64 string, or
     * null if that string is not a legal representation.
     */
    static public byte[] decode (String s)
    {
      int inLen = s.length ();
      if (inLen % 4 != 0) {
        return null; }
      
      int outLen = 3 * inLen / 4;
      if (inLen > 0 && s.charAt (inLen - 1) == '=') {
        inLen--;
        outLen--; }
      if (inLen > 0 && s.charAt (inLen - 1) == '=') {
        inLen--;
        outLen--; }
      byte[] bytes = new byte [outLen];
      int b = 0;
      
      int i = 0;
      while (i < inLen) {
        char c;
        
        c = s.charAt (i++);
        if (c > charToBits.length) {
          return null; }
        int d0 = charToBits [c];
        
        c = s.charAt (i++);
        if (c > charToBits.length) {
          return null; }
        int d1 = charToBits [c];
        
        int d2 = 0;
        if (i < inLen) {
          c = s.charAt (i++);
          if (c > charToBits.length) {
            return null; }
          d2 = charToBits [c]; }
        
        int d3 = 0;
        if (i < inLen) {
          c = s.charAt (i++);
          if (c > charToBits.length) {
            return null; }
          d3 = charToBits [c]; }
        
        if (d0 < 0 || d1 < 0 || d2 < 0 || d3 < 0) {
          return null; }
        
        bytes [b++] = (byte) (((d0 & 0x3f) << 2) | ((d1 & 0x30) >> 4));
        if (b < outLen) {
          bytes [b++] = (byte) (((d1 & 0x0f) << 4) | ((d2 & 0x3c) >> 2)); 
          if (b < outLen) {
            bytes [b++] = (byte) (((d2 & 0x03) << 6) | ((d3 & 0x3f))); }}}
      
      return bytes;
    }
}
