/* ****************************************************************************
 *
 *	File: QuickSort2.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2007 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.util;

import java.util.Comparator;

/**
 * WARNING
 * This class provides mirror implementations of the Acrobat
 * quicksort algorithm based on qsort.c in VisualStudio 6
 * (Acro6) and VisualStudio 2003 (Acro7). Compatibility with
 * these particular algorithms is critical for correct digital
 * signature processing.
 * 
 * These versions of quicksort are designed to provide the same
 * ordering characteristics as Acrobat in the presence of objects
 * in the array that have duplicate keys (that is, multiple objects
 * that compare as equals, such as annotation IDs, but whose values
 * differ (e.g., a digest of the annotation dictionary)). How
 * duplicate objects are sorted is a function of the partioning
 * pivot point and the threshold at which shortsort is invoked,
 * so CHANGES SHOULD NOT BE MADE to this implementation without
 * some very, very deep soul searching.
 */
public final class QuickSort2
{
	// This parameter defines the cutoff between using quick sort and
	// insertion sort for arrays; arrays with lengths shorter or equal to the
	// below value use insertion sort.
	private static final int CUTOFF = 8;

	// The number of stack entries required is no more than (1 + log2(array.length)),
	// so 30 is sufficient for any array.
	private static final int STKSIZ = 30;

	/**
	 * Performs quick sort on the array passed here based on the comparator.
	 */
	public static void sort_acro6(Object[] array, Comparator comparator)
	{
		int lo, hi;			// ends of sub-array currently sorting
		int mid;			// points to middle of subarray
		int loguy, higuy;	// traveling indices for partition step
		int size;			// size of the sub-array
		int lostk[] = new int[STKSIZ];
		int histk[] = new int[STKSIZ];
		int stkptr;			// stack for saving sub-array to be processed

		if (array.length < 2)
			return; // nothing to do

		stkptr = 0; // initialize stack

		lo = 0;
		hi = array.length - 1; // initialize limits

		// This loop provides for pseudo-recursion using the stacks defined
		// above. Setting lo and hi and looping (early in some cases by using
		// the continue command) is like recursion. The variable stkptr is
		// preserved, locals aren't, so we preserve stuff on the stack as
		// we "recurse."
		while (true)
		{
			size = (hi - lo) + 1;	// number of elements to sort

			// Below a certain size, it is faster to use a O(n^2) sorting method.
			if (size <= CUTOFF)
			{
				shortsort(array, lo, hi, comparator);
			}
			else
			{
				/* First we pick a partitioning element.  The efficiency of the
				 * algorithm demands that we find one that is approximately the median
				 * of the values, but also that we select one fast.  We choose the
				 * median of the first, middle, and last elements, to avoid bad
				 * performance in the face of already sorted data, or data that is made
				 * up of multiple sorted runs appended together.  Testing shows that a
				 * median-of-three algorithm provides better performance than simply
				 * picking the middle element for the latter case.
				 */
				mid = lo + (size / 2);	// find middle element

				// Sort the first, middle, last elements into order.
				swap(array, mid, lo);

				/* We now wish to partition the array into three pieces, one consisting
				 * of elements <= partition element, one of elements equal to the
				 * partition element, and one of elements > than it.  This is done
				 * below; comments indicate conditions established at every step.
				 */
				loguy = lo;
				higuy = hi + 1;

				// Note that higuy decreases and loguy increases on every iteration,
				// so loop must terminate.
				for (;;)
				{
					do
					{
						loguy++;
					} while (loguy <= hi
							&& comparator.compare(array[loguy], array[lo]) <= 0);

					do
					{
						higuy--;
					} while (higuy > lo
							&& comparator.compare(array[higuy], array[lo]) >= 0);

					if (higuy < loguy)
						break;

					swap(array, loguy, higuy);
				}

				swap(array, lo, higuy);
				
				if (higuy - 1 - lo >= hi - loguy)
				{
					if (lo + 1 < higuy)
					{
						lostk[stkptr] = lo;
						histk[stkptr] = higuy - 1;
						++stkptr;
					} // save big recursion for later

					if (loguy < hi)
					{
						lo = loguy;
						continue; // do small recursion
					}
				}
				else
				{
					if (loguy < hi)
					{
						lostk[stkptr] = loguy;
						histk[stkptr] = hi;
						++stkptr; // save big recursion for later
					}

					if (lo + 1 < higuy)
					{
						hi = higuy - 1;
						continue; // do small recursion
					}
				}
			}

			// We have sorted the array, except for any pending sorts on the stack.
			// Check if there are any, and do them.
			--stkptr;
			if (stkptr >= 0)
			{
				lo = lostk[stkptr];
				hi = histk[stkptr];
				continue; // pop subarray from stack
			}
			else
				return; // all subarrays done
		}
	}

	/**
	 * Insertion short for sorting short arrays.
	 * 
	 * @param array - The array to be sorted in place.
	 * @param lo - The index of the low element to sort.
	 * @param hi - The index of the high element to sort.
	 * @param comparator - Comparator to provide object ordering.
	 */
	static void shortsort(Object[] array, int lo, int hi, Comparator comparator)
	{
		// Sorts in descending order. That is, the max value is always swapped to
		// the current hi index, while hi is decremented to lo.
		while (hi > lo)
		{
			int max = lo;
			for (int i = lo + 1; i <= hi; i++)
			{
				// Find the next max value in the sub array.
				if (comparator.compare(array[i], array[max]) > 0)
					max = i;
			}
			swap(array, max, hi);
			hi--;
		}
	}

	static void swap(Object[] array, int i, int j)
	{
		if (i != j)
		{
			Object temp = array[i];
			array[i] = array[j];
			array[j] = temp;
		}
	}
}