/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.template.containers;

import com.adobe.xfa.Attribute;
import com.adobe.xfa.XFA;
import com.adobe.xfa.ut.Angle;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.MsgFormatPos;
import com.adobe.xfa.ut.ResId;
import com.adobe.xfa.ut.StringUtils;

/**
 * An attribute that describes the rotation of a container, i.e., a value in
 * degrees progressing counterclockwise. The point of rotation can be changed by
 * setting the anchorType attribute, but by default it is the top-left corner.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class Rotate extends Attribute {
	/**
	 * 
	 * Rotation value.
	 */
	private final Angle moValue;

	/**
	 * Constructor that supplies all possible member values.
	 * @param NS - namespace
	 * @param localName - local name
	 * @param qName - qualified name
	 * @param value - attribute value.
	 * @param internSymbols - indicates whether symbols are to be interned.
	 */
	private Rotate(String NS, String localName, String qName, String value, boolean internSymbols) {
		super(NS, localName, qName, value, internSymbols);
		if (StringUtils.isEmpty(value)) {
			moValue = Angle.ZERO;
		} 
		else {
			double dValue = 0.0d;
			try {
				dValue = Double.parseDouble(value);
			} catch (NumberFormatException n) {
				// Invalid value
				MsgFormatPos oMessage = new MsgFormatPos(ResId.InvalidPropertyValueException, value);
				oMessage.format(XFA.ROTATE);
				
				throw new ExFull(oMessage);
			}
			while (dValue < 0)
				dValue += 360;
			while (dValue > 360)
				dValue -= 360;
			moValue = new Angle(dValue);
		}
	}

	public Rotate(String name, Angle other) {
		super(name, other.toString());
		moValue = other;
	}

	// Copy constructor for Obj
	public Rotate(Rotate other) {
		super(other.getName(), other.toString());
		moValue = other.getAngle();
	}

	public Rotate(String name, String stringValue) {
		super(name, stringValue);
		if (StringUtils.isEmpty(stringValue)) {
			moValue = Angle.ZERO;
		} 
		else {
			double dValue = 0.0d;
			try {
				dValue = Double.parseDouble(stringValue);
			} catch (NumberFormatException n) {
				throw new ExFull(ResId.ArgumentMismatchException);
			}
			while (dValue < 0)
				dValue += 360;
			while (dValue > 360)
				dValue -= 360;
			moValue = new Angle(dValue);
		}
	}

	public Angle getAngle() {
		return moValue;
	}

	public double getValue() {
		return moValue.getAngle();
	}
	

	/**
	 * @see Attribute#newAttribute(String)
	 */
	public Attribute newAttribute(String value) {
		return newAttribute(getNS(), getLocalName(), getQName(), value, false);
	}

	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String)
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value) {
		return newAttribute(NS, localName, qName, value, true);
	}
	
	/* (non-Javadoc)
	 * @see com.adobe.xfa.Attribute#newAttribute(java.lang.String, java.lang.String, java.lang.String, java.lang.String, boolean)
	 */
	public Attribute newAttribute(String NS, String localName, String qName, String value, boolean internSymbols) {
		return new Rotate(NS, localName, qName, value, internSymbols);
	}

	public String toString() {
		String sRet = Double.toString(moValue.getAngle());
		
		// This is a bit silly, but the C++ version returns no decimal place
		// if it's an integral value.
		if (sRet.endsWith(".0"))
			sRet = sRet.substring(0, sRet.length() -2);
		return sRet;
	}
}
