/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.api.components;

import java.util.Set;

import org.apache.sling.api.resource.Resource;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.WCMMode;
import com.day.cq.wcm.api.designer.Cell;

/**
 * A component context is opened for each component include that provides a new
 * cell on the cell stack. this is usually the case if the cell name is not
 * empty.
 *
 * The component context is independent from the WCM mode and is always available
 * to the scripts. If the WCM is in {@link  WCMMode#EDIT edit} mode and if the
 * components are editable it also provides an {@link EditContext}.
 *
 * Note that an edit context can span several component contexts if not all of
 * the components that are included provide editing. for example pure
 * general purpose scripts.
 */
public interface ComponentContext {

    /**
     * Request attribute name for retrieving the current context
     */
    String CONTEXT_ATTR_NAME = "com.day.cq.wcm.componentcontext";

    /**
     * name of the request attribute to bypass component handling.
     */
    String BYPASS_COMPONENT_HANDLING_ON_INCLUDE_ATTRIBUTE =
        ComponentContext.class.getName() + "/bypass";

    /**
     * Returns the parent context or <code>null</code> if this context
     * is the root context.
     * @return the parent context or <code>null</code>
     */
    ComponentContext getParent();

    /**
     * Returns the root context.
     * @return the root context.
     */
    ComponentContext getRoot();

    /**
     * Checks if this is the root context.
     * @return <code>true</code> if this is the root context
     */
    boolean isRoot();

    /**
     * Returns the top most resource that is associated with this context.
     * it can occur that a component context spans several resources if a
     * resource of a sub include does not define a new cell.
     * @return the resource or <code>null</code> if not bound.
     */
    Resource getResource();

    /**
     * Returns the design cell.
     * @return the design cell.
     */
    Cell getCell();

    /**
     * Returns the edit context for this include. If the respective component
     * is not editable/designable or if the WCM is not in {@link WCMMode#EDIT}
     * or {@link WCMMode#DESIGN} <code>null</code> is returned.
     *
     * Please note that for different component contexts, the same edit context
     * can be returned, if the respective component spans several includes.
     *
     * @return edit context or <code>null</code>
     */
    EditContext getEditContext();
    
    /**
     * Returns the analyze context for this include. If the respective component
     * is not analyzable or if the WCM is not in
     * {@link com.day.cq.wcm.api.WCMMode#ANALYTICS}, <code>null</code> is
     * returned.
     * 
     * @return analyze context or <code>null</code>
     */
    AnalyzeContext getAnalyzeContext();

    /**
     * Returns the top most component that is associated with this context.
     *
     * @return the component or <code>null</code> if not bound.
     */
    Component getComponent();

    /**
     * Returns the page this component context is operating on.
     * @return the page.
     */
    Page getPage();

    /**
     * Returns the context attribute for the given name
     * @param name the name of the attribute
     * @return the attribute or <code>null</code> if not defined.
     */
    Object getAttribute(String name);

    /**
     * Sets a context attribute with the given name. If the given attribute is
     * <code>null</code> it is removed from the attribute map.
     * @param name the name of the attribute
     * @param value the attribute value
     * @return the old attribute or <code>null</code>
     */
    Object setAttribute(String name, Object value);

    /**
     * Returns the set of css class names to be added to the html tag. note
     * that the names from the include options are only added when the context
     * is included.
     *
     * @return the set of css class names
     */
    Set<String> getCssClassNames();

    /**
     * Checks if this contexts needs decoration at all.
     * @return <code>true</code> if this context needs decoration
     */
    boolean hasDecoration();

    /**
     * Sets if this context needs decoration.
     * @param decorate <code>true</code> if decoration should be drawn
     */
    void setDecorate(boolean decorate);

    /**
     * Returns the decoration tag name.
     * @return the decoration tag name
     * @see #setDecorationTagName(String)
     * @since 5.3
     */
    String getDecorationTagName();

    /**
     * Sets the decoration tag name. if set to <code>null</code> the default
     * name will be used, if set to an empty string, no decoration tag is rendered.
     * the tag name is initialized by the "cq:tagName" property of the
     * "cq:htmlTag" node of the component definition, unless it is overriden by a
     * include option.
     *
     * @param decorationTagName the name of the decoration  tag.
     * @see #setDefaultDecorationTagName(String) 
     * @see Component#getHtmlTagAttributes()
     * @see IncludeOptions#setDecorationTagName(String)
     * @since 5.3
     */
    void setDecorationTagName(String decorationTagName);

    /**
     * Returns the default decoration tag name.
     * @return the default decoration tag name
     * @see #setDefaultDecorationTagName(String) 
     * @since 5.3
     */
    String getDefaultDecorationTagName();

    /**
     * Sets the default decoration tag name. if set to <code>null</code> it will be
     * inherited from the parent context, if set to an empty string, no
     * decoration tag is rendered.
     * the default tag name is used if non is provided by the component or set via
     * include options. it defaults to <code>div</code>.
     *
     * @param decorationTagName the name of the decoration  tag.
     * @see #setDecorationTagName(String)
     * @see Component#getHtmlTagAttributes()
     * @see IncludeOptions#setDecorationTagName(String)
     * @since 5.3
     */
    void setDefaultDecorationTagName(String decorationTagName);



}