/*
 * File: FlateOutputStream.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.filter;

import java.io.FilterOutputStream;
import java.io.IOException;
import java.io.OutputStream;

/**
 * FlateOutputStream
 *  Implements the GZIP compression algorithm and ZLIB format
 *  RFCs 1950 and 1951
 *
 *	Important: This filter assumes the input stream is an ISO/Latin-1
 *	stream, of 1-byte (not Unicode) characters!
 *
 * Copyright (C) 1995-1996 Jean-loup Gailly
 * Copyright (C) 1996-2005 Adobe Systems Incorporated
 */
/* representation of a subtree class */
class FlateTree
{
	int		freq[];			/* freq and code are the same array */
	int		code[];
	int		dad[];			/* dad and len are the same array */
	int		len[];
	int		elems;			/* max number of elements in the tree */
	int		max_code;		/* maximum code with non-zero frequency */
	int		max_length;		/* max bit length for the codes */
	int		opt_len;		/* length metrics for judging optimal strategy */

	int		extra_bits[];		/* extra bits for each code or NULL */
	int		extra_base;		/* base index for extra_bits */

	FlateTree	static_tree;		/* static tree or NULL */

	short		heap[], depth[];
	int		heap_len, heap_max;

	//Methods

	public FlateTree(FlateTree s_tree, int treeSize, int elemSize, int maxlength, int bits[], int base)
	{
		elems = elemSize;
		max_length = maxlength;
		static_tree = s_tree;
		extra_bits = bits;
		extra_base = base;

		freq = code = new int[treeSize];
		dad = len = new int[treeSize];

		heap = null;		/* Release heap */
		depth = null;
	}

	void InitStaticLTree()		/* Only called for the static length tree */
	{
		/* Construct the codes of the static literal tree */
		int bl_count[] = new int[max_length+1];

		for (int bits = 0; bits <= max_length; bits++) bl_count[bits] = 0;
		int n = 0;

		while (n <= 143) {len[n++] = 8; bl_count[8]++;}
		while (n <= 255) {len[n++] = 9; bl_count[9]++;}
		while (n <= 279) {len[n++] = 7; bl_count[7]++;}
		while (n <= 287) {len[n++] = 8; bl_count[8]++;}
		/* Codes 286 and 287 do not exist, but we must include them in the
		 * tree construction to get a canonical Huffman tree (longest code
		 * all ones)
		 */

		max_code = elems+1;
		gen_codes(bl_count);
	}

	void InitStaticDTree()			/* Only called for the static distance tree */
	{
		/* The static distance tree is trivial: */
		for (int n = 0; n < elems; n++) {
			code[n] = bi_reverse(n, 5);
			len[n] = 5;
		}
	}

	void InitTree()				/* called once per block on dynamic trees only */
	{
		for (int i = 0; i < elems; i++)
			freq[i] = 0;

		if (elems > 256)
			freq[256] = 1;		/* for length tree, make sure the STOP code is counted */

		opt_len = 0;
		if (static_tree != null) static_tree.opt_len = 0;
	}

	/* ===========================================================================
	 * Reverse the first len bits of a code, using straightforward code (a faster
	 * method would use a table)
	 * IN assertion: 1 <= len <= 15
	 */
	private int bi_reverse (
				int	code,		/* the value to invert */
				int	len)		/* its bit length */
	{
		int res = 0;

		code <<= 1;
		while (len-- > 0) {
			code >>>= 1; res <<= 1;
			res |= code & 1;
		}
		return res;
	}

	/* ===========================================================================
	 * Generate the codes for a given tree and bit counts (which need not be
	 * optimal).
	 * IN assertion: the array bl_count contains the bit length statistics for
	 * the given tree and the field len is set for all tree elements.
	 * OUT assertion: the field code is set for all tree elements of non
	 *     zero code length.
	 */
	private void gen_codes (
				int		bl_count[])	/* bit length statistics */
	{
		int	next_code[] = new int [max_length+1];	/* next code value for each bit length */
		int	cv;					/* running code value */
		int bits;					/* bit index */
		int n;						/* code index */

		/* The distribution counts are first used to generate the code values
		 * without bit reversal.
		 */
		next_code[0] = cv = 0;
		for (bits = 1; bits <= max_length; bits++) {
			next_code[bits] = cv = (cv + bl_count[bits-1]) << 1;
		}
		/* Check that the bit counts in bl_count are consistent. The last code
		 * must be all ones.
		 */
		//ZAssert (cv + bl_count[MAX_BITS]-1 == (1<<MAX_BITS)-1,
		//		"inconsistent bit counts");
		//if (cv + bl_count[max_length] - 1 != (1 << max_length) - 1)
		//    Log.clogn("Inconsistent bit counts ");
		//ZTracev((z_trace,"\ngen_codes: max_code %d ", max_code));
		//    Log.clogn("Max_code = "); Log.clog(max_code);

		for (n = 0; n <= max_code; n++) {
			int l = len[n];
			if (l == 0) continue;
			/* Now reverse the bits */
			code[n] = bi_reverse(next_code[l]++, l);

			//ZTracecv(tree != static_ltree, (z_trace,"\nn %3d %c l %2d c %4x (%x) ",
			//	 n, (isgraph(n) ? n : ' '), len, tree[n].Code, next_code[len]-1));
		}
	}

	/* ===========================================================================
	 * Compute the optimal bit lengths for a tree and update the total bit length
	 * for the current block.
	 * IN assertion: the fields freq and dad are set, heap[heap_max] and
	 *    above are the tree nodes sorted by increasing frequency.
	 * OUT assertions: the field len is set to the optimal bit length, the
	 *     array bl_count contains the frequencies for each bit length.
	 *     The length opt_len is updated; static_len is also updated if stree is
	 *     not null.
	 */
	private void gen_bitlen (
		int		bl_count[])	/* bit length statistics */
	{
		int h;				/* heap index */
		int n, m;			/* iterate over the tree elements */
		int bits;			/* bit length */
		int xbits;			/* extra bits */
		int f;				/* frequency */
		int overflow = 0;		/* number of elements with bit length too large */

		for (bits = bl_count.length; bits-- > 0;) bl_count[bits] = 0;

		/* In a first pass, compute the optimal bit lengths (which may
		 * overflow in the case of the bit length tree).
		 */
		len[heap[heap_max]] = 0;		/* root of the heap */

		for (h = heap_max+1; h < heap.length; h++) {
			n = heap[h];
			bits = len[dad[n]] + 1;
			if (bits > max_length) {
				bits = max_length;
				overflow++;
			}
			len[n] = bits;			/* This overwrites 'dad', which is no longer needed */

			if (n > max_code) continue;	/* not a leaf node */

			bl_count[bits]++;
			xbits = 0;
			if (n >= extra_base) xbits = extra_bits[n-extra_base];
			f = freq[n];
			opt_len += f * (bits + xbits);
			if (static_tree != null) static_tree.opt_len += f * (static_tree.len[n] + xbits);
		}
		if (overflow == 0) return;

		//ZTracev((z_trace,"\nbit length overflow\n"));
		/* This happens for example on obj2 and pic of the Calgary corpus */

		/* Find the first bit length which could increase: */
		do {
			bits = max_length-1;
			while (bl_count[bits] == 0) bits--;
			bl_count[bits]--;			/* move one leaf down the tree */
			bl_count[bits+1] += 2;			/* move one overflow item as its brother */
			bl_count[max_length]--;
			/* The brother of the overflow item also moves one step up,
			 * but this does not affect bl_count[max_length]
			 */
			overflow -= 2;
		} while (overflow > 0);

		/* Now recompute all bit lengths, scanning in increasing frequency.
		 * h is still equal to HEAP_SIZE. (It is simpler to reconstruct all
		 * lengths instead of fixing only the wrong ones. This idea is taken
		 * from 'ar' written by Haruhiko Okumura.)
		 */
		for (bits = max_length; bits != 0; bits--) {
			n = bl_count[bits];
			while (n != 0) {
				m = heap[--h];
				if (m > max_code) continue;
				if (len[m] != bits) {
					//ZTracev((z_trace,"code %d bits %d->%d\n", m, tree[m].Len, bits));
					opt_len += (bits - len[m]) * freq[m];
					len[m] = bits;
				}
				n--;
			}
		}
	}

	/* ===========================================================================
	 * Remove the smallest element from the heap and recreate the heap with
	 * one less element. Updates heap and heap_len.
	 */
	private int pqremove()
	{
		int	top = heap[1];
		heap[1] = heap[heap_len--];
		pqdownheap(1);

		return top;
	}

	/* ===========================================================================
	 * Compares to subtrees, using the tree depth as tie breaker when
	 * the subtrees have equal frequency. This minimizes the worst case length.
	 */
	private boolean smaller(int n, int m)
	{
		return (freq[n] < freq[m] ||
			(freq[n] == freq[m] && depth[n] <= depth[m]));
	}

	/* ===========================================================================
	 * Restore the heap property by moving down the tree starting at node k,
	 * exchanging a node with the smallest of its two sons if necessary, stopping
	 * when the heap property is re-established (each father smaller than its
	 * two sons).
	 */
	private void pqdownheap (int k)
	{
		int v = heap[k];
		int j = k << 1;		/* left son of k */

		while (j <= heap_len) {
			/* Set j to the smallest of the two sons: */
			if (j < heap_len &&
			    smaller(heap[j+1], heap[j])) {
				j++;
			}
			/* Exit if v is smaller than both sons */
			if (smaller(v, heap[j])) break;

			/* Exchange v with the smallest son */
			heap[k] = heap[j]; k = j;

			/* And continue down the tree, setting j to the left son of k */
			j <<= 1;
		}
		heap[k] = (short)v;
	}

	/* ===========================================================================
	 * Construct one Huffman tree and assigns the code bit strings and lengths.
	 * Update the total bit length for the current block.
	 * IN assertion: the field freq is set for all tree elements.
	 * OUT assertions: the fields len and code are set to the optimal bit length
	 *     and corresponding code. The length opt_len is updated; static_len is
	 *     also updated if stree is not null. The field max_code is set.
	 */
	void build_tree()
	{
		int		bl_count[];
		int		n, m;		/* iterate over heap elements */
		int		node;		/* new node being created */

		/* Construct the initial heap, with least frequent element in
		 * heap[1]. The sons of heap[n] are heap[2*n] and heap[2*n+1].
		 * heap[0] is not used.
		 */
		max_code = -1;				/* largest code with non zero frequency */
		heap_len = 0; heap_max = 2*elems+1;
		heap = new short[heap_max];		/* control allocation even though not local */
		depth = new short[heap_max];

		for (n = 0; n < elems; n++) {
			if (freq[n] != 0) {
				heap[++heap_len] = (short)(max_code = n);
				depth[n] = 0;
			} else {
				len[n] = 0;
			}
		}

		/* The pkzip format requires that at least one distance code exists,
		 * and that at least one bit should be sent even if there is only one
		 * possible code. So to avoid special checks later on we force at least
		 * two codes of non zero frequency.
		 */
		while (heap_len < 2) {
			node = heap[++heap_len] = (short)(max_code < 2 ? ++max_code : 0);
			freq[node] = 1;
			depth[node] = 0;
			opt_len--; if (static_tree != null) static_tree.opt_len -= static_tree.len[node];
			/* node is 0 or 1 so it does not have extra bits */
		}

		/* The elements heap[heap_len/2+1 .. heap_len] are leaves of the tree,
		 * establish sub-heaps of increasing lengths:
		 */
		for (n = heap_len/2; n >= 1; n--) pqdownheap(n);

		/* Construct the Huffman tree by repeatedly combining the least two
		 * frequent nodes. The nodes are moved to the top of the heap for
		 * later use. The nodes will end up in order of decreasing frequency
		 */
		node = elems;				/* next internal node of the tree */
		do {
			n = pqremove();			/* n = node of least frequency */
			m = heap[1];			/* m = node of next least frequency */

			heap[--heap_max] = (short)n;	/* keep the nodes sorted by frequency */
			heap[--heap_max] = (short)m;

			/* Create a new node father of n and m */
			freq[node] = freq[n] + freq[m];
			depth[node] = (short)(Math.max(depth[n], depth[m]) + 1);
			dad[n] = dad[m] = node;
			/* and insert the new node in the heap */
			heap[1] = (short)node++;
			pqdownheap(1);

		} while (heap_len >= 2);

		heap[--heap_max] = heap[1];		/* the most frequent node */

		/* At this point, the fields freq and dad are set. We can now
		 * generate the bit lengths.
		 */
		bl_count = new int[max_length+1];
		gen_bitlen(bl_count);

		/* The field len is now set, we can generate the bit codes */
		gen_codes (bl_count);
		heap = null;				/* Release heap */
		depth = null;
	}
}

/* The Flate encoding engine */
class FlateEncode
{
	//Constants

	static final int	Z_DEFLATED = 8;
	/* The deflate compression method (the only one supported in this version) */

	static final int	Z_NO_COMPRESSION =		0;
	static final int	Z_BEST_SPEED =			1;
	static final int	Z_BEST_COMPRESSION =		9;
	static final int	Z_DEFAULT_COMPRESSION =		-1;
	/* compression levels */

	static final int	Z_FILTERED =			1;
	static final int	Z_HUFFMAN_ONLY =		2;
	static final int	Z_DEFAULT_STRATEGY =		0;
	/* compression strategy; see deflateInit2() below for details */

	static final int	MAX_MEM_LEVEL =			9;
	static final int	DEF_MEM_LEVEL =			5;
	/* mem_level controls the hash table and block length (list of dist codes) */
	static final int	MAX_WBITS =			15;
	static final int	DEF_WBITS =			12;
	/* wbits controls the look-back buffer */
	/* 2*(2**MAX_WBITS) must be <= 65K so Prev can be short */

	static final int	Z_EOF =				-1;
	static final int	Z_PARTIAL_FLUSH =		-2;
	/* Partial flush control */

	private static final int	STORED_BLOCK = 0;
	private static final int	STATIC_TREES = 1;
	private static final int	DYN_TREES = 2;
	/* The three kinds of block type */

	private static final int	MIN_MATCH = 3;
	private static final int	MAX_MATCH = 258;
	/* The minimum and maximum match lengths */
	private static final int	TOO_FAR = 4096;
	/* Matches of length 3 are discarded if their distance exceeds TOO_FAR */
	private static final int	MIN_LOOKAHEAD = 262;	/* (MAX_MATCH+MIN_MATCH+1) */
	/* Minimum amount of lookahead, except at the end of the input file.*/

	private static final int	LENGTH_CODES = 29;
	/* number of length codes, not counting the special END_BLOCK code */

	private static final int	LITERALS = 256;
	/* number of literal bytes 0..255 */

	private static final int	L_CODES = 286;		/* (LITERALS+1+LENGTH_CODES) */
	/* number of Literal or Length codes, including the END_BLOCK code */

	private static final int	D_CODES = 30;
	/* number of distance codes */

	private static final int	BL_CODES = 19;
	/* number of codes used to transfer the bit lengths */

	private static final int	MAX_BITS = 15;
	/* All codes must not exceed MAX_BITS bits */

	private static final int	INIT_STATE = 42;
	private static final int	BUSY_STATE = 113;
	private static final int	FINISH_STATE = 666;
	/* Stream status */

	private static final int	MAX_BL_BITS = 7;
	/* Bit length codes must not exceed MAX_BL_BITS bits */

	private static final int	END_BLOCK = 256;
	/* end of block literal code */

	private static final int	REP_3_6 = 16;
	/* repeat previous bit length 3-6 times (2 bits of repeat count) */

	private static final int	REPZ_3_10 = 17;
	/* repeat a zero length 3-10 times (3 bits of repeat count) */

	private static final int	REPZ_11_138 = 18;
	/* repeat a zero length 11-138 times (7 bits of repeat count) */

	private static final int	extra_lbits[/*LENGTH_CODES*/]	/* extra bits for each length code */
		= {0,0,0,0,0,0,0,0,1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,4,5,5,5,5,0};

	private static final int	extra_dbits[/*D_CODES*/]	/* extra bits for each distance code */
		= {0,0,0,0,1,1,2,2,3,3,4,4,5,5,6,6,7,7,8,8,9,9,10,10,11,11,12,12,13,13};

	private static final int	extra_blbits[/*BL_CODES*/]	/* extra bits for each bit length code */
		= {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,2,3,7};

	private static final int	bl_order[/*BL_CODES*/]
		= {16,17,18,0,8,7,9,6,10,5,11,4,12,3,13,2,14,1,15};
	/* The lengths of the bit length codes are sent in order of decreasing
	 * probability, to avoid transmitting the lengths for unused bit length codes.
	 */

	// Hideous crock for Acrobat compatibility
	private static final int ACRO_BUFSIZE = 4096;
	private long acroLastAvailOut = -1;
	private boolean acroRoundByteCountUp = false;
	private boolean acroEarlyEOF = false;

	/* ===========================================================================
	 * Local data. These are initialized only once.
	 */
	FlateTree	static_ltree;
	/* The static literal tree. Since the bit lengths are imposed, there is no
	 * need for the L_CODES extra codes used during heap construction. However
	 * The codes 286 and 287 are needed to build a canonical tree (see _tr_init
	 * below).
	 */

	FlateTree	static_dtree;
	/* The static distance tree. (Actually a trivial tree since all codes use
	 * 5 bits.)
	 */

	int	dist_code[];			/*512*/
	/* distance codes. The first 256 values correspond to the distances
	 * 3 .. 258, the last 256 values correspond to the top 8 bits of
	 * the 15 bit distances.
	 */

	int	length_code[];			/*MAX_MATCH-MIN_MATCH+1*/
	/* length code for each normalized match length (0 == MIN_MATCH) */

	int	base_length[];			/*LENGTH_CODES*/
	/* First normalized length for each code (0 == MIN_MATCH) */

	int base_dist[];			/*D_CODES*/
	/* First normalized distance for each code (0 == distance of 1) */

	FlateTree	dyn_ltree;
	FlateTree	dyn_dtree;
	FlateTree	bl_tree;

	short		l_buf[];		/* buffer for literals or lengths */
	short		d_buf[];
	/* Buffer for distances. To simplify the code, d_buf and l_buf have
	 * the same number of elements. To use different lengths, an extra flag
	 * array would be necessary.
	 */

	int			lit_bufsize;
	/* Size of match buffer for literals/lengths. There are 4 reasons for
	 * limiting lit_bufsize to 64K:
	 *   - frequencies can be kept in 16 bit counters
	 *   - if compression is not successful for the first block, all input
	 *     data is still in the window so we can still emit a stored block even
	 *     when input comes from standard input. (This can also be done for
	 *     all blocks if lit_bufsize is not greater than 32K.)
	 *   - if compression is not successful for a file smaller than 64K, we can
	 *     even emit a stored file instead of a stored block (saving 5 bytes).
	 *     This is applicable only for zip (not gzip or zlib).
	 *   - creating new Huffman trees less frequently may not provide fast
	 *     adaptation to changes in the input data statistics. (Take for
	 *     example a binary file with poorly compressible code followed by
	 *     a highly compressible string table.) Smaller buffer sizes give
	 *     fast adaptation but have of course the overhead of transmitting
	 *     trees more frequently.
	 *   - I can't count above 4
	 */
	int			last_lit;	/* running index in l_buf */

	int			matches;	/* number of string matches in current block */
	int			last_eob_len;	/* bit length of EOB code for last block */

	int			level;		/* compression level (1..9) */
	int			strategy;	/* favor or force Huffman coding*/
	int			status;		/* encoding status */

	// BitStream handling
	private int		rawBits, rawBitsLeft;
	private OutputStream	out;		/* engine's copy of the output stream */
	private long		totalOut;

	//Methods
	public FlateEncode(OutputStream out)
	{
		int	n;		/* iterates over tree elements */
		int	length;		/* length value */
		int	code;		/* code value */
		int	dist;		/* distance index */

		static_ltree = new FlateTree(null, L_CODES+2, L_CODES, MAX_BITS, extra_lbits, LITERALS+1);
		static_ltree.InitStaticLTree();

		static_dtree = new FlateTree(null, D_CODES, D_CODES, MAX_BITS, extra_dbits, 0);
		static_dtree.InitStaticDTree();

		dyn_ltree =	new FlateTree(static_ltree, 2*L_CODES+1, L_CODES, MAX_BITS, extra_lbits, LITERALS+1);
		dyn_dtree =	new FlateTree(static_dtree, 2*D_CODES+1, D_CODES, MAX_BITS, extra_dbits, 0);
		bl_tree =	new FlateTree(null, 2*BL_CODES+1, BL_CODES, MAX_BL_BITS, extra_blbits, 0);

		dyn_ltree.InitTree();		/* initialize for first block */
		dyn_dtree.InitTree();
		bl_tree.InitTree();

		dist_code =	new int[512];
		length_code =	new int[MAX_MATCH-MIN_MATCH+1];
		base_length =	new int[LENGTH_CODES];
		base_dist =	new int[D_CODES];

		/* number of codes at each bit length for an optimal tree */
		/* Initialize the mapping length (0..255) -> length code (0..28) */
		length = 0;
		for (code = 0; code < LENGTH_CODES-1; code++) {
			base_length[code] = length;
			for (n = 0; n < (1<<extra_lbits[code]); n++) {
				length_code[length++] = code;
			}
		}

		/* Note that the length 255 (match length 258) can be represented
		 * in two different ways: code 284 + 5 bits or code 285, so we
		 * overwrite length_code[255] to use the best encoding:
		 */
		length_code[length-1] = code;

		/* Initialize the mapping dist (0..32K) -> dist code (0..29) */
		dist = 0;
		for (code = 0 ; code < 16; code++) {
			base_dist[code] = dist;
			for (n = 0; n < (1<<extra_dbits[code]); n++) {
				dist_code[dist++] = code;
			}
		}

		dist >>>= 7;	/* from now on, all distances are divided by 128 */
		for ( ; code < D_CODES; code++) {
			base_dist[code] = dist << 7;
			for (n = 0; n < (1<<(extra_dbits[code]-7)); n++) {
				dist_code[256 + dist++] = code;
			}
		}

		/* Initialize the filter */
		this.out = out;
		rawBits = rawBitsLeft = 0;
		last_eob_len = 8;
		level = Z_DEFAULT_COMPRESSION;	/* Initialization not done */
		status = 0;
	}

	/* ===========================================================================
	 * Put out the bits and bytes
	 * There is always room for up to 24 bits!
	 */
	private void sendBits(int b, int n)
		throws IOException
	{
		rawBits += (b & ((1 << n) - 1)) << rawBitsLeft;
		rawBitsLeft += n;
		while (rawBitsLeft >= 8) {
			out.write(rawBits & 0x0ff);
			rawBits >>>= 8;
			rawBitsLeft -= 8;
			++totalOut;
			acroRoundByteCountUp = false;
		}
	}

	private void sendCode(int cv, FlateTree tree)
		throws IOException
	{
		sendBits(tree.code[cv], tree.len[cv]);
	}

	private void putByte(int b)
		throws IOException
	{
		sendBits(b, 8);
	}

	private void putShort(int b)
		throws IOException
	{
		sendBits(b, 16);
	}

	private void flushToByte()
		throws IOException
	{
		if ((rawBitsLeft & 7) != 0) {
			sendBits (0, 8 - (rawBitsLeft & 7));
			acroRoundByteCountUp = true;
		}
	}

	private void putShortMSB(int b)
		throws IOException
	{
		flushToByte();
		sendBits(b >>> 8, 8);
		sendBits(b, 8);
	}

	/* ===========================================================================
	 * Scan a literal or distance tree to determine the frequencies of the codes
	 * in the bit length tree.
	 */
	private void scan_tree (FlateTree tree)
	{
		int n;			/* iterates over all tree elements */
		int prevlen = -1;	/* last emitted length */
		int curlen;		/* length of current code */
		int nextlen;		/* length of next code */
		int count;		/* repeat count of the current code */
		int max_count;		/* max repeat count */
		int min_count;		/* min repeat count */

		tree.len[tree.max_code+1] = -1;		/* guard */
		count = 0; max_count = 7; min_count = 4;
		nextlen = tree.len[0];
		if (nextlen == 0) {max_count = 138; min_count = 3;}

		for (n = 0; n <= tree.max_code; n++) {
			curlen = nextlen; nextlen = tree.len[n+1];
			if (++count < max_count && curlen == nextlen) {
				continue;
			} else if (count < min_count) {
				bl_tree.freq[curlen] += count;
			} else if (curlen != 0) {
				if (curlen != prevlen) bl_tree.freq[curlen]++;
				bl_tree.freq[REP_3_6]++;
			} else if (count <= 10) {
				bl_tree.freq[REPZ_3_10]++;
			} else {
				bl_tree.freq[REPZ_11_138]++;
			}
			count = 0; prevlen = curlen;
			if (nextlen == 0) {
				max_count = 138; min_count = 3;
			} else if (curlen == nextlen) {
				max_count = 6; min_count = 3;
			} else {
				max_count = 7; min_count = 4;
			}
		}
	}

	/* ===========================================================================
	 * Send a literal or distance tree in compressed form.
	 */
	private void send_tree (
				FlateTree	tree)	/* the tree to send */
		throws IOException
	{
		int n;					/* iterates over all tree elements */
		int prevlen = -1;			/* last emitted length */
		int curlen;				/* length of current code */
		int nextlen;				/* length of next code */
		int count;				/* repeat count of the current code */
		int max_count;				/* max repeat count */
		int min_count;				/* min repeat count */

		/* tree.len[tree.max_code+1] = -1; */	/* guard already set */
		count = 0; max_count = 7; min_count = 4;
		nextlen = tree.len[0];
		if (nextlen == 0) {max_count = 138; min_count = 3;}

		for (n = 0; n <= tree.max_code; n++) {
			curlen = nextlen; nextlen = tree.len[n+1];
			if (++count < max_count && curlen == nextlen) {
				continue;
			} else if (count < min_count) {
				do { sendCode(curlen, bl_tree); }while (--count > 0);

			} else if (curlen != 0) {
				if (curlen != prevlen) {
					sendCode(curlen, bl_tree); count--;
				}
				sendCode(REP_3_6, bl_tree); sendBits(count-3, 2);

			} else if (count <= 10) {
				sendCode(REPZ_3_10, bl_tree); sendBits(count-3, 3);

			} else {
				sendCode(REPZ_11_138, bl_tree); sendBits(count-11, 7);
			}
			count = 0; prevlen = curlen;
			if (nextlen == 0) {
				max_count = 138; min_count = 3;
			} else if (curlen == nextlen) {
				max_count = 6; min_count = 3;
			} else {
				max_count = 7; min_count = 4;
			}
		}
	}

	/* ===========================================================================
	 * Send the header for a block using dynamic Huffman trees: the counts, the
	 * lengths of the bit length codes, the literal tree and the distance tree.
	 * IN assertion: lcodes >= 257, dcodes >= 1, blcodes >= 4.
	 */
	private void send_all_trees (
				     int blcodes)	/* maximum bl table index */
		throws IOException
	{
		int rank;				/* index in bl_order */

		sendBits(dyn_ltree.max_code+1 -257, 5);	/* not +255 as stated in appnote.txt */
		sendBits(dyn_dtree.max_code+1 -1, 5);
		sendBits(blcodes+1 -4, 4);		/* not -3 as stated in appnote.txt */
		for (rank = 0; rank <= blcodes; rank++) {
			sendBits(bl_tree.len[bl_order[rank]], 3);
		}

		send_tree(dyn_ltree);			/* literal tree */
		send_tree(dyn_dtree);			/* distance tree */
	}

	/* ===========================================================================
	 * Construct the Huffman tree for the bit lengths and return the index in
	 * bl_order of the last bit length code to send.
	 */
	private int build_bl_tree ()
	{
		int	max_blindex;		/* index of last bit length code of non zero freq */

		/* Determine the bit length frequencies for literal and distance trees */
		scan_tree(dyn_ltree);
		scan_tree(dyn_dtree);

		/* Build the bit length tree: */
		bl_tree.build_tree();
		/* opt_len now includes the length of the tree representations, except
		 * the lengths of the bit lengths codes and the 5+5+4 bits for the counts.
		 */

		/* Determine the number of bit length codes to send. The pkzip format
		 * requires that at least 4 bit length codes be sent. (appnote.txt says
		 * 3 but the actual value used is 4.)
		 */
		for (max_blindex = BL_CODES-1; max_blindex >= 3; max_blindex--) {
			if (bl_tree.len[bl_order[max_blindex]] != 0) break;
		}
		/* Update opt_len to include the bit length tree and counts */
		bl_tree.opt_len += 3*(max_blindex+1) + 5+5+4;

		return max_blindex;
	}

	/* ===========================================================================
	 * Copy a stored block, storing first the length and its
	 * one's complement if requested.
	 */
	private void copy_block (
				 byte		buf[],		/* the input data */
				 int		buf_start,	/* starting position in buf */
				 int		len,		/* its length */
				 boolean	header)		/* true if block header must be written */
		throws IOException
	{
		flushToByte();			/* align on byte boundary */
		last_eob_len = 8;		/* enough lookahead for inflate */

		if (header) {
			putShort(len);
			putShort(~len);
		}
		for (int i = 0; i < len; i++)
			putByte(buf[buf_start + i]);
	}

	/* ===========================================================================
	 * Send a stored block
	 */
	private void tr_stored_block (
				      byte	buf[],			/* input block */
				      int	buf_start,		/* starting position in buf */
				      int	stored_len,		/* length of input block */
				      boolean	eof)			/* true if this is the last block for a file */
		throws IOException
	{
		sendBits((STORED_BLOCK<<1)+(eof ? 1 : 0), 3);	/* send block type */
		copy_block(buf, buf_start, stored_len, true);	/* with header */
	}

	/* ===========================================================================
	 * Determine the best encoding for the current block: dynamic trees, static
	 * trees or store, and output the encoded block to the zip file. This function
	 * returns the total compressed length for the file so far.
	 */
	private void tr_flush_block (
				     boolean	eof)			/* true if this is the last block for a file */
		throws IOException
	{
		int		opt_len, static_len;			/* total lengths in bits */
		int		opt_lenb, static_lenb;			/* opt_len and static_len in bytes */
		int		max_blindex = 0;			/* index of last bit length code of non zero freq */
		int		stored_len = strstart - block_start;
		long		beforeCount = getAcroTotalOut();

		/* Build the Huffman trees unless a stored block is forced */
		if (level > 0) {

			/* Construct the literal and distance trees */
			dyn_ltree.build_tree();
			dyn_dtree.build_tree();
			/* At this point, opt_len and static_len are the total bit lengths of
			 * the compressed block data, excluding the tree representations.
			 */

			/* Build the bit length tree for the above two trees, and get the index
			 * in bl_order of the last bit length code to send.
			 */
			max_blindex = build_bl_tree();

			opt_len = dyn_ltree.opt_len + dyn_dtree.opt_len + bl_tree.opt_len;
			static_len = static_ltree.opt_len + static_dtree.opt_len;

			/* Determine the best encoding. Compute first the block length in bytes*/
			opt_lenb = (opt_len+3+7)>>>3;
			static_lenb = (static_len+3+7)>>>3;

			if (static_lenb <= opt_lenb)
				opt_lenb = static_lenb;

		} else {
			opt_lenb = static_lenb = stored_len + 5;	/* force a stored block */
		}

		if (stored_len+4 <= opt_lenb && block_start >= 0) {
			/* 4: two words for the lengths */
			/* The test block_start >= 0 is only necessary if LIT_BUFSIZE > WSIZE.
			 * Otherwise we can't have processed more than WSIZE input bytes since
			 * the last block flush, because compression would have been
			 * successful. If LIT_BUFSIZE <= WSIZE, it is never too late to
			 * transform a block into a stored block.
			 */
			tr_stored_block(window, block_start, stored_len, eof);

		} else if (static_lenb == opt_lenb) {
			sendBits((STATIC_TREES<<1)+(eof ? 1 : 0), 3);
			compress_block(static_ltree, static_dtree);

		} else {
			sendBits((DYN_TREES<<1)+(eof ? 1 : 0), 3);
			send_all_trees(max_blindex);
			compress_block(dyn_ltree, dyn_dtree);
		}

		dyn_ltree.InitTree();		/* initialize for next block */
		dyn_dtree.InitTree();
		bl_tree.InitTree();
		block_start = strstart;
		last_lit = 0;

		if (eof) {
			flushToByte();		/* align on byte boundary */
		}
		getAcroAvailOut(beforeCount);
	}

	/* ===========================================================================
	 * Save the match info and tally the frequency counts. Return true if
	 * the current block must be flushed.
	 */
	private int d_code(int dist)
	{
		return ((dist) < 256 ? dist_code[dist] : dist_code[256+((dist)>>>7)]);
		/* Mapping from a distance to a distance code. dist is the distance - 1 and
		 * must not have side effects. dist_code[256] and dist_code[257] are never
		 * used.
		 */
	}

	private boolean tr_tally (
				  int		dist,	/* distance of matched string */
				  int		lc)	/* match length-MIN_MATCH or unmatched char (if dist==0) */
	{
		if (dist == 0) lc &= 0x0ff;		/* literal */
		d_buf[last_lit] = (short)dist;
		l_buf[last_lit++] = (short)lc;

		if (dist == 0) {
			/* lc is the unmatched char */
			dyn_ltree.freq[lc]++;
		} else {
			matches++;
			/* Here, lc is the match length - MIN_MATCH */
			dist--;		/* dist = match distance - 1 */
			//ZAssert((ush)dist < (ush)MAX_DIST(s) &&
			//	   (ush)lc <= (ush)(MAX_MATCH-MIN_MATCH) &&
			//	   (ush)d_code(dist) < (ush)D_CODES, "_tr_tally: bad match");

			dyn_ltree.freq[length_code[lc]+LITERALS+1]++;
			dyn_dtree.freq[d_code(dist)]++;
		}

		/* Try to guess if it is profitable to stop the current block here */
		if (level > 2 && (last_lit & 0x0fff) == 0) {
			/* Compute an upper bound for the compressed length */
			int out_length = last_lit*8;
			int in_length = strstart - block_start;
			int dcode;
			for (dcode = 0; dcode < D_CODES; dcode++) {
				out_length += dyn_dtree.freq[dcode] *
					(5+extra_dbits[dcode]);
			}
			out_length >>>= 3;
			//ZTracev((z_trace,"\nlast_lit %u, in %ld, out ~%ld(%ld%%) ",
			//	   s->last_lit, in_length, out_length,
			//	   100L - out_length*100L/in_length));
			if (matches < last_lit/2 && out_length < in_length/2) return true;
		}
		return (last_lit == lit_bufsize-1);
		/* We avoid equality with lit_bufsize because of wraparound at 64K
		 * on 16 bit machines and because stored blocks are restricted to
		 * 64K-1 bytes.
		 */
	}

	/* ===========================================================================
	 * Send the block data compressed using the given Huffman trees
	 */
	private void compress_block (
				     FlateTree	ltree,		/* literal tree */
				     FlateTree	dtree)		/* distance tree */
		throws IOException
	{
		int dist;			/* distance of matched string */
		int lc;				/* match length or unmatched char (if dist == 0) */
		int lx = 0;			/* running index in l_buf */
		int code;			/* the code to send */
		int extra;			/* number of extra bits to send */

		if (last_lit != 0) do {
			dist = d_buf[lx];
			lc = l_buf[lx++];

			if (dist == 0) {
				sendCode(lc, ltree);					/* send a literal byte */
				//	ZTracecv(isgraph(lc), (z_trace," '%c' ", lc));
			} else {
				/* Here, lc is the match length - MIN_MATCH */
				code = length_code[lc];
				sendCode(code+LITERALS+1, ltree);			/* send the length code */
				extra = extra_lbits[code];
				if (extra != 0) {
					lc -= base_length[code];
					sendBits(lc, extra);				/* send the extra length bits */
				}
				dist--;							/* dist is now the match distance - 1 */
				code = d_code(dist);
				//	ZAssert (code < D_CODES, "bad d_code");

				sendCode(code, dtree);					/* send the distance code */
				extra = extra_dbits[code];
				if (extra != 0) {
					dist -= base_dist[code];
					sendBits(dist, extra);				/* send the extra distance bits */
				}
			}								/* literal or match pair ? */

		} while (lx < last_lit);

		sendCode(END_BLOCK, ltree);
		last_eob_len = ltree.len[END_BLOCK];
	}

	/*
	 *  ALGORITHM
	 *
	 *      The "deflation" process depends on being able to identify portions
	 *      of the input text which are identical to earlier input (within a
	 *      sliding window trailing behind the input currently being processed).
	 *
	 *      The most straightforward technique turns out to be the fastest for
	 *      most input files: try all possible matches and select the longest.
	 *      The key feature of this algorithm is that insertions into the string
	 *      dictionary are very simple and thus fast, and deletions are avoided
	 *      completely. Insertions are performed at each input character, whereas
	 *      string matches are performed only when the previous match ends. So it
	 *      is preferable to spend more time in matches to allow very fast string
	 *      insertions and avoid deletions. The matching algorithm for small
	 *      strings is inspired from that of Rabin & Karp. A brute force approach
	 *      is used to find longer strings when a small match has been found.
	 *      A similar algorithm is used in comic (by Jan-Mark Wams) and freeze
	 *      (by Leonid Broukhis).
	 *	A previous version of this file used a more sophisticated algorithm
	 *      (by Fiala and Greene) which is guaranteed to run in linear amortized
	 *      time, but has a larger average cost, uses more memory and is patented.
	 *      However the F&G algorithm may be faster for some highly redundant
	 *      files if the parameter max_chain_length (described below) is too large.
	 *
	 *  ACKNOWLEDGEMENTS
	 *
	 *      The idea of lazy evaluation of matches is due to Jan-Mark Wams, and
	 *      I found it in 'freeze' written by Leonid Broukhis.
	 *      Thanks to many people for bug reports and testing.
	 *
	 *  REFERENCES
	 *
	 *      Deutsch, L.P.,"'Deflate' Compressed Data Format Specification".
	 *      Available in ftp.uu.net:/pub/archiving/zip/doc/deflate-1.1.doc
	 *
	 *      A description of the Rabin and Karp algorithm is given in the book
	 *	"Algorithms" by R. Sedgewick, Addison-Wesley, p252.
	 *
	 *      Fiala, E.R., and Greene,D.H.
	 *	Data Compression with Finite Windows, Comm.ACM, 32,4 (1989) 490-595
	 *
	 */

	/* $Id: deflate.c,v 1.15 1996/07/24 13:40:58 me Exp $ */
	/* McCreight: 05 Aug 96 */
	/*
	  If you use the zlib library in a product, an acknowledgment is welcome
	  in the documentation of your product. If for some reason you cannot
	  include such an acknowledgment, I would appreciate that you keep this
	  copyright string in the executable of your product.
	*/

	/* typedef struct internal_state { */
	int		noheader;		/* suppress zlib header and adler32 */

	int		w_size;			/* LZ77 window size (32K by default) */
	int		w_bits;			/* log2(w_size) (8..16) */
	int		w_mask;			/* w_size - 1 */
	int		max_dist;
	/* In order to simplify the code, particularly on 16 bit machines, match
	 * distances are limited to MAX_DIST instead of WSIZE.
	 */

	byte	window[];
	/* Sliding window. Input bytes are read into the second half of the window,
	 * and move to the first half later to keep a dictionary of at least wSize
	 * bytes. With this organization, matches are limited to a distance of
	 * wSize-MAX_MATCH bytes, but this ensures that IO is always
	 * performed with a length multiple of the block size. Also, it limits
	 * the window size to 64K, which is quite useful on MSDOS.
	 * To do: use the user input buffer as sliding window.
	 */

	int		window_size;
	/* Actual size of window: 2*wSize
	 */

	short		prev[];
	/* Link to older string with same hash index. To limit the size of this
	 * array to 64K, this link is maintained only for the last <wSize> strings.
	 * An index in this array is thus a window index modulo <wSize>.
	 */

	short		head[];			/* Heads of the hash chains or NIL. */

	int		ins_h;			/* hash index of string to be inserted */
	int		hash_size;		/* number of elements in hash table */
	int		hash_bits;		/* log2(hash_size) */
	int		hash_mask;		/* hash_size-1 */

	int		hash_shift;
	/* Number of bits by which ins_h must be shifted at each input
	 * step. It must be such that after MIN_MATCH steps, the oldest
	 * byte no longer takes part in the hash key, that is:
	 *   hash_shift * MIN_MATCH >= hash_bits
	 */

	int		block_start;
	/* Window position at the beginning of the current output block. Gets
	 * negative when the window is moved backwards.
	 */

	int		match_length;		/* length of best match */
	int		prev_match;		/* previous match */
	boolean		match_available;	/* set if previous match exists */
	int		strstart;		/* start of string to insert */
	int		match_start;		/* start of matching string */
	int		lookahead;		/* number of valid bytes ahead in window */

	int		prev_length;
	/* Length of the best match at previous step. Matches not greater than this
	 * are discarded. This is used in the lazy match evaluation.
	 */

	int		max_chain_length;
	/* To speed up deflation, hash chains are never searched beyond this
	 * length. A higher limit improves compression ratio but degrades the
	 * speed.
	 */

	int		max_lazy_match;
	/* Attempt to find a better match only when the current match is strictly
	 * smaller than this value. This mechanism is used only for compression
	 * levels >= 4.
	 */
	//define max_insert_length max_lazy_match
	/* Insert new strings in the hash table only if the match length is not
	 * greater than this length. This saves time but degrades compression.
	 * max_insert_length is used only for compression levels <= 3.
	 */

	int		good_match;
	/* Use a faster search when the previous match is longer than this */

	int		nice_match;	/* Stop searching when current match exceeds this */
	//} FAR deflate_state;

	/* ===========================================================================
	 * Local data
	 */

	// Adler32 computation
	private static final int	Adler_BASE = 65521;	/* largest prime smaller than 65536 */
	private int			adler_s1, adler_s2;

	private void Adler32(int b)
	{
		adler_s1 += b & 0x0ff;
		if (adler_s1 >= Adler_BASE)
			adler_s1 -= Adler_BASE;

		adler_s2 += adler_s1;
		if (adler_s2 >= Adler_BASE)
			adler_s2 -= Adler_BASE;
	}

	/* ===========================================================================
	 * Update a hash value with the given input byte
	 * IN  assertion: all calls to to UPDATE_HASH are made with consecutive
	 *    input characters, so that a running hash key can be computed from the
	 *    previous key instead of complete recalculation each time. (This is
	 *    because hash_shift * (MIN_MATCH+1) > hash_bits
	 */
	private void UPDATE_HASH(int c)
	{
		ins_h = ((ins_h << hash_shift) ^ (c & 0x0ff)) & hash_mask;
	}

	/* ===========================================================================
	 * Insert string str in the dictionary and set match_head to the previous head
	 * of the hash chain (the most recent string with same hash key). Return
	 * the previous length of the hash chain.
	 * IN  assertion: all calls to to INSERT_STRING are made with consecutive
	 *    input characters and the first MIN_MATCH bytes of str are valid
	 *    (except for the last MIN_MATCH-1 bytes of the input file).
	 */
	private int INSERT_STRING()
	{
		int hash_head;

		UPDATE_HASH(window[strstart + MIN_MATCH-1]);
		hash_head = prev[strstart & w_mask] = head[ins_h];
		head[ins_h] = (short)strstart;
		return hash_head & 0x0ffff;
	}

	/* ===========================================================================
	 * Initialize the hash table (avoiding 64K overflow for 16 bit systems).
	 * prev[] will be initialized on the fly.
	 */
	private void CLEAR_HASH()
	{
		for (int i = head.length; i-- > 0;)
			head[i] = 0;
	}

	/* ========================================================================= */
	void DeflateInit (
			  int		compression_level)
	{
		DeflateInit2(compression_level, DEF_WBITS, DEF_MEM_LEVEL, Z_DEFAULT_STRATEGY);
	}

	/* ========================================================================= */
	void DeflateInit2 (
			   int		compression_level,
			   int		windowBits,
			   int		memLevel,
			   int		strategy)
	{
		if (level != Z_DEFAULT_COMPRESSION)
			return;		/* We have already initialized once */

		level = compression_level;
		if (level == Z_DEFAULT_COMPRESSION || level < 0 || level > 9)
			level = 6;

		if (memLevel < 1 || memLevel > MAX_MEM_LEVEL)
			memLevel = DEF_MEM_LEVEL;

		noheader = 0;
		if (windowBits < 0) {			/* undocumented feature: suppress zlib header */
			noheader = 1;
			windowBits = -windowBits;
		}
		if (windowBits < 8)
			windowBits = 8;
		else if (windowBits > MAX_WBITS)
			windowBits = DEF_WBITS;

		if (strategy < 0 || strategy > Z_HUFFMAN_ONLY) {
			strategy = Z_DEFAULT_STRATEGY;
		}
		this.strategy = strategy;

		w_bits = windowBits;
		w_size = 1 << w_bits;
		w_mask = w_size - 1;
		max_dist = w_size-MIN_LOOKAHEAD;

		hash_bits = memLevel + 7;
		hash_size = 1 << hash_bits;
		hash_mask = hash_size - 1;
		hash_shift = ((hash_bits+MIN_MATCH-1)/MIN_MATCH);

		window = new byte[2*w_size];
		prev = new short[w_size];
		head = new short[hash_size];

		lit_bufsize = 1 << (memLevel + 6);		/* 16K elements by default */
		l_buf = new short[lit_bufsize];
		d_buf = new short[lit_bufsize];

		deflateReset();
	}

	/* ========================================================================= */
	void deflateReset ()
	{
		if (noheader < 0) {
			noheader = 0;	/* was set to -1 by deflate(..., Z_FINISH); */
		}
		status = noheader != 0 ? BUSY_STATE : INIT_STATE;

		lm_init();
	}

	/* ===========================================================================
	 * Initialize the "longest match" routines for a new zlib stream
	 */
	private void lm_init ()
	{
		window_size = 2*w_size;

		CLEAR_HASH();		/* Clear Hash table */

		/* Set the default configuration parameters:
		 * Values for max_lazy_match, good_match and max_chain_length, depending on
		 * the desired pack level (0..9). The values given below have been tuned to
		 * exclude worst case performance for pathological files. Better values may be
		 * found for specific files.
		 */

		switch (level)
			{
			case 0:
				good_match = 0;		max_lazy_match = 0;		nice_match = 0;		max_chain_length = 0;
				break;

			case 1:
				good_match = 4;		max_lazy_match = 4;		nice_match = 8;		max_chain_length = 4;
				break;

			case 2:
				good_match = 4;		max_lazy_match = 5;		nice_match = 16;	max_chain_length = 8;
				break;

			case 3:
				good_match = 4;		max_lazy_match = 6;		nice_match = 32;	max_chain_length = 32;
				break;

			case 4:
				good_match = 4;		max_lazy_match = 4;		nice_match = 16;	max_chain_length = 16;
				break;

			case 5:
				good_match = 8;		max_lazy_match = 16;	nice_match = 32;	max_chain_length = 32;
				break;

			default:
			case 6:
				good_match = 8;		max_lazy_match = 16;	nice_match = 128;	max_chain_length = 128;
				break;

			case 7:
				good_match = 8;		max_lazy_match = 32;	nice_match = 128;	max_chain_length = 256;
				break;

			case 8:
				good_match = 32;	max_lazy_match = 128;	nice_match = 258;	max_chain_length = 1024;
				break;

			case 9:
				good_match = 32;	max_lazy_match = 258;	nice_match = 258;	max_chain_length = 4096;
				break;
			}

		strstart = 0;
		block_start = 0;
		lookahead = 0;
		match_length = prev_length = MIN_MATCH-1;
		match_available = false;
		ins_h = 0;
	}

	/* ===========================================================================
	 * Set match_start to the longest match starting at the given string and
	 * return its length. Matches shorter or equal to prev_length are discarded,
	 * in which case the result is equal to prev_length and match_start is
	 * garbage.
	 * IN assertions: cur_match is the head of the hash chain for the current
	 *   string (strstart) and its distance is <= MAX_DIST, and prev_length >= 1
	 * OUT assertion: the match length is not greater than s->lookahead.
	 */
	private int longest_match (
				   int		cur_match)		/* current match */
	{
		int		chain_length = max_chain_length;	/* max hash chain length */
		int		scan = strstart;			/* current string */
		int		match;					/* matched string */
		int		len;					/* length of current match */
		int		best_len = prev_length;			/* best match length so far */
		int		nice_match = this.nice_match;		/* stop if match long enough */
		int		limit = strstart > max_dist ?
			strstart - max_dist : 0;
		/* Stop when cur_match becomes <= limit. To simplify the code,
		 * we prevent matches with the string of window index 0.
		 */

		int	strend = strstart + MAX_MATCH;
		byte	scan_end1 = window[scan + best_len-1];
		byte	scan_end = window[scan + best_len];

		/* Do not waste too much time if we already have a good match: */
		if (prev_length >= good_match) {
			chain_length >>= 2;
		}
		/* Do not look for matches beyond the end of the input. This is necessary
		 * to make deflate deterministic.
		 */
		if (nice_match > lookahead) nice_match = lookahead;

		//ZAssert((ulg)s->strstart <= s->window_size-MIN_LOOKAHEAD, "need lookahead");

		do {
			//ZAssert(cur_match < s->strstart, "no future");
			match = cur_match;

			/* Skip to next match if the match length cannot increase
			 * or if the match length is less than 2:
			 */
			if (window[match + best_len] != scan_end ||
			    window[match + best_len-1] != scan_end1 ||
			    window[match] != window[scan] ||
			    window[++match] != window[scan + 1]) continue;

			/* The check at best_len-1 can be removed because it will be made
			 * again later. (This heuristic is not always a win.)
			 * It is not necessary to compare scan[2] and match[2] since they
			 * are always equal when the other bytes match, given that
			 * the hash keys are equal and that HASH_BITS >= 8.
			 */
			scan += 2; match++;
			//ZAssert(*scan == *match, "match[2]?");

			do {
			} while (window[++scan] == window[++match] &&
				 scan < strend);

			//ZAssert(scan <= s->window+(unsigned)(s->window_size-1), "wild scan");
			len = scan - strstart;
			scan = strstart;

			if (len > best_len) {
				match_start = cur_match;
				best_len = len;
				if (len >= nice_match) break;
				scan_end1 = window[scan + best_len-1];
				scan_end = window[scan + best_len];
			}
		} while ((cur_match = prev[cur_match & w_mask]) > limit
			 && --chain_length != 0);

		return (best_len <= lookahead) ? best_len : lookahead;
	}

	/* ===========================================================================
	 * Fill the window when the lookahead becomes insufficient.
	 * Updates strstart and lookahead.
	 *
	 * IN assertion: lookahead < MIN_LOOKAHEAD
	 * OUT assertions: strstart <= window_size-MIN_LOOKAHEAD
	 *    At least one byte has been read, or avail_in == 0; reads are
	 *    performed for at least two bytes (required for the zip translate_eol
	 *    option -- not supported here).
	 */
	private void fill_window (int ch)
	{
		/* Make sure there is enough buffer room for a maximum match */
		if (strstart >= w_size+max_dist) {
			int n, p;

			System.arraycopy(window, w_size, window, 0, w_size);
			match_start -= w_size;
			strstart    -= w_size;	/* we now have strstart >= MAX_DIST */
			block_start -= w_size;

			/* Slide the hash table */
			for (n = 0; n < hash_size; n++) {
				p = (head[n] & 0x0ffff) - w_size;
				head[n] = (short)((p >= 0) ? p : 0);
			}

			for (n = 0; n < w_size; n++) {
				p = (prev[n] & 0x0ffff) - w_size;
				prev[n] = (short)((p >= 0) ? p : 0);
				/* If n is not on any hash chain, prev[n] is garbage but
				 * its value will never be used.
				 */
			}
		}

		window[strstart + lookahead++] = (byte)ch;
	}

	/* ===========================================================================
	 * Copy without compression as much as possible from the input stream.
	 * This function does not insert new strings in the dictionary since
	 * uncompressible data is probably not useful. This function is used
	 * only for the level=0 compression option.
	 * NOTE: this function should be optimized to avoid extra copying.
	 */
	private void deflate_stored (int ch)
		throws IOException
	{
		boolean flush = ch < 0;

		if (!flush) {
			window[strstart++] = (byte)ch;

			if (strstart < window_size && strstart - block_start < 65535)
				return;
		}

		if (strstart - block_start > 0) {
			tr_flush_block (ch == -1);	/* True for EOF, false for partial flush */
			strstart = block_start = 0;
		}
	}

	/* ===========================================================================
	 * Compress as much as possible from the input stream, return the current
	 * block state.
	 * This function does not perform lazy evaluation of matches and inserts
	 * new strings in the dictionary only for unmatched strings or for short
	 * matches. It is used only for the fast compression options.
	 */
	private void deflate_fast (int ch)
		throws IOException
	{
		boolean	flush = ch < 0;
		boolean	bflush;
		int	hash_head = 0;		/* head of the hash chain */

		if (!flush) {			/* accumulate bytes */
			fill_window(ch);
			if (lookahead < MIN_LOOKAHEAD)
				return;
		}

		if (lookahead > 0) do {
			if (lookahead >= MIN_MATCH) {
				ins_h = window[strstart];
				UPDATE_HASH(window[strstart+1]);
				//#if MIN_MATCH != 3
				//	Call UPDATE_HASH() MIN_MATCH-3 more times
				//#endif
				/* Insert the string window[strstart .. strstart+2] in the
				 * dictionary, and set hash_head to the head of the hash chain:
				 */
				hash_head = INSERT_STRING();
			}
			/* If the whole input has less than MIN_MATCH bytes, ins_h is garbage,
			 * but this is not important since only literal bytes will be emitted.
			 */

			/* Find the longest match, discarding those <= prev_length.
			 * At this point we have always match_length < MIN_MATCH
			 */
			if (hash_head != 0 && strstart - hash_head <= max_dist) {
				/* To simplify the code, we prevent matches with the string
				 * of window index 0 (in particular we have to avoid a match
				 * of the string with itself at the start of the input file).
				 */
				if (strategy != Z_HUFFMAN_ONLY) {
					match_length = longest_match (hash_head);
				}
				/* longest_match() sets match_start */
			}

			if (match_length >= MIN_MATCH) {
				bflush = tr_tally(strstart - match_start, match_length - MIN_MATCH);
				lookahead -= match_length;

				/* Insert new strings in the hash table only if the match length
				 * is not too large. This saves time but degrades compression.
				 */
				if (match_length <= max_lazy_match &&
				    lookahead >= MIN_MATCH) {
					match_length--;		/* string at strstart already in hash table */
					do {
						strstart++;
						INSERT_STRING();
						/* strstart never exceeds WSIZE-MAX_MATCH, so there are
						 * always MIN_MATCH bytes ahead.
						 */
					} while (--match_length != 0);
					strstart++;
				} else {
					strstart += match_length;
					match_length = 0;
					ins_h = window[strstart];
					UPDATE_HASH(window[strstart+1]);
					//#if MIN_MATCH != 3
					//				Call UPDATE_HASH() MIN_MATCH-3 more times
					//#endif
					/* If lookahead < MIN_MATCH, ins_h is garbage, but it does not
					 * matter since it will be recomputed at next deflate call.
					 */
				}
			} else {
				/* No match, output a literal byte */
				bflush = tr_tally(0, window[strstart]);
				lookahead--;
				strstart++;
			}

			if (bflush)
				tr_flush_block(false);
		} while (flush && lookahead > 0);		/* Repeat at end of data */

		if (flush) {
			boolean eof = (ch == Z_EOF);
			if (ch == Z_PARTIAL_FLUSH && acroLastAvailOut == 0) {
				eof = true;
				acroEarlyEOF = true;
			}
			tr_flush_block (eof);
		} else {
			acroLastAvailOut = -1;
		}
	}

	/* ===========================================================================
	 * Same as above, but achieves better compression. We use a lazy
	 * evaluation for matches: a match is finally adopted only if there is
	 * no better match at the next window position.
	 */
	private void deflate_slow (int ch)
		throws IOException
	{
		boolean	flush = ch < 0;
		boolean	bflush = false;
		int	hash_head = 0;		/* head of the hash chain */

		if (!flush) {			/* accumulate bytes */
			fill_window(ch);
			if (lookahead < MIN_LOOKAHEAD)
				return;
		}

		if (lookahead > 0) do {
			if (lookahead >= MIN_MATCH) {
				ins_h = window[strstart];
				UPDATE_HASH(window[strstart+1]);
				//#if MIN_MATCH != 3
				//	Call UPDATE_HASH() MIN_MATCH-3 more times
				//#endif
				/* Insert the string window[strstart .. strstart+2] in the
				 * dictionary, and set hash_head to the head of the hash chain:
				 */
				hash_head = INSERT_STRING();
			}
			/* If the whole input has less than MIN_MATCH bytes, ins_h is garbage,
			 * but this is not important since only literal bytes will be emitted.
			 */

			/* Find the longest match, discarding those <= prev_length.
			 */
			prev_length = match_length; prev_match = match_start;
			match_length = MIN_MATCH-1;

			if (hash_head != 0 && prev_length < max_lazy_match &&
			    strstart - hash_head <= max_dist) {
				/* To simplify the code, we prevent matches with the string
				 * of window index 0 (in particular we have to avoid a match
				 * of the string with itself at the start of the input file).
				 */
				if (strategy != Z_HUFFMAN_ONLY) {
					match_length = longest_match (hash_head);
				}
				/* longest_match() sets match_start */

				if (match_length <= 5 && (strategy == Z_FILTERED ||
							  (match_length == MIN_MATCH &&
							   strstart - match_start > TOO_FAR))) {

					/* If prev_match is also MIN_MATCH, match_start is garbage
					 * but we will ignore the current match anyway.
					 */
					match_length = MIN_MATCH-1;
				}
			}

			/* If there was a match at the previous step and the current
			 * match is not better, output the previous match:
			 */
			if (prev_length >= MIN_MATCH && match_length <= prev_length) {
				int max_insert = strstart + lookahead - MIN_MATCH;
				/* Do not insert strings in hash table beyond this. */

				bflush = tr_tally(strstart -1 - prev_match, prev_length - MIN_MATCH);

				/* Insert in hash table all strings up to the end of the match.
				 * strstart-1 and strstart are already inserted. If there is not
				 * enough lookahead, the last two strings are not inserted in
				 * the hash table.
				 */
				lookahead -= prev_length-1;
				prev_length -= 2;
				do {
					if (++strstart <= max_insert) {
						INSERT_STRING();
					}
				} while (--prev_length != 0);
				match_available = false;
				match_length = MIN_MATCH-1;
				strstart++;

				if (bflush)
					tr_flush_block(false);
			} else if (match_available) {
				/* If there was no match at the previous position, output a
				 * single literal. If there was a match but the current match
				 * is longer, truncate the previous match to a single literal.
				 */
				//ZTracevv((z_trace,"%c", s->window[s->strstart-1]));
				bflush = tr_tally (0, window[strstart-1]);

				if (bflush)
					tr_flush_block(false);
				strstart++;
				lookahead--;
			} else {
				/* There is no previous match to compare with, wait for
				 * the next step to decide.
				 */
				match_available = true;
				strstart++;
				lookahead--;
			}

		} while (flush && lookahead > 0);		/* Repeat at end of data */

		if (flush) {
			if (match_available) {
				tr_tally (0, window[strstart-1]);
				match_available = false;
			}
			boolean eof = (ch == Z_EOF);
			if (ch == Z_PARTIAL_FLUSH && acroLastAvailOut == 0) {
				eof = true;
				acroEarlyEOF = true;
			}
			tr_flush_block (eof);
		} else {
			acroLastAvailOut = -1;
		}
	}

	/* ========================================================================= */
	void write (int ch)
		throws IOException
	{
		if (ch >= 0) {
			if (level == Z_DEFAULT_COMPRESSION)	/* Output not yet initialized */
				DeflateInit(Z_DEFAULT_COMPRESSION);

			/* Write the zlib header */
			if (status == INIT_STATE) {

				int header = (Z_DEFLATED + ((w_bits-8) << 4)) << 8;
				int level_flags = (level-1) >> 1;

				if (level_flags < 0 || level_flags > 3) level_flags = 3;
				header |= (level_flags << 6);
				header += 31 - (header % 31);

				status = BUSY_STATE;
				putShortMSB(header);

				adler_s1 = 1;	adler_s2 = 0;
			}

			Adler32(ch);

			if (level == 0)
				deflate_stored(ch & 0x0ff);
			else if (level <= 3)
				deflate_fast(ch & 0x0ff);
			else
				deflate_slow(ch & 0x0ff);
		}
		else if (status != FINISH_STATE) {
			if (!acroEarlyEOF) {
				if (level == 0)
					deflate_stored(ch);
				else if (level <= 3)
					deflate_fast(ch);
				else
					deflate_slow(ch);
			}

			if (ch == -1) {
				status = FINISH_STATE;

				/* Write the zlib trailer (adler32) */
				putShortMSB(adler_s2);
				putShortMSB(adler_s1);
			} else {
				if (acroLastAvailOut != 0 && !acroEarlyEOF) {
					/* Partial flush */
					sendBits(STATIC_TREES<<1, 3);
					sendCode(END_BLOCK, static_ltree);
					if (1 + last_eob_len + 10 - rawBitsLeft < 9) {
						sendBits(STATIC_TREES<<1, 3);
						sendCode(END_BLOCK, static_ltree);
					}
					last_eob_len = 7;
				}
			}
		}
	}

	long getTotalOut()
	{
		return totalOut;
	}

	private long getAcroTotalOut()
	{
		long total = totalOut;
		if (total == 2 || acroRoundByteCountUp)
			return total;
		if (total != 0 && (total & 1) == 0 && rawBitsLeft == 0)
			return total - 2;
		return total & ~1;
	}

	private long getAcroAvailOut(long countBefore)
	{
		long availOut;
		long countAfter = getAcroTotalOut();
		if (countAfter !=0 && (countAfter % ACRO_BUFSIZE) == 0)
			availOut = 0;
		else if (countAfter % ACRO_BUFSIZE < countBefore % ACRO_BUFSIZE)
			availOut = 0;
		else
			availOut = ACRO_BUFSIZE - (countAfter % ACRO_BUFSIZE);
		acroLastAvailOut = availOut;
		return availOut;
	}
}

public class FlateOutputStream extends FilterOutputStream
{
	// Variables
	FlateEncode fl;

	// Constructors
	/**
	 * Creates an output stream filter.
	 * @param out	the output stream
	 */
	public FlateOutputStream(OutputStream out, FilterParams p)
	{
		super(out);
		fl = new FlateEncode(out);

		int compression =	FlateEncode.Z_DEFAULT_COMPRESSION;
		int strategy =		FlateEncode.Z_DEFAULT_STRATEGY;
		int windowBits =	FlateEncode.DEF_WBITS;
		int memLevel =		FlateEncode.DEF_MEM_LEVEL;
		if (p != null) {
			if (p.containsKey(FilterParams.CompressionLevel_K))
				compression = ((Integer)p.get(FilterParams.CompressionLevel_K)).intValue();

			if (p.containsKey(FilterParams.CompressionStrategy_K))
				strategy = ((Integer)p.get(FilterParams.CompressionStrategy_K)).intValue();

			if (p.containsKey(FilterParams.WindowBits_K))
				windowBits = ((Integer)p.get(FilterParams.WindowBits_K)).intValue();

			if (p.containsKey(FilterParams.MemoryLevel_K))
				memLevel = ((Integer)p.get(FilterParams.MemoryLevel_K)).intValue();
		}

		fl.DeflateInit2(compression, windowBits, memLevel, strategy);
	}

	public FlateOutputStream(OutputStream out)
	{
		this(out, null);
	}

	// Methods
	/**
	 * Writes a byte. Will block until the byte is actually
	 * written.
	 * @param b the byte
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(int b)
		throws IOException
	{
		fl.write(b & 0xff);
	}

	/**
	 * Writes a subarray of bytes.
	 * @param b	the data to be written
	 * @param off	the start offset in the data
	 * @param len	the number of bytes that are written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte b[], int off, int len)
		throws IOException
	{
		int	maxWrite = b.length-off;

		if (maxWrite > len)
			maxWrite = len;

		while (maxWrite-- > 0)
			write(b[off++]);
	}

	/**
	 * Writes an array of bytes. Will block until the bytes
	 * are actually written.
	 * @param b	the data to be written
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void write(byte b[])
		throws IOException
	{
		write(b, 0, b.length);
	}

	/**
	 * Closes the stream and writes the stream trailer.
	 * This method must be called to release any resources
	 * associated with the stream.
	 * @exception IOException If an I/O error has occurred.
	 */
	@Override
	public void close()
		throws IOException
	{
		fl.write(FlateEncode.Z_PARTIAL_FLUSH);
		fl.write(FlateEncode.Z_EOF);
		super.close();
	}

	/**
	 * Counts the number of bytes written by this filter.
	 * @return	actual number of bytes written
	 */
	public long getTotalOut()
	{
		return fl.getTotalOut();
	}
}
