/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.formcalc;


import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.ResId;


/**
 *
 * Class <b>FrameTable</b> defines the runtime frame table (stack)
 * that's used by the <a href=CalcParser.html">FormCalc scripting engine</a>,
 * when invoking user-defined functions.
 * 
 * @author Mike P. Tardif.
 *
 * @exclude from published api.
 */
public final class FrameTable {

	static final int SIZE = 128;    // initial size of FrameTable

    
	/**
	 * Instantiate an FrameTable object.
	 */
	FrameTable() {
		// empty
	}


	/**
	 * Create storage for this FrameTable object.
	 * @param nTableSize an initial size for the frame table.
	 * @return integer 1 upon success, and 0 otherwise.
	 */
	int create(int nTableSize /* = SIZE */) {
		mnFrameTableSize = nTableSize;
		mpFrameTableBase = new Frame[mnFrameTableSize];
		mpFrameTablePtr = 0;
		return 1;
	}

	/**
	 * Reinitialize this Frame object.
	 */
	void init() {
		for (int i = 0; i < mnFrameTableSize; i++)
			mpFrameTableBase[i] = null;
		mpFrameTablePtr = 0;
	}


	/**
	 * Get this object's frame table size.
	 * @return the current frame table size.
	 */
	int getSize() {
		return mnFrameTableSize;
	}


	/**
	 * Get this object's frame stack depth.
	 * @return the current frame table stack depth.
	 */
	public int getDepth() {
		return mpFrameTablePtr;
	}



	/**
	 * Pop the frame on top of this FrameTable object.
	 * @return the frame on top of frame table.
	 * The frame is popped off the table. 
	 */
	Frame pop() {
		assert(mpFrameTablePtr > 0);
		return mpFrameTableBase[--mpFrameTablePtr];
	}


	/**
	 * Push the given frame onto this FrameTable object.
	 * @param oFrame the frame to oush onto the FrameTable.
	 */
	void push(Frame oFrame) {
		//
		// grow Frame as needed.
		//
		if (mpFrameTablePtr >= mnFrameTableSize) {
			mnFrameTableSize <<= 1;
			Frame[] oNewBase = new Frame[mnFrameTableSize];
			System.arraycopy(mpFrameTableBase, 0, oNewBase, 0, 
														mnFrameTableSize >> 1);
			mpFrameTableBase = oNewBase;
			mpFrameTablePtr = mnFrameTableSize >> 1;
		}
		mpFrameTableBase[mpFrameTablePtr++] = oFrame;
	}

	/**
	 * Peek at the frame on top of this FrameTable object.
	 * @return the frame on top of FrameTable.
	 */
	public Frame peek() {
		assert(mpFrameTablePtr > 0);
		return mpFrameTableBase[mpFrameTablePtr - 1];
	}


	/**
	 * Return a string representing the current function call stack.
	 */
	public String getStackTrace(Stack oStack) {
		throw new ExFull(ResId.UNSUPPORTED_OPERATION, "FrameTable#getStackTrace");
	// Javaport: do uncomment if ever needed.
	//	StringBuilder sStackTrace = new StringBuilder("[toplevel]");
	//	for (int f = 0; f < mnFrameTableSize; f++) {
	//		sStackTrace.append('\n');
	//		CalcSymbol oFuncSym = mpFrameTableBase[f].getFuncSym();
	//		int nStackAddr = mpFrameTableBase[f].getStackAddr();
	//		int nArgCount = mpFrameTableBase[f].getArgCount();
	//		sStackTrace.append(oFuncSym.getName());
	//		sStackTrace.append('(');
	//		for (int i = 0; i < nArgCount; i++) {
	//			if (i != 0)
	//				sStackTrace.append(',');
	//			CalcSymbol oSym = oStack.peek(nStackAddr - nArgCount + i + 1);
	//			switch (oSym.getType()) {
	//			case CalcSymbol.TypeNull:
	//				break;
	//			case CalcSymbol.TypeVariable:
	//			case CalcSymbol.TypeString:
	//				sStackTrace.append('"');
	//				sStackTrace.append(oSym.getStringValue());
	//				sStackTrace.append('"');
	//				break;
	//			case CalcSymbol.TypeDouble:
	//				String sNum
	//					= FormCalcUtil.dblToStr(oSym.getNumericValue(), 8);
	//				sStackTrace.append(sNum);
	//				break;
	//			case CalcSymbol.TypeAccessor:
	//				sStackTrace.append(oSym.getName());
	//				break;
	//			default:
	//				sStackTrace.append('?');
	//				break;
	//			}
	//		}
	//		sStackTrace.append(')');
	//	}
	//	return sStackTrace.toString();
	}


	private int     mnFrameTableSize;        // size of frame table.
	private Frame[]	mpFrameTableBase;        // frame table.
	private int		mpFrameTablePtr;         // frame table oointer.

}
