// =================================================================================================
// ADOBE SYSTEMS INCORPORATED
// Copyright 2011 Adobe Systems Incorporated
// All Rights Reserved
//
// NOTICE:  Adobe permits you to use, modify, and distribute this file in accordance with the terms
// of the Adobe license agreement accompanying it.
// =================================================================================================

package com.adobe.xmp.schema.model;

import java.util.List;



/**
 * Describes an XMP array property.
 * There are four types of arrays:
 * <ul>
 * 		<li>Bag</li> (unorderen array)
 * 		<li>Seq</li> (ordered array)
 * 		<li>Alt</li> (alternative array)
 * 		<li>Alt-Text</li> (language alternative, syntactically equal to Alt,
 * 			but contains only text properties with a language qualifier)
 * </ul>
 * Additionally, a rule can be added to contrain an array to have
 * a minimum and/or maximum number of Items (LengthRule).
 * 
 * @author Stefan Makswit
 */
public interface ArrayType extends PropertyType
{
	/**
	 * Enumeration of the four XMP array forms.
	 * TOTRACK: Consider removing ALTERNATIVE_TEXT
	 * 
	 * @author Stefan Makswit
	 */
	public enum ArrayForm
	{
		/** an unordered "Bag"-Array */
		UNORDERED, 
		/** an ordered "Seq"-Array */
		ORDERED, 
		/** an alternative "Alt"-Array */
		ALTERNATIVE,
		/** an alternative text array, where each item is the same string in a different translation */
		ALTERNATIVE_TEXT;
	}


	/**
	 * @return Returns the array form.
	 */
	ArrayForm getForm();
	
	
	/**
	 * @return Returns the type of the array items. 
	 *	 	This can be any XMP type including another array.
	 *		<em>Note:</em> All array items must have the same type, 
	 *		even if RDF does not enforce this.
	 */
	PropertyType getItemType();
	
	
	/**
	 * @param itemType the type of the array items. 
	 *	 	This can be any XMP type including another array.
	 *		<em>Note:</em> All array items must have the same type, 
	 *		even if RDF does not enforce this.
	 */
	void setItemType(PropertyType itemType);
	
	
	// ========================
	// Array Item Qualifier API
	// ========================

	/**
	 * Add an array item qualifier to each array item.
	 * As array items have the same type in XMP,
	 * the qualfiers are valid for all array items. 
	 * 
	 * @param qualifier the qualifier to add.
	 */
	void addItemQualifier(PropertyDescription qualifier);
	
	
	/**
	 * Gets an item qualifier by namespace and local name if existing.
	 * 
	 * @param namespaceURI the qualifier's namespace
	 * @param localName the qualifier's local name
	 * @return Returns the qualifier or <code>null</code> if not existing.
	 */
	PropertyDescription getItemQualifier(String namespaceURI, String localName);
	
	
	/**
	 * @return Returns the collection of all item qualifiers
	 */
	List<PropertyDescription> getItemQualifiers();
	
	
	/**
	 * Removes an item qualifier by namespace and local name.
	 * 
	 * @param namespaceURI the qualifier's namespace
	 * @param localName the qualifier's local name
	 */
	void removeItemQualifier(String namespaceURI, String localName);
	
	
	/**
	 * @return Returns true if the property has at least on qualifier.
	 */
	boolean hasItemQualifiers();
}
