package com.adobe.xmp.schema.service;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;
import java.util.Set;

import javax.xml.namespace.QName;

import com.adobe.xmp.path.XMPPath;


/**
 * Provides all available RelaxNG schema resources and referenced includes.
 * The returned streams shall contain the Relax NG source files.
 * 
 * @author Stefan Makswit
 */
public interface RelaxNGProvider
{
	/**
	 * Provides a RelaxNG resource by its namespace URI. 
	 * <code>null</code> is returned if there is no resource available for this namespace.
	 * If only the existence shall be checked, the call {@link #isAvailable(String)} should be used.
	 * <em>Note:</em> The provider contains only "top-level" XMP schemas (such as Dublin Core) and
	 * no XMP type schemas (such as Dimensions).
	 * 
	 * @param namespaceURI the namespace URI of the schema according to the XMP specification
	 * @return Returns the RelaxNG resource as stream
	 * @throws IOException Forwards exceptions that occur during schema file loading.
	 */
	InputStream getSchema(String namespaceURI) throws IOException;

	
	/**
	 * Provides a RelaxNG include, referenced by its name
	 * 
	 * @param resourceName the name of the resource, usually a file name,
	 * 					   e.g. "include.rng".
	 * @return Returns the RelaxNG resource as stream or <code>null</code> if it has not been found.
	 * @throws IOException Forwards exceptions that occur during schema file loading.
	 */
	InputStream getInclude(String includeName) throws IOException;
	
	
	/**
	 * Checks if a schema (identified by its URI) can be provided.
	 * The result of this call <em>has to be</em> symmetric to {@link #getSchema(String)}.
	 * <em>Note:</em> For all namespaces where isAvailable() returns <code>true</code>,
	 * the method getSchema() has to return a result as well.
	 * <em>Note:</em> a call to this method should only check whether the RelaxNG resource is available,
	 * and should not already load the resource.
	 * 
	 * @param namespaceURI the namespace URI of the schema according to the XMP specification
	 * 
	 * @return Returns true if the schema is registered. 
	 * 		   This does not imply that it can be loaded successfully.
	 */
	boolean isAvailable(String namespaceURI);
	
	
	/**
	 * Runtime decorators are applied to the schema once it is parsed and 
	 * are stored in the schema model:
	 * <ul>
	 * 		<li>The Map key is an {@link XMPPath} which models a path such as "xmpDM:duration.xmpDM:scale".</li>
	 * 		<li>The value is another Map, where the keys are QNames of decorators.</li>
	 * 		<li>Each decorator value is a property/value list.</li>
	 * </ul>
	 * <em>Note:</em>
	 * 
	 * @return
	 */
	Map<XMPPath, Map<QName, Map<String, String>>> getRuntimeDecorators();
	
	
	/**
	 * Returns a list of all schema namespaces that this provider can deliver.
	 * @deprecated TODO Check if this is really needed. 
	 * 
	 * @return Returns a list of all available schema namespaces of this provider.
	 */
	Set<String> getNamespaces();
}