/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

package com.adobe.cq.dam.download.api;

import java.net.URI;
import java.util.Optional;

import org.osgi.annotation.versioning.ProviderType;

import com.adobe.cq.dam.download.spi.DownloadTargetProcessor;

/**
 * <p>
 * Represents a repository file whose binary contents can be included in a download
 * artifact. This is the result of converting a {@link DownloadTarget} into one or
 * more files by a {@link DownloadTargetProcessor}.
 * </p>
 * <p>
 * In addition to some common values, the interface provides the ability to specify
 * additional parameters in the form of name value pairs.
 * </p>
 * @see DownloadService
 * @see DownloadTarget
 * @see DownloadTargetProcessor
 */
@ProviderType
public interface DownloadFile {

    /**
     * Retrieves the size of the file, in bytes.
     * @return Byte size.
     */
    Optional<Long> getSize();

    /**
     * Retrieves a URI to the file's binary. Should be accessible to the service
     * being used to generate the archive.
     * @return URI to a binary.
     */
    URI getBinaryURI();

    /**
     * Retrieves one of the file's parameters by name. The value will be converted to
     * the specified type.
     * @param <T> Target type for the parameter value.
     * @param parameterName The name of the parameter whose value to retrieve.
     * @param clazz The type to which that the parameter's value will be converted.
     * @return The given parameter's value.
     */
    <T> T getParameter(String parameterName, Class<T> clazz);

    /**
     * Retrieves one of the file's parameters by name. If the parameter doesn't have
     * a value then the method will return the given default value instead. The parameter
     * value will be converted to the specified type.
     * @param <T> Target type for the parameter value.
     * @param parameterName The name of the parameter whose value to retrieve.
     * @param defaultValue Value to return if the parameter has no value.
     * @return The given parameter's value, or the given defalt value.
     */
    <T> T getParameter(String parameterName, T defaultValue);

    /**
     * Retrieves the names of all parameters included in the file.
     * @return Parameter names.
     */
    String[] getParameterNames();
}
