package com.adobe.xfa.service.image;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;

import com.adobe.xfa.Attribute;
import com.adobe.xfa.Node;
import com.adobe.xfa.ProtoableNode;
import com.adobe.xfa.XFA;
import com.adobe.xfa.content.ImageValue;
import com.adobe.xfa.protocol.ProtocolUtils;
import com.adobe.xfa.template.TemplateModel;
import com.adobe.xfa.template.containers.Subform;
import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.StringHolder;

public class TemplateUnfolder {

	public enum TemplateUnfoldOptions {
		UNFOLD_RELATIVE_ASSETS, UNFOLD_ABSOLUTE_ASSETS
	}

	// Encode all the following characters only. URI class requires all these to
	// be encoded.
	// Good thing about this is that even if the string comes encoded, we can
	// still transparently encode it. It won't be encoded twice
	private final static char[] charsToEncodeBeforePassingToURIClass = { ' ',
			'"', '<', '>', '[', '\\', ']', '^', '`', '{', '|', '}' };

	public static TemplateModel unfoldTemplate(TemplateModel oTemplate,
			String rootPath,
			EnumSet<TemplateUnfoldOptions> oTemplateUnfoldOptions)
			throws ExFull {
		try {
			assert(null != oTemplate);
			assert(null != rootPath);
			assert(null != oTemplateUnfoldOptions);
			//If assertions are disabled throw a suitable exception if any parameter is null
			if (null == oTemplate){
				throw new ExFull(new NullPointerException("A null template is being passed. It should not be null."));
			}
			if (null == rootPath){
				throw new ExFull(new NullPointerException("Root path is set to null. It should not be null."));
			}
			if (null == oTemplateUnfoldOptions){
				throw new ExFull(new NullPointerException("Template Unfolding options are set to null. They should not be null."));
			}			
			processNode(oTemplate, rootPath, oTemplateUnfoldOptions);
			return oTemplate;
		} catch (Exception e) {
			e.printStackTrace();
			ExFull ne = new ExFull(e);
			throw ne;
		}
	}

	// Method to encode URIs for consumption by java.net.URI class
	private static String encodeForURIConsumption(String s) {
		s = s.replaceAll("\\\\", "/");
		char[] sortedArray = new char[charsToEncodeBeforePassingToURIClass.length];
		System.arraycopy(charsToEncodeBeforePassingToURIClass, 0, sortedArray,
				0, sortedArray.length);
		Arrays.sort(sortedArray);
		StringBuffer b = new StringBuffer();
		for (int i = 0; i < s.length(); ++i) {
			char c = s.charAt(i);
			if (Arrays.binarySearch(sortedArray, c) >= 0) {
				// Found special char in the String. Encode it.
				if (c == '%') {
					// %00 to %7f has special meaning, so ignore it
					boolean dontProcessPercentage = false;
					if (i + 2 < s.length())
						if (s.charAt(1 + i) >= '0' && s.charAt(1 + i) <= '7')
							if (s.charAt(1 + i) >= '0'
									&& s.charAt(1 + i) <= '9'
									|| s.charAt(1 + i) >= 'a'
									&& s.charAt(1 + i) <= 'f'
									|| s.charAt(1 + i) >= 'A'
									&& s.charAt(1 + i) <= 'F')
								dontProcessPercentage = true;
					if (dontProcessPercentage)
						b.append(c);
					else
						b.append(toHexString(c));
				} else {
					b.append(toHexString(c));
				}
			} else {
				b.append(c);
			}
		}
		return b.toString();
	}

	private static void processNode(Node node, String parent,
			EnumSet<TemplateUnfoldOptions> oTemplateUnfoldOptions)
			throws Exception {

		if (node instanceof ProtoableNode) {
			Attribute att = ((ProtoableNode) node).getAttributeByName(
					"usehref", true);
			if (att != null
					&& node instanceof com.adobe.xfa.template.containers.Subform) {
				String sFragmentPath = att.getAttrValue();
				sFragmentPath = encodeForURIConsumption(sFragmentPath);
				boolean doIt = oTemplateUnfoldOptions
						.contains(TemplateUnfoldOptions.UNFOLD_ABSOLUTE_ASSETS)
						&& ProtocolUtils.isAbsolute(sFragmentPath)
						|| oTemplateUnfoldOptions
								.contains(TemplateUnfoldOptions.UNFOLD_RELATIVE_ASSETS)
						&& !ProtocolUtils.isAbsolute(sFragmentPath);
				if (doIt) {
					int lastFwdSlash = sFragmentPath.lastIndexOf('/');
					if (lastFwdSlash > -1) {
						String fragmentUrl = sFragmentPath.substring(0,
								lastFwdSlash);
						if (ProtocolUtils.isAbsolute(fragmentUrl))
							parent = fragmentUrl;
						else
							parent += "/" + fragmentUrl;
					}
					((Subform) node).removeAttr(null, "usehref");
				}
			}
		}
		if (node instanceof ImageValue) {
			ImageValue imageValue = (ImageValue) node;
			if (imageValue.isPropertySpecified(XFA.HREFTAG, true, 0)
					&& !imageValue.isPropertySpecified(XFA.TRANSFERENCODINGTAG,
							true, 0)) {
				// not inline; get hRef value
				String sHref = imageValue.getAttribute(XFA.HREFTAG)
						.getAttrValue();
				sHref = encodeForURIConsumption(sHref);
				boolean doIt = oTemplateUnfoldOptions
						.contains(TemplateUnfoldOptions.UNFOLD_ABSOLUTE_ASSETS)
						&& ProtocolUtils.isAbsolute(sHref)
						|| oTemplateUnfoldOptions
								.contains(TemplateUnfoldOptions.UNFOLD_RELATIVE_ASSETS)
						&& !ProtocolUtils.isAbsolute(sHref);
				if (doIt) {
					String sContentType = imageValue.getAttribute(
							XFA.CONTENTTYPETAG).getAttrValue();
					if (!ProtocolUtils.isAbsolute(sHref)) {
						while (parent.endsWith("/"))
							parent = parent.substring(0, parent.length() - 1);
						sHref = parent + "/" + sHref;
					}
					String baseUrl = null, relativeUrl = null;
					int idx = sHref.lastIndexOf("/");
					baseUrl = sHref.substring(0, idx);
					relativeUrl = sHref.substring(1 + idx);
					InputStream in = ProtocolUtils.checkUrl(baseUrl,
							relativeUrl, true, new StringHolder());
					if (null == in)
						throw new IOException("Could not load image: " + sHref);
					int sz = 1 << 10;
					ArrayList<byte[]> list = new ArrayList<byte[]>();
					byte[] array = new byte[sz];
					int totalBytes = 0;
					int readSz = -1;
					while ((readSz = in.read(array)) > -1) {
						byte[] clonedArray = new byte[readSz];
						System.arraycopy(array, 0, clonedArray, 0, readSz);
						list.add(clonedArray);
						totalBytes += readSz;
					}
					byte[] finalArray = new byte[totalBytes];
					int sofar = 0;
					for (byte[] _t : list) {
						System.arraycopy(_t, 0, finalArray, sofar, _t.length);
						sofar += _t.length;
					}
					imageValue.setValue(finalArray, sContentType);
				}
			}
		}
		for (Node child = node.getFirstXFAChild(); child != null; child = child
				.getNextXFASibling()) {
			processNode(child, parent, oTemplateUnfoldOptions);
		}
	}

	private static final String toHexString(char c) {
		String r = "%" + Integer.toHexString((int) c);
		return r;
	}
}
