package com.adobe.xfa.text;


/**
 * Class TextPosition represents a position in a particular text
 * stream.	We refer to it as a <em>tracked</em> position, meaning that
 * the stream tracks it and will update it as insertions and deletions
 * occur at locations in the stream before this position.  For
 * information on a position object that is not tracked, please see the
 * base class, TextPosnBase.
 * </p>
 * <p>
 * For more information, please see the extenral documentation and the
 * base class.
 * </p>
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public class TextPosn extends TextPosnBase {
	TextPosn mpoNext; //  Text stream's association chain

/**
 * Default constructor.
 * <p>
 * The position is not initially associated with any stream.
 */
	public TextPosn () {
	}

/**
 * Copy constructor.
 * <p>
 * Copy all contents of the source position, including stream
 * association, index and before/after state.
 * @param oSource Source position object to copy.
 */
	public TextPosn (TextPosn oSource) {
		super (oSource);
		streamAttach();
	}

/**
 * Copy (base class) constructor.
 * <p>
 * This method creates the position objecgt by copying the stream
 * association, index and before/after type from an object in the base
 * class, TextPosnBase.  Note: even though this copies the base class,
 * the new position object will be tracked by the stream.
 * @param oSource Base class position object to copy.
 */
	public TextPosn (TextPosnBase oSource) {
		super (oSource);
		streamAttach();
	}

/**
 * Constructor with stream and optional position type.
 * <p>
 * Construct a position object associated with the given stream and
 * optional position type.	The position is placed before the first
 * non-attribute item in the stream.
 * @param poNewStream Stream to associate with.  NULL creates a
 * position object with no initial association.
 */
//	public TextPosn (TextStream poNewStream, PosnCode eNewPosn) {
//		TextPosnBase = poNewStream, eNewPosn;
//		StreamAttach();
//	}
	public TextPosn (TextStream poNewStream) {
		super (poNewStream);
		streamAttach();
	}

/**
 * Constructor with stream, index and optional position type.
 * <p>
 * Construct a position object associated with the given stream and
 * optional position type.	The position is placed at the specified
 * index.
 * @param poNewStream Stream to associate with.  NULL creates a
 * position object with no initial association.
 * @param nNewIndex Index number for the position.	Will be truncated if too
 * large.
 * @param eNewPosn (optional) Position type to use for the object.
 * Default is POSN_AFTER.
 */
	public TextPosn (TextStream poNewStream, int nNewIndex, int eNewPosn) {
		super (poNewStream, nNewIndex, eNewPosn);
		streamAttach();
	}
	public TextPosn (TextStream poNewStream, int nNewIndex) {
		super (poNewStream, nNewIndex);
		streamAttach();
	}
	
	public boolean equals(Object object) { // NOPMD - UselessOverridingMethod - here for documentation/FindBugs suppression
		// The fields added in this derived class do not participate in equality
		return super.equals(object);
	}
	
	public int hashCode() { // NOPMD - UselessOverridingMethod - here for documentation/FindBugs suppression
		// The fields added in this derived class do not participate in equality
		return super.hashCode();
	}

/**
 * Assignment operator.
 * <p>
 * Copy the entire content of the source position object including
 * stream association, index and before/after type.  The position object
 * will be tracked by the new stream (if not NULL) and will no longer be
 * tracked by its old stream.
 * @param oSource Source position object to copy.
 */
	public void copyFrom (TextPosn oSource) {
		copyFrom ((TextPosnBase) oSource);
	}

/**
 * Assignment operator from base class.
 * <p>
 * Copy the entire content of the source position object including
 * stream association, index and before/after type.  The position object
 * will be tracked by the new stream (if not NULL) and will no longer be
 * tracked by its old stream.
 * @param oSource Source (base class) position object to copy.
 */
	public void copyFrom (TextPosnBase oSource) {
		boolean bReattach = conditionalDetach (oSource.stream());
		super.copyFrom (oSource);
		conditionalAttach (bReattach);
	}

/**
 * Overridden: Associate position with a new stream.
 * <p>
 * This method overrides the base class implementation.  It associates
 * the position object with the specified stream, immediately before the
 * first non-attribute item.  The position object will be tracked by the
 * new stream (if not NULL) and will no longer be tracked by its old
 * stream.
 * @param poNewStream Stream to associate with.  NULL leaves the
 * position object unassociated (and untracked).
 * @param eNewPosn (optional) Position type to use for the object.
 * Default is POSN_AFTER.
 */
	public void associate (TextStream poNewStream, int eNewPosn) {
		boolean bReattach = conditionalDetach (poNewStream);
		super.associate (poNewStream, eNewPosn);
		conditionalAttach (bReattach);
	}

/**
 * Overridden: Associate position with a new stream and index.
 * <p>
 * This method overrides the base class implementation.  It associates
 * the position object with the specified stream, at the specified index
 * position.  The position object will be tracked by the new stream (if
 * not NULL) and will no longer be tracked by its old stream.
 * @param poNewStream Stream to associate with.  NULL leaves the
 * position object unassociated (and untracked).
 * @param nNewIndex Index of this position in the new stream.  If the
 * value is too large, it is truncated.
 * @param eNewPosn (optional) Position type to use for the object.
 * Default is POSN_AFTER.
 */
	public void associate (TextStream poNewStream, int nNewIndex, int eNewPosn) {
		boolean bReattach = conditionalDetach (poNewStream);
		super.associate (poNewStream, nNewIndex, eNewPosn);
		conditionalAttach (bReattach);
	}

	public void associate (TextStream poNewStream) {
		boolean bReattach = conditionalDetach (poNewStream);
		super.associate (poNewStream);
		conditionalAttach (bReattach);
	}

	public void para () {
		insertPara();
	}

	public void text (String sText) {
		insert (sText);
	}

	public void attr (TextAttr oAttr) {
		attribute (oAttr);
	}

//----------------------------------------------------------------------
//
//		ConditionalDetach(), ConditionalAttach(): Used when our
//		stream is about to be reassigned.
//
//		Stream detaching and attaching can be time-consuming.
//		These methods are to be used in pairs, calling
//		ConditionalDetach() first.	It determines whether we
//		really need to detach and reattach and returns a flag for
//		ConditionalAttach().
//
//----------------------------------------------------------------------
	private boolean conditionalDetach (TextStream poNewStream) {
		if (stream() == poNewStream) {
			return false;
		}

		if (stream() != null) {
			streamDetach (stream());
		}

		return poNewStream != null;
	}

	private void conditionalAttach (boolean bReattach) {
		if (bReattach) {
			streamAttach (stream());
		}
	}

//----------------------------------------------------------------------
//
//		StreamAttach, StreamDetach: the raw mechanics of
//		connecting to our stream.
//
//----------------------------------------------------------------------
	private void streamAttach () {
		if (stream() != null) {
			streamAttach (stream());
		}
	}

//	private void streamDetach () {
//		if (stream() != null) {
//			streamDetach (stream());
//		}
//	}

	private void streamAttach (TextStream poNewStream) {
		poNewStream.posnAttach (this);
	}

	private void streamDetach (TextStream poOldStream) {
		poOldStream.posnDetach (this);
	}
}
