/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.util;

import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.designimporter.api.ImporterConstants;
import com.day.cq.wcm.designimporter.impl.common.PathSchemeHelper;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.api.resource.ValueMap;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

/**
 * Utility class for Importer component related stuff
 */
public class ImporterUtil {

    /**
     * Looks up importer in the passed page resource
     *
     * @param resource The page resource which needs to be searched for importer resource
     *
     * @return The found importer resource. In case of multiple, the first found importer resource is returned
     */
    static public Resource findImporter(Resource resource) {
        List<Resource> found = findImporters(resource);
        if(found.size() > 0)
            return found.get(0);
        else
            return null;
    }

    /**
     * Finds importer resources within the page resource passed. Doesn't look into sub pages.
     *
     * @param resource The page resource which needs to be searched for importer resource
     *
     * @return The list of importer resources found
     */
    static public List<Resource> findImporters(Resource resource) {
        return findImporters(resource, false);
    }

    /**
     * Finds importer resources within the passed page resource. Looks into sub pages if the recursive param is passed true
     *
     * @param resource The page resource which needs to be searched for importer resource
     *
     * @param recursive if true, all sub pages are looked up as well
     *
     * @return The list of importer resources found
     */
    static public List<Resource> findImporters(Resource resource, boolean recursive) {
        List<Resource> found = new LinkedList<Resource>();
        if(resource != null) {
            Page page = resource.adaptTo(Page.class);
            if(recursive)
                findImporterComponent(resource, found);
            else
                findImporterComponent(page.getContentResource(), found);
        }
        return found;
    }

    /**
     * Checks if the passed canvas is primary.
     *
     * @param resource The resource corresponding to a canvas
     * @return true if the passed canvas resource is primary in nature
     *
     * @throws IllegalArgumentException When the passed resource is not a valid canvas resource
     */
    static public boolean checkCanvasPrimary(Resource resource) throws IllegalArgumentException{
        return !checkCanvasSecondary(resource);
    }

    /**
     * Checks if the passed canvas is secondary.
     *
     * <p>
     * Note: A design import typically results in one primary canvas
     * accompanied by zero or more secondary canvases.
     * </p>
     *
     * @param resource The resource corresponding to a canvas
     * @return true if the passed canvas resource is secondary in nature
     *
     * @throws IllegalArgumentException When the passed resource is not a valid canvas resource
     */
    static public boolean checkCanvasSecondary(Resource resource) throws IllegalArgumentException{
        if(!isCanvas(resource))
            throw new IllegalArgumentException("The passed resource is not a valid canvas resource");

        Resource canvas = getCanvas(resource);
        Resource importer = canvas.getParent();
        if(canvas != null) {
            ValueMap valueMap = importer.adaptTo(ValueMap.class);
            if(valueMap.containsKey(ImporterConstants.PN_SECONDARY_CANVAS))
                return valueMap.get(ImporterConstants.PN_SECONDARY_CANVAS, Boolean.class).booleanValue();
        }
        return false;
    }

    /**
     * Returns the resource corresponding to the canvas design
     *
     * @param resource The resource corresponding to a canvas or its container importer component node
     *
     * @return Resource corresponding to the canvas design
     */
    static public Resource getCanvasDesign(Resource resource) {
        // If the passed canvas or importer is secondary, get its primary counterpart first.
        Resource canvas = getCanvas(resource);
        PageManager pm = resource.getResourceResolver().adaptTo(PageManager.class);
        if(checkCanvasSecondary(canvas)) {
            Page primaryPage = pm.getContainingPage(canvas).getParent();
            Resource importer = findImporter(primaryPage.adaptTo(Resource.class));
            canvas = getCanvas(importer);
        }

        String canvasDesignPath = PathSchemeHelper.getCanvasDesignPath(canvas);
        return resource.getResourceResolver().resolve(canvasDesignPath);

        /*Resource importer = canvas.getParent();
        Page page = pm.getContainingPage(importer);
        Designer designer = resource.getResourceResolver().adaptTo(Designer.class);
        String designPath = designer.getDesignPath(page);

        // {designPath}+"canvas"+{resourcePath}
        String canvasDesignPath = designPath + "/" + "canvas" + importer.getPath();
        return resource.getResourceResolver().resolve(canvasDesignPath);*/
    }

    /**
     * Convenience method to get canvas from the importer. If a canvas resource is passed,
     * it's returned back
     *
     * @param resource
     * @return
     */
    static public Resource getCanvas(Resource resource) {
        if(ResourceUtil.isA(resource, ImporterConstants.RESOURCE_TYPE_IMPORTER))
            return resource.getChild(ImporterConstants.NN_CANVAS);
        else if(ResourceUtil.isA(resource.getParent(), ImporterConstants.RESOURCE_TYPE_IMPORTER))
            return resource;

        return null;
    }

    /* Looks up importer component nodes within the tree held by the passed root.
     * Upon finding, it pushes the importer component to the passed components list*/
    static private void findImporterComponent(Resource root, List<Resource> components) {
        if(isImporter(root)) {
            components.add(root);
        } else {
            Iterator<Resource> childIterator = root.listChildren();
            while(childIterator.hasNext()) {
                Resource child = childIterator.next();
                findImporterComponent(child, components);
            }
        }
    }

    /**
     * Checks if the passed resource corresponds to a generated
     * canvas node based on the following heuristics:
     *
     * <ul>
     *     <li>Its parent has the resource type {@link com.day.cq.wcm.designimporter.api.ImporterConstants#RESOURCE_TYPE_IMPORTER}</li>
     * </ul>
     * @param resource
     * @return
     */
    static public boolean isCanvas(Resource resource) {
        if(resource == null)
            return false;

        if(isImporter(resource.getParent()))
            return true;

        return false;
    }

    /**
     * Checks if the passed resource corresponds to importer
     * @param resource
     * @return
     */
    static public boolean isImporter(Resource resource) {
        return ResourceUtil.isA(resource, ImporterConstants.RESOURCE_TYPE_IMPORTER);
    }

    /**
     * Checks if the passed page is created using the {@link ImporterConstants#PAGE_TEMPLATE_IMPORTER_PAGE} template
     *
     * @param page
     * @return
     */
    static public boolean isImporterPage(Page page) {
        if(page == null)
            return false;
        return ResourceUtil.isA(page.getContentResource(), ImporterConstants.RESOURCE_TYPE_IMPORTER_PAGE);
    }

    /**
     * Checks if the passed resource is a, or belongs to a page created using the {@link ImporterConstants#PAGE_TEMPLATE_IMPORTER_PAGE} template
     *
     * @param resource The page resource or a resource contained within the page
     * @return
     */
    static public boolean isImporterPage(Resource resource) {
        PageManager pageManager = resource.getResourceResolver().adaptTo(PageManager.class);
        Page containingPage = pageManager.getContainingPage(resource);
        return isImporterPage(containingPage);
    }

    /**
     * Deletes the canvas artifact along with any dangling ascendants.
     *
     * @param resource Resource corresponding to a canvas artifact like canvas design or canvas component
     * @throws RepositoryException
     */
    static public void deleteCanvasArtifact(Resource resource) throws RepositoryException {
        deleteCanvasArtifact(resource, true);
    }

    /**
     * Deletes the canvas artifact
     *
     * @param resource Resource corresponding to a canvas artifact like canvas design or canvas component
     * @param cleanupDanglingAscendants flag to indicate if ascendants left dangling,
     *                                  because of the deleted canvas, also need to be cleaned up.
     * @throws RepositoryException
     */
    static public void deleteCanvasArtifact(Resource resource, boolean cleanupDanglingAscendants) throws RepositoryException {
        if(resource != null && !ResourceUtil.isNonExistingResource(resource)) {
            Resource parent = resource.getParent();
            Node node = resource.adaptTo(Node.class);
            node.remove();
            node.getSession().save();

            if(cleanupDanglingAscendants)
                cleanupDanglingAscendants(parent);
        }
    }
    /**
     * This method traverses up the hierarchy (uptil canvas dir) of the passed resource and deletes
     * all nodes without children.
     * Since the canvas artifacts are created in convention based structured folders, deletion of artifacts
     * requires cleanup of the folders that were created to contain the artifact.
     */
    static public void cleanupDanglingAscendants(Resource resource) throws RepositoryException {
        if (resource.getPath().contains("/canvas/")) {

            Resource last = null;
            while (resource != null) {
                Iterator<Resource> iter = resource.listChildren();
                if (iter.hasNext()) iter.next(); // We're checking for numChildren < 2
                if ("canvas".equals(resource.getName()) || iter.hasNext()) {
                    break;
                }
                last = resource;
                resource = resource.getParent();
            }
            if (last != null) {
                last.adaptTo(Node.class).remove();
            }
        }
    }

    /**
     * For internal use
     */
    static public Resource getDanglingAscendantRoot(Resource resource) throws RepositoryException {
        if(resource.getPath().contains("/canvas/")) {
            // We should proceed only if the resource was the lone child of its parent
            if(!resource.getParent().listChildren().hasNext()) {
                Resource last, curr = resource;
                boolean isParentCanvas, parentHasMoreChildren;
                do {
                    last = curr;
                    curr = curr.getParent();
                    isParentCanvas = "canvas".equals(curr.getName());
                    Iterator<Resource> iter = curr.listChildren();
                    if(iter.hasNext()) iter.next(); // We're checking for numChildren < 2
                    parentHasMoreChildren = iter.hasNext();
                } while(!isParentCanvas && !parentHasMoreChildren);

                return last;
            }
        }
        return resource;
    }

}
